package com.digiwin.athena.semc.service.mobile.impl;

import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.digiwin.athena.appcore.auth.AppAuthContextHolder;
import com.digiwin.athena.semc.common.Constants;
import com.digiwin.athena.semc.entity.mobile.MobileApplicatonColumn;
import com.digiwin.athena.semc.entity.mobile.MobileDatasourceClassification;
import com.digiwin.athena.semc.entity.mobile.MobileDatasourceInfo;
import com.digiwin.athena.semc.env.EnvProperties;
import com.digiwin.athena.semc.mapper.mobile.MobileApplicatonColumnMapper;
import com.digiwin.athena.semc.mapper.mobile.MobileDatasourceClassificationMapper;
import com.digiwin.athena.semc.mapper.mobile.MobileDatasourceInfoMapper;
import com.digiwin.athena.semc.service.mobile.MobileDatasourceClassificationService;
import com.digiwin.athena.semc.service.mobile.MobileUserAuthService;
import com.digiwin.athena.semc.util.DateUtils;
import com.digiwin.athena.semc.util.Utils;
import com.digiwin.athena.semc.vo.mobile.MobileClassificatonResp;
import com.digiwin.athena.semc.vo.mobile.MobileDatasourceInfoVO;
import io.github.linpeilie.Converter;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.ObjectUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import javax.annotation.Resource;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

/**
 * 自定义应用分类服务实现类
 *
 * @author qutong
 * @since 2024-09-03
 */
@Service
public class MobileDatasourceClassificationServiceImpl extends ServiceImpl<MobileDatasourceClassificationMapper, MobileDatasourceClassification> implements MobileDatasourceClassificationService {

    @Resource
    private MobileDatasourceClassificationMapper mobileDatasourceClassificationMapper;

    @Resource
    private MobileDatasourceInfoMapper mobileDatasourceInfoMapper;

    @Resource
    private MobileApplicatonColumnMapper columnMapper;

    @Resource
    MobileUserAuthService mobileUserAuthService;

    @Resource
    private EnvProperties envProperties;

    @Resource
    private Converter converter;

    /**
     * 查询应用分类
     *
     * @return 返回
     */
    @Override
    public List<MobileDatasourceClassification> queryApplicationClassificationList() {
        return mobileDatasourceClassificationMapper.queryApplicationClassificationList(AppAuthContextHolder.getContext().getAuthoredUser().getTenantId());
    }

    /**
     * 判断名称是否重复
     *
     * @param id   主键id
     * @param name 公告类型名称
     * @return 返回
     */
    @Override
    public List<MobileDatasourceClassification> existNameValid(Long id, String name) {
        return mobileDatasourceClassificationMapper.existNameValid(name, id, AppAuthContextHolder.getContext().getAuthoredUser().getTenantId());
    }

    /**
     * 删除自定义类型
     *
     * @param id 主键id
     */
    @Override
    @Transactional
    public void delApplicationClassification(Long id) {
        // 先删除公告类型
        mobileDatasourceClassificationMapper.deleteById(id);
        // 引用该公告的类型变为其他类型
        mobileDatasourceClassificationMapper.updateClassificationById(id, DateUtils.getNowTime(DateUtils.DATE_TIME_NORMAL_FORMATTER), AppAuthContextHolder.getContext().getAuthoredUser().getUserId(), AppAuthContextHolder.getContext().getAuthoredUser().getTenantId());
    }

    @Override
    public List<MobileClassificatonResp> getClassificationList(Boolean isAll) {
        List<MobileClassificatonResp> result = new ArrayList<>();
        // 先查自定义分类
        List<MobileDatasourceClassification> mobileApplicationClassifications = this.queryApplicationClassificationList();
        if (CollectionUtils.isEmpty(mobileApplicationClassifications)) {
            return result;
        }
        //isAll 是否需要查询全部的数据，true：查询所有分类，false：需要根据用户已选的分类过滤
        List<Long> appIds = new ArrayList<>();
        //查询已经配置到栏目中的应用
        LambdaQueryWrapper<MobileApplicatonColumn> columnLambdaQueryWrapper = new LambdaQueryWrapper<>();
        columnLambdaQueryWrapper.eq(MobileApplicatonColumn::getCreateUserId, Utils.getUserId()).isNotNull(MobileApplicatonColumn::getCustomApplicationId);
        List<MobileApplicatonColumn> mobileApplicatonColumns = columnMapper.selectList(columnLambdaQueryWrapper);
        if (CollectionUtils.isNotEmpty(mobileApplicatonColumns)) {
            appIds = mobileApplicatonColumns.stream().map(MobileApplicatonColumn::getCustomApplicationId).collect(Collectors.toList());
        }

        // 根据分类id查询所有自定义应用
        List<Long> classIdList = mobileApplicationClassifications.stream().map(MobileDatasourceClassification::getId).collect(Collectors.toList());
        LambdaQueryWrapper<MobileDatasourceInfo> queryWrapper = new LambdaQueryWrapper<>();
        queryWrapper.eq(MobileDatasourceInfo::getStatus, Constants.VALID_STATUS_ENABLE).in(MobileDatasourceInfo::getClassificationId, classIdList);
        List<MobileDatasourceInfo> mobileDatasourceInfos = mobileDatasourceInfoMapper.selectList(queryWrapper);
        Map<Long, List<MobileDatasourceInfo>> classIdMap = mobileDatasourceInfos.stream().collect(Collectors.groupingBy(MobileDatasourceInfo::getClassificationId));

        // 过滤出用户有权限的自定义应用列表，不是鼎捷非雅典娜应用，不参与校验
        List<Long> mobileAppIdList = mobileDatasourceInfos.stream()
                .filter(x -> !Constants.MobileDataSourceTye.DIGIWIN_NOT_ATHENA_APPLICATION.getValue().equals(x.getType())).map(MobileDatasourceInfo::getId).collect(Collectors.toList());
        // 鼎捷非雅典娜应用，校验权限，返回有权限的自定义应用id
        List<MobileDatasourceInfo> notAthenaAppList = mobileDatasourceInfos.stream().filter(x -> Constants.MobileDataSourceTye.DIGIWIN_NOT_ATHENA_APPLICATION.getValue().equals(x.getType())).collect(Collectors.toList());
        if (CollectionUtils.isNotEmpty(notAthenaAppList)) {
            List<Long> withPermissionList = mobileUserAuthService.validPermission(notAthenaAppList);
            mobileAppIdList.addAll(withPermissionList);
        }

        // 自定义分类的应用
        for (MobileDatasourceClassification mobileApplicationClassification : mobileApplicationClassifications) {
            MobileClassificatonResp mobileClassificatonResp = converter.convert(mobileApplicationClassification, MobileClassificatonResp.class);;
            List<MobileDatasourceInfoVO> mobileDatasourceInfoVOList = new ArrayList<>();

            // 每个分类下的自定义应用列表
            List<MobileDatasourceInfo> datasourceInfos = classIdMap.get(mobileApplicationClassification.getId());
            if (CollectionUtils.isNotEmpty(datasourceInfos)) {
                for (MobileDatasourceInfo mobileDatasourceInfo : datasourceInfos) {

                    if (!mobileAppIdList.contains(mobileDatasourceInfo.getId())) {
                        continue;
                    }
                    MobileDatasourceInfoVO mobileDatasourceInfoVO = converter.convert(mobileDatasourceInfo, MobileDatasourceInfoVO.class);
                    mobileDatasourceInfoVO.setClassificationName(mobileApplicationClassification.getClassificationName());
                    //已添加到栏目中的应用,给应用加标识，1:已添加
                    if (appIds.contains(mobileDatasourceInfo.getId())) {
                        mobileDatasourceInfoVO.setIsAdd(1);
                        //isAll 是否需要查询全部的数据，true：查询所有分类，false：需要根据用户已选的分类过滤
                        if (!Boolean.TRUE.equals(isAll)) {
                            continue;
                        }
                    }
                    String icon = mobileDatasourceInfoVO.getIcon();
                    if (StringUtils.isEmpty(mobileDatasourceInfoVO.getIcon())) {
                        icon = envProperties.getSemcWebUrl() + Constants.CONTENT_ICON_DEF;
                    }
                    //处理图标-图标库
                    if (null != mobileDatasourceInfo.getIconType()
                            && mobileDatasourceInfo.getIconType().equals(Constants.MobileIconTypeEnum.ICON_LIB.getFlag())) {
                        icon = envProperties.getSemcWebUrl() + icon;
                    }
                    //处理图标-自定义
                    if (null != mobileDatasourceInfo.getIconType()
                            && mobileDatasourceInfo.getIconType().equals(Constants.MobileIconTypeEnum.ICON_CUSTOM.getFlag())) {
                        icon = envProperties.getDmcUri() + Constants.DMC_FILE_PREVIEW_PATH + icon;
                    }
                    mobileDatasourceInfoVO.setIcon(icon);
                    mobileDatasourceInfoVOList.add(mobileDatasourceInfoVO);
                }
            }
            mobileClassificatonResp.setMobileCustomAppList(mobileDatasourceInfoVOList);
            result.add(mobileClassificatonResp);
        }
        return result;
    }

    @Override
    public List<MobileClassificatonResp> getClassificationListNew(Boolean isAll) {
        List<MobileClassificatonResp> result = new ArrayList<>();
        // 先查自定义分类
        List<MobileDatasourceClassification> mobileApplicationClassifications = this.queryApplicationClassificationList();
        if (CollectionUtils.isEmpty(mobileApplicationClassifications)) {
            return result;
        }
        //isAll 是否需要查询全部的数据，true：查询所有分类，false：需要根据用户已选的分类过滤
        List<Long> appIds = new ArrayList<>();
        //查询已经配置到栏目中的应用
        LambdaQueryWrapper<MobileApplicatonColumn> columnLambdaQueryWrapper = new LambdaQueryWrapper<>();
        columnLambdaQueryWrapper.eq(MobileApplicatonColumn::getCreateUserId, Utils.getUserId()).isNotNull(MobileApplicatonColumn::getCustomApplicationId);
        List<MobileApplicatonColumn> mobileApplicatonColumns = columnMapper.selectList(columnLambdaQueryWrapper);
        if (CollectionUtils.isNotEmpty(mobileApplicatonColumns)) {
            appIds = mobileApplicatonColumns.stream().map(MobileApplicatonColumn::getCustomApplicationId).collect(Collectors.toList());
        }

        // 根据分类id查询所有自定义应用
        List<Long> classIdList = mobileApplicationClassifications.stream().map(MobileDatasourceClassification::getId).collect(Collectors.toList());
        LambdaQueryWrapper<MobileDatasourceInfo> queryWrapper = new LambdaQueryWrapper<>();
        queryWrapper.eq(MobileDatasourceInfo::getStatus, Constants.VALID_STATUS_ENABLE).in(MobileDatasourceInfo::getClassificationId, classIdList);
        List<MobileDatasourceInfo> mobileDatasourceInfos = mobileDatasourceInfoMapper.selectList(queryWrapper);
        Map<Long, List<MobileDatasourceInfo>> classIdMap = mobileDatasourceInfos.stream().collect(Collectors.groupingBy(MobileDatasourceInfo::getClassificationId));

        // 过滤出用户有权限的自定义应用列表，不是鼎捷非雅典娜应用，不参与校验
        List<Long> mobileAppIdList = mobileDatasourceInfos.stream()
                .filter(x -> !Constants.MobileDataSourceTye.DIGIWIN_NOT_ATHENA_APPLICATION.getValue().equals(x.getType())).map(MobileDatasourceInfo::getId).collect(Collectors.toList());
        // 鼎捷非雅典娜应用，校验权限，返回有权限的自定义应用id
        List<MobileDatasourceInfo> notAthenaAppList = mobileDatasourceInfos.stream().filter(x -> Constants.MobileDataSourceTye.DIGIWIN_NOT_ATHENA_APPLICATION.getValue().equals(x.getType())).collect(Collectors.toList());
        if (CollectionUtils.isNotEmpty(notAthenaAppList)) {
            List<Long> withPermissionList = mobileUserAuthService.validPermission(notAthenaAppList);
            mobileAppIdList.addAll(withPermissionList);
        }

        // 自定义分类的应用
        for (MobileDatasourceClassification mobileApplicationClassification : mobileApplicationClassifications) {
            MobileClassificatonResp mobileClassificatonResp = converter.convert(mobileApplicationClassification, MobileClassificatonResp.class);
            List<MobileDatasourceInfoVO> mobileDatasourceInfoVOList = new ArrayList<>();

            // 每个分类下的自定义应用列表
            List<MobileDatasourceInfo> datasourceInfos = classIdMap.get(mobileApplicationClassification.getId());
            if (CollectionUtils.isNotEmpty(datasourceInfos)) {
                for (MobileDatasourceInfo mobileDatasourceInfo : datasourceInfos) {

                    if (!mobileAppIdList.contains(mobileDatasourceInfo.getId())) {
                        continue;
                    }
                    MobileDatasourceInfoVO mobileDatasourceInfoVO = converter.convert(mobileDatasourceInfo, MobileDatasourceInfoVO.class);
                    mobileDatasourceInfoVO.setClassificationName(mobileApplicationClassification.getClassificationName());
                    //已添加到栏目中的应用,给应用加标识，1:已添加
                    if (appIds.contains(mobileDatasourceInfo.getId())) {
                        mobileDatasourceInfoVO.setIsAdd(1);
                        //isAll 是否需要查询全部的数据，true：查询所有分类，false：需要根据用户已选的分类过滤
                        if (!Boolean.TRUE.equals(isAll)) {
                            continue;
                        }
                    }
                    String icon = mobileDatasourceInfoVO.getIcon();
                    if (StringUtils.isEmpty(mobileDatasourceInfoVO.getIcon())) {
                        icon = envProperties.getSemcWebUrl() + Constants.CONTENT_ICON_DEF;
                    }
                    //处理图标-图标库
                    if (null != mobileDatasourceInfo.getIconType()
                            && mobileDatasourceInfo.getIconType().equals(Constants.MobileIconTypeEnum.ICON_LIB.getFlag())) {
                        icon = envProperties.getSemcWebUrl() + icon;
                    }
                    //处理图标-自定义
                    if (null != mobileDatasourceInfo.getIconType()
                            && mobileDatasourceInfo.getIconType().equals(Constants.MobileIconTypeEnum.ICON_CUSTOM.getFlag())) {
                        icon = envProperties.getDmcUri() + Constants.DMC_FILE_PREVIEW_PATH + icon;
                    }
                    mobileDatasourceInfoVO.setIcon(icon);
                    mobileDatasourceInfoVOList.add(mobileDatasourceInfoVO);
                }
            }
            mobileClassificatonResp.setMobileCustomAppList(mobileDatasourceInfoVOList);
            result.add(mobileClassificatonResp);
        }
        return result;
    }

    @Override
    public void saveClassification(MobileDatasourceClassification mobileDatasourceClassification) {
        if (ObjectUtils.isEmpty(mobileDatasourceClassification.getId())) {
            mobileDatasourceClassificationMapper.insert(mobileDatasourceClassification);
        } else {
            MobileDatasourceClassification classification = mobileDatasourceClassificationMapper.selectClassificationById(mobileDatasourceClassification.getId());
            classification.setClassificationName(mobileDatasourceClassification.getClassificationName());
            classification.setClassificationDescription(mobileDatasourceClassification.getClassificationDescription());
            classification.setPriority(mobileDatasourceClassification.getPriority());
            mobileDatasourceClassificationMapper.updateClassificationInfo(classification);
        }
    }

}
