package com.digiwin.athena.semc.service.mobile.impl;

import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.baomidou.mybatisplus.core.conditions.update.UpdateWrapper;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.digiwin.athena.appcore.auth.AppAuthContextHolder;
import com.digiwin.athena.semc.common.Constants;
import com.digiwin.athena.semc.common.PageInfoResp;
import com.digiwin.athena.semc.dto.mobile.CustomApplicationsListPageReq;
import com.digiwin.athena.semc.dto.mobile.SaveAppcationReq;
import com.digiwin.athena.semc.dto.mobile.UpdateAppcationConfigReq;
import com.digiwin.athena.semc.dto.mobile.UpdateAppcationStatusReq;
import com.digiwin.athena.semc.entity.applink.AppLinkDTO;
import com.digiwin.athena.semc.entity.bench.SyncJobInfo;
import com.digiwin.athena.semc.entity.mobile.MobileDatasourceClassification;
import com.digiwin.athena.semc.entity.mobile.MobileDatasourceInfo;
import com.digiwin.athena.semc.env.EnvProperties;
import com.digiwin.athena.semc.mapper.mobile.MobileApplicatonColumnMapper;
import com.digiwin.athena.semc.mapper.mobile.MobileDatasourceClassificationMapper;
import com.digiwin.athena.semc.mapper.mobile.MobileDatasourceInfoMapper;
import com.digiwin.athena.semc.proxy.iam.service.IamService;
import com.digiwin.athena.semc.service.applink.AppLinkService;
import com.digiwin.athena.semc.service.mobile.MobileDatasourceInfoService;
import com.digiwin.athena.semc.util.DateUtils;
import com.digiwin.athena.semc.util.Utils;
import com.digiwin.athena.semc.vo.common.PermissionUserFunctionalDTO;
import com.digiwin.athena.semc.vo.common.UserApplicationDTO;
import com.digiwin.athena.semc.vo.common.UserApplicationModulesDTO;
import com.digiwin.athena.semc.vo.common.UserApplicationSimpleDTO;
import com.digiwin.athena.semc.vo.mobile.MobileDatasourceInfoVO;

import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.collections.MapUtils;
import org.apache.commons.lang3.ObjectUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.time.LocalDateTime;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

import javax.annotation.Resource;

import io.github.linpeilie.Converter;


/**
 * 自定义应用服务实现类
 *
 * @author qutong
 * @since 2024-09-03
 */
@Service
public class MobileDatasourceInfoServiceImpl extends ServiceImpl<MobileDatasourceInfoMapper, MobileDatasourceInfo> implements MobileDatasourceInfoService {

    @Resource
    private MobileDatasourceInfoMapper mobileDatasourceInfoMapper;

    @Resource
    private MobileDatasourceClassificationMapper mobileApplicationClassificationMapper;

    @Autowired
    private AppLinkService appLinkService;

    @Autowired
    private IamService iamService;

    @Autowired
    EnvProperties envProperties;

    @Resource
    private MobileApplicatonColumnMapper applicatonColumnMapper;

    @Resource
    private Converter converter;

    /**
     * 分页查询移动应用/作业列表
     *
     * @param customApplicationsListPageReq 查询参数
     * @return 返回
     */
    @Override
    public PageInfoResp<MobileDatasourceInfoVO> queryCustomApplicationsByPage(CustomApplicationsListPageReq customApplicationsListPageReq) {

        QueryWrapper<MobileDatasourceInfo> condition = new QueryWrapper<>();

        // 名称
        if (StringUtils.isNotBlank(customApplicationsListPageReq.getName())) {
            condition.like("name", customApplicationsListPageReq.getName());
        }
        // 状态
        if (ObjectUtils.isNotEmpty(customApplicationsListPageReq.getStatus())) {
            condition.eq("status", customApplicationsListPageReq.getStatus());
        }
        // 类型
        if (ObjectUtils.isNotEmpty(customApplicationsListPageReq.getType())) {
            condition.eq("type", customApplicationsListPageReq.getType());
        }
        // 数据源应用
        if (ObjectUtils.isNotEmpty(customApplicationsListPageReq.getDataSource()) && ObjectUtils.isNotEmpty(customApplicationsListPageReq.getPrimaryId())) {
            condition.eq("application_primary_id", customApplicationsListPageReq.getPrimaryId());
            condition.eq("application_source", customApplicationsListPageReq.getDataSource());
        }
        // 分类
        if (ObjectUtils.isNotEmpty(customApplicationsListPageReq.getClassificationIdList())) {
            condition.in("classification_id", customApplicationsListPageReq.getClassificationIdList());
        }
        // 作业
        if (ObjectUtils.isNotEmpty(customApplicationsListPageReq.getWorkCodeIdList())) {
            condition.in("work_code", customApplicationsListPageReq.getWorkCodeIdList());
        }


        condition.orderByDesc("modify_time");
        condition.eq("tenant_id", AppAuthContextHolder.getContext().getAuthoredUser().getTenantId());

        // 分页查询列表
        PageInfoResp<MobileDatasourceInfoVO> pageInfoResp = new PageInfoResp<>();
        Page<MobileDatasourceInfo> pageResult = mobileDatasourceInfoMapper.selectPage(new Page<>(customApplicationsListPageReq.getPageNum(), customApplicationsListPageReq.getPageSize()), condition);
        if (pageResult == null || CollectionUtils.isEmpty(pageResult.getRecords())) {
            return pageInfoResp;
        }

        // 获取类型的名称
        List<MobileDatasourceInfo> applicationsList = pageResult.getRecords();
        List<MobileDatasourceClassification> classificationList = mobileApplicationClassificationMapper.queryApplicationClassificationList(AppAuthContextHolder.getContext().getAuthoredUser().getTenantId());
        Map<Long, String> idNameMap = classificationList.stream().collect(Collectors.toMap(MobileDatasourceClassification::getId, MobileDatasourceClassification::getClassificationName, (a, b) -> a));

        //获取数据源名称
        List<AppLinkDTO> appLinkListDTO = appLinkService.queryManageListSync(AppAuthContextHolder.getContext().getAuthoredUser());
        Map<String, String> appLinkMap = appLinkListDTO.stream().filter(x -> Constants.APP_TYPE_BS.equals(x.getSystemType())).collect(Collectors.toList()).stream().collect(Collectors.toMap(app -> app.getPrimaryId() + "-" + app.getDataSource(), AppLinkDTO::getName, (a, b) -> a));

        Map<String, String> appCodeMap = appLinkListDTO.stream().filter(x -> Constants.APP_TYPE_BS.equals(x.getSystemType())).collect(Collectors.toList()).stream().filter(y -> !StringUtils.isEmpty(y.getApplicationAppId())).collect(Collectors.toMap(app -> app.getPrimaryId() + "-" + app.getDataSource(), AppLinkDTO::getApplicationAppId, (a, b) -> a));

        List<MobileDatasourceInfoVO> mobileCustomApplicationVOList = applicationsList.stream().map(x -> {
            MobileDatasourceInfoVO mobileCustomApplicationVO = converter.convert(x, MobileDatasourceInfoVO.class);
            // 分类名称
            if (MapUtils.isNotEmpty(idNameMap)) {
                mobileCustomApplicationVO.setClassificationName(idNameMap.get(x.getClassificationId()));
            }

            //三方应用取对应名称，非雅典娜新应用直接取值
            if(Constants.MobileDataSourceTye.MOBILE_THIRD_APPLICATION.getValue().equals(x.getType())){
                // 数据源名称
                if (MapUtils.isNotEmpty(appLinkMap)) {
                    mobileCustomApplicationVO.setApplicationSourceName(appLinkMap.get(x.getApplicationPrimaryId() + "-" + x.getApplicationSource()));
                    if (Constants.DEFAULT_CONFIGURED_FLAG_NO.equals(x.getIsJob())){
                        //根据appid和workcode取作业名称
                        List<SyncJobInfo> jobInfos = mobileDatasourceInfoMapper.queryWorkName(appCodeMap.get(x.getApplicationPrimaryId() + "-" + x.getApplicationSource()), x.getWorkCode());
                        if (!CollectionUtils.isEmpty(jobInfos)){
                            mobileCustomApplicationVO.setWorkName(jobInfos.get(0).getJobName());
                        }
                    }
                }
            }else {
                mobileCustomApplicationVO.setApplicationSourceName(x.getApplicationConfig());
            }

            //图标url拼接
            if (!StringUtils.isEmpty(mobileCustomApplicationVO.getIcon())) {
                mobileCustomApplicationVO.setIconUrl(mobileCustomApplicationVO.getIcon());
            }
            if (null != x.getIconType() && Constants.MobileIconTypeEnum.ICON_CUSTOM.getFlag().equals(x.getIconType()) && !StringUtils.isEmpty(x.getIcon())) {
                mobileCustomApplicationVO.setIconUrl(envProperties.getDmcUri() + Constants.DMC_FILE_PREVIEW_PATH + x.getIcon());
            }

            return mobileCustomApplicationVO;
        }).collect(Collectors.toList());
        pageInfoResp.setTotalPages(Integer.parseInt(String.valueOf(pageResult.getPages())));
        pageInfoResp.setPageNo(customApplicationsListPageReq.getPageNum());
        pageInfoResp.setPageSize(customApplicationsListPageReq.getPageSize());
        pageInfoResp.setTotalRecords(Integer.parseInt(String.valueOf(pageResult.getTotal())));
        pageInfoResp.setList(mobileCustomApplicationVOList);
        return pageInfoResp;
    }

    /**
     * 更新应用启用状态
     *
     * @param updateAppcationStatusReq 入参
     */
    @Override
    public void updateStatus(UpdateAppcationStatusReq updateAppcationStatusReq) {

        UpdateWrapper<MobileDatasourceInfo> updateWrapper = new UpdateWrapper<>();
        updateWrapper.set("status", updateAppcationStatusReq.getStatus())
                .set("modify_user_id", AppAuthContextHolder.getContext().getAuthoredUser().getUserId())
                .set("modify_user_name", AppAuthContextHolder.getContext().getAuthoredUser().getUserName())
                .set("modify_time", LocalDateTime.now());
        updateWrapper.in("id", updateAppcationStatusReq.getIdList());
        this.update(updateWrapper);
    }

    /**
     * 删除移动应用/作业信息
     *
     * @param idList 自定义应用id列表
     */
    @Override
    @Transactional
    public void delApplicationInfo(List<Long> idList) {
        // 删除应用信息
        mobileDatasourceInfoMapper.deleteBatchIds(idList);
        //删除栏目中的应用信息
        applicatonColumnMapper.delColumnByApp(idList);
    }

    /**
     * 通过ID查询单条数据
     *
     * @param id 主键
     * @return 实例对象
     */
    @Override
    public MobileDatasourceInfoVO queryCustomApplicationDetail(Long id) {
        MobileDatasourceInfoVO mobileCustomApplicationVO = new MobileDatasourceInfoVO();
        MobileDatasourceInfo mobileCustomApplication = mobileDatasourceInfoMapper.selectById(id);
        if (ObjectUtils.isEmpty(mobileCustomApplication)) {
            return mobileCustomApplicationVO;
        }
        converter.convert(mobileCustomApplication, mobileCustomApplicationVO);
        List<MobileDatasourceClassification> classificationList = mobileApplicationClassificationMapper.queryApplicationClassificationList(AppAuthContextHolder.getContext().getAuthoredUser().getTenantId());

        //获取数据源名称
        List<AppLinkDTO> appLinkListDTO = appLinkService.queryManageListSync(AppAuthContextHolder.getContext().getAuthoredUser());
        List<AppLinkDTO> collect = appLinkListDTO.stream().filter(x -> x.getDataSource().equals(mobileCustomApplication.getApplicationSource()) && x.getPrimaryId().equals(mobileCustomApplication.getApplicationPrimaryId())).collect(Collectors.toList());
        if (!CollectionUtils.isEmpty(collect)) {
            mobileCustomApplicationVO.setApplicationSourceName(collect.get(0).getName());
        }

        List<MobileDatasourceClassification> collect1 = classificationList.stream().filter(x -> x.getId().equals(mobileCustomApplication.getClassificationId())).collect(Collectors.toList());
        if (!CollectionUtils.isEmpty(collect1)) {
            mobileCustomApplicationVO.setClassificationName(collect1.get(0).getClassificationName());
        }

        return mobileCustomApplicationVO;
    }

    /**
     * 保存移动应用/作业管理
     *
     * @param saveAppcationReq 入参
     * @return 返回
     */
    @Override
    @Transactional
    public void saveCustomApplication(SaveAppcationReq saveAppcationReq) {
        if (ObjectUtils.isEmpty(saveAppcationReq.getId())) {
            MobileDatasourceInfo mobileCustomApplication = new MobileDatasourceInfo();
            buildCustomApplicationInfo(saveAppcationReq, mobileCustomApplication);
            mobileDatasourceInfoMapper.insert(mobileCustomApplication);
        } else {
            MobileDatasourceInfo mobileCustomApplication = mobileDatasourceInfoMapper.selectById(saveAppcationReq.getId());

            buildCustomApplicationInfo(saveAppcationReq, mobileCustomApplication);

            mobileDatasourceInfoMapper.updateByIdNeedNull(mobileCustomApplication);

            if (Constants.MobileDataSourceTye.MOBILE_THIRD_APPLICATION.getValue().equals(mobileCustomApplication.getType())){
                mobileDatasourceInfoMapper.updateIamDate(mobileCustomApplication);
            }
        }

    }

    /**
     * 同名数量查询
     *
     * @param saveAppcationReq 入参
     * @return
     */
    @Override
    public Integer countByParam(SaveAppcationReq saveAppcationReq) {
        return mobileDatasourceInfoMapper.countByParam(saveAppcationReq.getName(), saveAppcationReq.getId(), AppAuthContextHolder.getContext().getAuthoredUser().getTenantId());
    }

    @Override
    public List<MobileDatasourceInfo> queryDatasourceInfo(CustomApplicationsListPageReq customApplicationReq) {

        QueryWrapper<MobileDatasourceInfo> queryWrapper = new QueryWrapper<>();
        // 状态
        if (ObjectUtils.isNotEmpty(customApplicationReq.getStatus())) {
            queryWrapper.eq("status", customApplicationReq.getStatus());
        }
        // 应用id
        if (StringUtils.isNotBlank(customApplicationReq.getPrimaryId())) {
            queryWrapper.eq("application_primary_id", customApplicationReq.getPrimaryId());
        }
        queryWrapper.eq("type", Constants.MobileDataSourceTye.DIGIWIN_NOT_ATHENA_APPLICATION.getValue());
        queryWrapper.orderByDesc("modify_time");
        return mobileDatasourceInfoMapper.selectList(queryWrapper);

    }

    @Override
    public List<UserApplicationSimpleDTO> queryApplicationByUser() {
        //筛选过滤有效期的商品+模组
        return iamService.queryUserApplicationSimple(true,false,false);
    }

    @Override
    @Deprecated //该方法后台不调用了
    public List<UserApplicationModulesDTO> queryModuleByApplication(PermissionUserFunctionalDTO permissionUserFunctionalDTO) {
        List<UserApplicationDTO> applicationDTOS = iamService.queryUserApplication();
        List<UserApplicationDTO> collect = applicationDTOS.stream().filter(x -> x.getId().equals(permissionUserFunctionalDTO.getId())).collect(Collectors.toList());

        List<UserApplicationModulesDTO> moduleList = new ArrayList<>();
        if (!CollectionUtils.isEmpty(collect) && !CollectionUtils.isEmpty(collect.get(0).getEnabledModules())){
            //筛选过滤有效期的模组
            moduleList = collect.get(0).getEnabledModules().stream().filter(x -> DateUtils.strToDate(x.getExpiredTime()).getTime() > new Date().getTime()).collect(Collectors.toList());

        }
        return moduleList;
    }

    @Override
    public List<PermissionUserFunctionalDTO> queryWorkByApplication(PermissionUserFunctionalDTO permissionUserFunctionalDTO) {

        List<PermissionUserFunctionalDTO> responseList = new ArrayList<>();

        if (!StringUtils.isEmpty(permissionUserFunctionalDTO.getSid())){
            //拼接target请求iam接口
            String target = Constants.IAM_FUNCTIONAL_STR + permissionUserFunctionalDTO.getSid();
            List<PermissionUserFunctionalDTO> userFunctionalDTOS = iamService.queryUserFunctional(target, AppAuthContextHolder.getContext().getAuthoredUser());
            //过滤对应模组所属信息
            if (!CollectionUtils.isEmpty(userFunctionalDTOS)) {
                responseList = userFunctionalDTOS.stream().filter(x -> !StringUtils.isEmpty(x.getModuleId()) && x.getModuleId().equals(permissionUserFunctionalDTO.getModuleId())).collect(Collectors.toList());

                //过滤已有绑定关系的作业id集合
                List<String> workIdList = mobileDatasourceInfoMapper.selectAllWorkId(permissionUserFunctionalDTO);
                if (!CollectionUtils.isEmpty(workIdList)){
                    responseList = responseList.stream().filter(x -> !workIdList.contains(x.getId())).collect(Collectors.toList());
                }
            }
        }
        return responseList;
    }

    @Override
    public void updateConfig(UpdateAppcationConfigReq updateAppcationConfigReq) {

        UpdateWrapper<MobileDatasourceInfo> updateWrapper = new UpdateWrapper<>();
        updateWrapper.set("iam_application_id", updateAppcationConfigReq.getIamApplicationId())
                .set("iam_module_id", updateAppcationConfigReq.getIamModuleId())
                .set("iam_work_id", updateAppcationConfigReq.getIamWorkId())
                .set("modify_user_id", AppAuthContextHolder.getContext().getAuthoredUser().getUserId())
                .set("modify_user_name", AppAuthContextHolder.getContext().getAuthoredUser().getUserName())
                .set("modify_time", LocalDateTime.now());
        updateWrapper.eq("id", updateAppcationConfigReq.getId());
        this.update(updateWrapper);
    }

    /**
     * 构建对象
     *
     * @param saveAppcationReq        请求入参
     * @param mobileCustomApplication 对象
     */
    private void buildCustomApplicationInfo(SaveAppcationReq saveAppcationReq, MobileDatasourceInfo mobileCustomApplication) {
        mobileCustomApplication.setName(saveAppcationReq.getName());
        mobileCustomApplication.setIcon(saveAppcationReq.getIcon());
        mobileCustomApplication.setIconType(saveAppcationReq.getIconType());
        mobileCustomApplication.setApplicationPrimaryId(saveAppcationReq.getApplicationPrimaryId());
        mobileCustomApplication.setApplicationSource(saveAppcationReq.getApplicationSource());
        mobileCustomApplication.setType(saveAppcationReq.getType());
        mobileCustomApplication.setClassificationId(saveAppcationReq.getClassificationId());
        mobileCustomApplication.setStatus(saveAppcationReq.getStatus());
        mobileCustomApplication.setRemarks(saveAppcationReq.getRemarks());
        mobileCustomApplication.setWorkCode(saveAppcationReq.getWorkCode());
        mobileCustomApplication.setIsJob(saveAppcationReq.getIsJob());
        mobileCustomApplication.setCallBackUrl(saveAppcationReq.getCallBackUrl());
        mobileCustomApplication.setSpecialSign(saveAppcationReq.getSpecialSign());
        mobileCustomApplication.setApplicationConfig(saveAppcationReq.getApplicationConfig());
        mobileCustomApplication.setModifyUserName(AppAuthContextHolder.getContext().getAuthoredUser().getUserName());
        mobileCustomApplication.setCreateUserName(AppAuthContextHolder.getContext().getAuthoredUser().getUserName());
        mobileCustomApplication.setCreateUserId(Utils.getUserId());
        mobileCustomApplication.setModifyUserId(Utils.getUserId());
        mobileCustomApplication.setCreateTime(DateUtils.getNowTime(""));
        mobileCustomApplication.setModifyTime(DateUtils.getNowTime(""));
    }

}
