package com.digiwin.athena.semc.service.mobile.impl;

import com.google.common.collect.Maps;

import com.digiwin.athena.appcore.auth.AppAuthContextHolder;
import com.digiwin.athena.appcore.auth.domain.AuthoredUser;
import com.digiwin.athena.semc.common.Constants;
import com.digiwin.athena.semc.common.enums.ApplicationTypeEnum;
import com.digiwin.athena.semc.common.enums.VirtualApplicationEnum;
import com.digiwin.athena.semc.entity.mobile.MobileDatasourceInfo;
import com.digiwin.athena.semc.entity.portal.LabelSystemData;
import com.digiwin.athena.semc.mapper.mobile.MobileDatasourceInfoMapper;
import com.digiwin.athena.semc.proxy.iam.service.IamService;
import com.digiwin.athena.semc.service.mobile.MobileUserAuthService;
import com.digiwin.athena.semc.vo.common.PermissionUserFunctionalDTO;
import com.digiwin.athena.semc.vo.common.UserApplicationModulesSimpleDTO;
import com.digiwin.athena.semc.vo.common.UserApplicationSimpleDTO;
import com.digiwin.athena.semc.vo.common.UserAuthSystemLabelVo;

import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.compress.utils.Lists;
import org.apache.commons.lang3.ObjectUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.stereotype.Service;

import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.Executor;
import java.util.function.Function;
import java.util.stream.Collectors;

import javax.annotation.Resource;

import lombok.extern.slf4j.Slf4j;

@Service
@Slf4j
public class MobileUserAuthServiceImpl implements MobileUserAuthService {

    @Resource
    private MobileDatasourceInfoMapper mobileCustomApplicationMapper;

    @Resource
    private IamService iamService;

    @Resource(name = "semcAsyncExecutor")
    private Executor asyncTaskExecutor;

    @Override
    public List<Long> validUserAppAuth(List<Long> customApplicationIdList) {
        if (CollectionUtils.isEmpty(customApplicationIdList)) {
            return Lists.newArrayList();
        }
        // 查询移动数据源配置
        List<MobileDatasourceInfo> mobileDatasourceInfoList = mobileCustomApplicationMapper.selectBatchIds(customApplicationIdList);
        if (CollectionUtils.isEmpty(mobileDatasourceInfoList)) {
            return Lists.newArrayList();
        }

        // 不是鼎捷非雅典娜应用，不参与校验
        List<Long> mobileAppIdList = mobileDatasourceInfoList.stream()
                .filter(x -> !Constants.MobileDataSourceTye.DIGIWIN_NOT_ATHENA_APPLICATION.getValue().equals(x.getType())).map(MobileDatasourceInfo::getId).collect(Collectors.toList());

        // 过滤出鼎捷非雅典娜应用，校验权限
        List<MobileDatasourceInfo> notAthenaAppList = mobileDatasourceInfoList.stream().filter(x -> Constants.MobileDataSourceTye.DIGIWIN_NOT_ATHENA_APPLICATION.getValue().equals(x.getType())).collect(Collectors.toList());
        if (CollectionUtils.isNotEmpty(notAthenaAppList)) {
            // 权限校验，返回有权限的自定义应用id
            List<Long> withPermissionList = validPermission(notAthenaAppList);
            mobileAppIdList.addAll(withPermissionList);
        }
        return mobileAppIdList;
    }

    @Override
    public UserAuthSystemLabelVo buildSystemIdApp(List<LabelSystemData> systemDataList) {
        UserAuthSystemLabelVo authSystemLabelVo=new UserAuthSystemLabelVo();
        //没有权限的自定义组件ID
        List<Long> noPermSystemIdList = Lists.newArrayList();
        //有权限的自定义组件ID
        List<Long> permSystemIdList = Lists.newArrayList();
        authSystemLabelVo.setNoPermSystemIdList(noPermSystemIdList);
        authSystemLabelVo.setPermSystemIdList(permSystemIdList);
        if (CollectionUtils.isEmpty(systemDataList)) {
            return authSystemLabelVo;
        }
        //过滤得到源应用为鼎捷雅典娜APP 且 作业集成中关联的移动应用作业
        List<LabelSystemData> systemDataAppList = systemDataList.stream().filter(x -> ApplicationTypeEnum.VIRTUAL_APPLICATION.getType().equals(x.getDataType())
                && VirtualApplicationEnum.ATHENA_APP.getPrimaryId().equals(x.getAppCode())
                && Constants.DataCategoryEnum.CATEGORY_WORK.getVal().equals(x.getDataCategory())).collect(Collectors.toList());
        //没有数据源配置 源应用为鼎捷雅典娜APP 且 作业集成
        if (CollectionUtils.isEmpty(systemDataAppList)) {
            return authSystemLabelVo;
        }
        //得到源应用为鼎捷雅典娜APP 且 作业集成中关联的移动应用作业ID集合
        List<String> dataSourceIdList = systemDataAppList.stream().map(LabelSystemData::getWorkCode).collect(Collectors.toList());
        //没有数据源配置 源应用为鼎捷雅典娜APP 且 作业集成
        if (CollectionUtils.isEmpty(dataSourceIdList)) {
            return authSystemLabelVo;
        }
        //校验用户是否有该移动应用作业权限
        List<Long> customApplicationIdList = dataSourceIdList.stream().map(Long::parseLong).collect(Collectors.toList());
        //获取有权限的移动应用作业ID
        List<Long> applicationIdList = validUserAppAuth(customApplicationIdList);
        //为空则说明都没有权限
        if (CollectionUtils.isEmpty(applicationIdList)) {
            noPermSystemIdList = systemDataAppList.stream().map(LabelSystemData::getSystemId).collect(Collectors.toList());
            authSystemLabelVo.setNoPermSystemIdList(noPermSystemIdList);
            return authSystemLabelVo;
        }
        //部分有权限，则过滤，得到没有权限的组件ID
        noPermSystemIdList = systemDataAppList.stream().filter(x -> !applicationIdList.contains(Long.parseLong(x.getWorkCode()))).map(LabelSystemData::getSystemId).collect(Collectors.toList());
        authSystemLabelVo.setNoPermSystemIdList(noPermSystemIdList);
        permSystemIdList = systemDataAppList.stream().filter(x -> applicationIdList.contains(Long.parseLong(x.getWorkCode()))).map(LabelSystemData::getSystemId).collect(Collectors.toList());
        authSystemLabelVo.setPermSystemIdList(permSystemIdList);
        return authSystemLabelVo;
    }

    @Override
    public List<Long> validPermission(List<MobileDatasourceInfo> mobileAppList) {
        List<Long> withPermissionList = Lists.newArrayList();
        List<MobileDatasourceInfo> hasWorkList = Lists.newArrayList();
        // 查询用户授权的商品及模组的权限
        List<UserApplicationSimpleDTO> userApplicationList = iamService.queryUserApplicationSimple(true,true,true);
        for (MobileDatasourceInfo mobileDatasourceInfo : mobileAppList) {
            // 每刻应用默认有权限
            if (ObjectUtils.isNotEmpty(mobileDatasourceInfo.getSpecialSign()) &&
                    mobileDatasourceInfo.getSpecialSign().equals(Constants.SpecialSignEnum.MAYCUR.getFlag())) {
                withPermissionList.add(mobileDatasourceInfo.getId());
                continue;
            }
            // 配置的关联商品id为空，说明没有权限
            if (StringUtils.isEmpty(mobileDatasourceInfo.getIamApplicationId())) {
                continue;
            }
            // 配置的关联模组id为空，说明没有权限
            if (StringUtils.isEmpty(mobileDatasourceInfo.getIamModuleId())) {
                continue;
            }
            // 当前用户关联的商品和模组为空，说明没有权限
            if (CollectionUtils.isEmpty(userApplicationList)) {
                continue;
            }
            // 没有当前商品权限或当前商品过期，说明没有权限
            Map<String, UserApplicationSimpleDTO> userApplicationMap = userApplicationList.stream().collect(Collectors.toMap(UserApplicationSimpleDTO::getId, Function.identity()));
            UserApplicationSimpleDTO userApplication = userApplicationMap.get(mobileDatasourceInfo.getIamApplicationId());
            if (ObjectUtils.isEmpty(userApplication)) {
                continue;
            }

            // 取出当前商品下模组列表
            List<UserApplicationModulesSimpleDTO> enabledModuleList = userApplication.getEnabledModules();
            Map<String, UserApplicationModulesSimpleDTO> userApplicationModulesMap = enabledModuleList.stream().collect(Collectors.
                    toMap(UserApplicationModulesSimpleDTO::getId, Function.identity(), (a, b) -> a));
            // 没有当前模组权限或模组过期，说明没有权限
            if (!userApplicationModulesMap.containsKey(mobileDatasourceInfo.getIamModuleId())) {
                continue;
            }
            // 存在作业的自定义应用列表，后续进一步判断
            if (StringUtils.isNotEmpty(mobileDatasourceInfo.getIamWorkId())) {
                hasWorkList.add(mobileDatasourceInfo);
            } else {
                withPermissionList.add(mobileDatasourceInfo.getId());
            }
        }

        // 判断用户是否有作业权限
        if (CollectionUtils.isNotEmpty(hasWorkList)) {
            // 异步查询应用下用户的功能权限列表
            List<String> applicationIdList = hasWorkList.stream().map(MobileDatasourceInfo::getIamApplicationId).distinct().collect(Collectors.toList());
            HashMap<String, CompletableFuture<List<PermissionUserFunctionalDTO>>> applicationIdFutureMap = Maps.newHashMap(Maps.newHashMapWithExpectedSize(applicationIdList.size()));
            AuthoredUser authoredUser = AppAuthContextHolder.getContext().getAuthoredUser();
            applicationIdList.forEach(x -> {
                CompletableFuture<List<PermissionUserFunctionalDTO>> future = CompletableFuture.supplyAsync(() -> {
                    // 查询应用的用户功能权限
                    String target = Constants.IAM_FUNCTIONAL_STR + x;
                    return iamService.queryUserFunctional(target, authoredUser);
                }, asyncTaskExecutor);
                // 将所有future放到map中
                applicationIdFutureMap.put(x, future);
            });
            Collection<CompletableFuture<List<PermissionUserFunctionalDTO>>> values = applicationIdFutureMap.values();
            CompletableFuture.allOf(values.toArray(new CompletableFuture[values.size()]));

            HashMap<String, List<PermissionUserFunctionalDTO>> applicationIdMap = Maps.newHashMap();
            applicationIdFutureMap.forEach((k, v) -> {
                try {
                    List<PermissionUserFunctionalDTO> permissionUserFunctionalList = v.get();
                    if (CollectionUtils.isNotEmpty(permissionUserFunctionalList)) {
                        applicationIdMap.put(k, permissionUserFunctionalList);
                    }
                } catch (Exception e) {
                    log.error("deal query user functional result occur error", e);
                }
            });

            for (MobileDatasourceInfo mobileDatasourceInfo : hasWorkList) {
                List<PermissionUserFunctionalDTO> permissionUserFunctionalList = applicationIdMap.get(mobileDatasourceInfo.getIamApplicationId());
                // 当前应用下用户的功能权限为空，说明没有权限
                if (CollectionUtils.isEmpty(permissionUserFunctionalList)) {
                    continue;
                }

                // 按作业的target属性分组，用于判断
                Map<String, PermissionUserFunctionalDTO> userFunctionalMap = permissionUserFunctionalList.stream().collect(Collectors.
                        toMap(PermissionUserFunctionalDTO::getTarget, Function.identity(), (a, b) -> a));
                String iamWorkIdTarget = Constants.IAM_FUNCTIONAL_STR + mobileDatasourceInfo.getIamApplicationId()
                        + Constants.COLON + mobileDatasourceInfo.getIamModuleId()
                        + Constants.COLON + mobileDatasourceInfo.getIamWorkId();
                if (ObjectUtils.isEmpty(userFunctionalMap.get(iamWorkIdTarget))) {
                    continue;
                }
                withPermissionList.add(mobileDatasourceInfo.getId());
            }
        }
        return withPermissionList;
    }
}
