package com.digiwin.athena.semc.service.module.impl;

import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.digiwin.athena.appcore.auth.AppAuthContextHolder;
import com.digiwin.athena.semc.dto.module.TenantModuleDto;
import com.digiwin.athena.semc.entity.module.ModuleInfo;
import com.digiwin.athena.semc.mapper.module.ModuleInfoMapper;
import com.digiwin.athena.semc.proxy.km.service.KmService;
import com.digiwin.athena.semc.service.module.IModuleInfoService;

import com.digiwin.athena.semc.util.Utils;
import com.digiwin.athena.semc.vo.module.ModuleOrderVo;
import com.google.common.collect.Maps;
import io.github.linpeilie.Converter;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.collections.MapUtils;
import org.apache.commons.compress.utils.Lists;
import org.apache.commons.lang3.BooleanUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import javax.annotation.Resource;
import java.util.*;
import java.util.stream.Collectors;


/**
 * @description: 模块信息实现类
 * @createDate: 2025/8/25 16:45
 * @author: lilliaancheng
 */
@Slf4j
@Service
public class ModuleInfoServiceImpl extends ServiceImpl<ModuleInfoMapper, ModuleInfo> implements IModuleInfoService {

    @Autowired
    private KmService kmService;

    @Resource
    private Converter converter;

    /**
     * semc模组类型对km作业pattern的映射关系
     */
    private static final Map<String, String> patternMap;

    static {
        patternMap = new HashMap<>();
        patternMap.put("DATA_ENTRY", "DATA_ENTRY");
        patternMap.put("STATEMENT", "STATEMENT");
    }

    /**
     * 取得当前租户已排序好的作业清单(含应用/群组)
     *
     * @param moduleCode 模组类型(基础作业or报表)
     *
     * @return List<ModuleOrderVo>用于前端展示
     */
    @Override
    public List<ModuleOrderVo> queryModuleOrder(String moduleCode){
        List<TenantModuleDto> kmModuleDtos = kmService.queryTenantModuleList(patternMap.get(moduleCode));
        String tenantId = AppAuthContextHolder.getContext().getAuthoredUser().getTenantId();
        List<ModuleInfo> moduleInfos = baseMapper.queryByModuleCode(moduleCode, tenantId);
        return convertToModuleOrderVo(kmModuleDtos, moduleInfos);
    }

    /**
     * 取得当前租户设定的作业排序设置
     *
     * @param moduleCode 模组类型(基础作业or报表)
     *
     * @return List<ModuleInfo>供其他组件取得排序设置
     */
    @Override
    public List<ModuleInfo> queryTenantModuleOrder(String moduleCode){
        String tenantId = AppAuthContextHolder.getContext().getAuthoredUser().getTenantId();
        return baseMapper.queryByModuleCode(moduleCode, tenantId);
    }

    /**
     * 保存当前租户设定的作业排序
     *
     * @param moduleOrderVoList 作业排序
     * @param moduleCode 模组类型(基础作业or报表)
     *
     * @return
     */
    @Override
    @Transactional
    public Integer saveModuleOrder(List<ModuleOrderVo> moduleOrderVoList, String moduleCode){
        String tenantId = AppAuthContextHolder.getContext().getAuthoredUser().getTenantId();
        List<ModuleInfo> moduleInfos = convertToModuleInfo(moduleOrderVoList, moduleCode, tenantId);
        baseMapper.deleteByModuleCode(moduleCode, tenantId);
        return baseMapper.batchInsert(moduleInfos);
    }

    /**
     * 刪除当前租户设定的作业排序
     *
     * @param moduleCode 模组类型(基础作业or报表)
     *
     * @return
     */
    @Override
    public List<ModuleOrderVo> delModuleOrder(String moduleCode){
        String tenantId = AppAuthContextHolder.getContext().getAuthoredUser().getTenantId();
        baseMapper.deleteByModuleCode(moduleCode, tenantId);
        return queryModuleOrder(moduleCode);
    }

    private List<ModuleOrderVo> convertToModuleOrderVo(List<TenantModuleDto> tenantModuleDtos, List<ModuleInfo> moduleInfos){
        List<ModuleOrderVo> moduleOrderVos = new ArrayList<>();
        if(CollectionUtils.isNotEmpty(tenantModuleDtos)){
            Map<String,List<TenantModuleDto>> appCodeToModuleMap = tenantModuleDtos.stream().filter(e-> BooleanUtils.isNotTrue(e.getSubReport())).collect(Collectors.groupingBy(TenantModuleDto::getAppCode));;
            Map<String, List<ModuleInfo>> appCodeToCustomerModuleMap = null;
            Map<String,List<TenantModuleDto>> sortedDataMap = Maps.newLinkedHashMap();
            // 应用层级排序
            if(CollectionUtils.isNotEmpty(moduleInfos)){
                // 有租户级排序记忆
                appCodeToCustomerModuleMap = moduleInfos.stream().sorted(Comparator.comparing(ModuleInfo::getAppSort)).collect(Collectors.groupingBy(ModuleInfo::getAppCode,LinkedHashMap::new,Collectors.toList()));
                for (Map.Entry<String, List<ModuleInfo>> entry : appCodeToCustomerModuleMap.entrySet()) {
                    // 仅添加未被处理过的额外key
                    List<TenantModuleDto> remove = appCodeToModuleMap.remove(entry.getKey());
                    if(remove != null){
                        sortedDataMap.put(entry.getKey(),remove);
                    }
                }
                sortedDataMap.putAll(appCodeToModuleMap);
            }else {
                // 应用级排序 公共和app也在最前面
                if(appCodeToModuleMap.containsKey("PBD")){
                    List<TenantModuleDto> remove = appCodeToModuleMap.remove("PBD");
                    sortedDataMap.put("PBD",remove);
                }
                Iterator<Map.Entry<String, List<TenantModuleDto>>> iterator = appCodeToModuleMap.entrySet().iterator();
                while (iterator.hasNext()){
                    Map.Entry<String, List<TenantModuleDto>> next = iterator.next();
                    List<TenantModuleDto> value = next.getValue();
                    Optional<TenantModuleDto> first = value.stream().filter(TenantModuleDto::isCommonApp).findFirst();
                    if(first.isPresent()){
                        sortedDataMap.put(next.getKey(),value);
                        iterator.remove();
                    }
                }
                sortedDataMap.putAll(appCodeToModuleMap);
            }
            // 生成分组
            for (Map.Entry<String, List<TenantModuleDto>> entry : sortedDataMap.entrySet()) {
                String key = entry.getKey();
                List<TenantModuleDto> value = entry.getValue();
                if(CollectionUtils.isEmpty(value)){
                    continue;
                }
                // 生成第一层分组 app的分组
                ModuleOrderVo moduleOrderVo = new ModuleOrderVo(entry.getKey(),"APP_CODE",value.get(0).getAppName(), Lists.newArrayList());
                moduleOrderVos.add(moduleOrderVo);
                List<ModuleOrderVo> groupList = moduleOrderVo.getGroupList();
                // 是否有第二次分组 group的分组
                Optional<TenantModuleDto> first = value.stream().filter(e -> StringUtils.isBlank(e.getGroupCode())).findFirst();
                // 租户设定的分组
                List<ModuleInfo> customerModule = appCodeToCustomerModuleMap != null?appCodeToCustomerModuleMap.get(key):null;
                if(first.isPresent()){
                    // 不含有第二层分组
                    handleItemModule(groupList,value,customerModule);
                }else {
                    // 生成第二层分组
                    Map<String, Integer> groupSortMap; // 租户设定的分组
                    Map<String, List<ModuleInfo>> groupToItemModule = Maps.newHashMap() ; // 租户的分组编码对应的模块集合
                    if(CollectionUtils.isNotEmpty(customerModule)){
                        groupSortMap = Maps.newHashMap();
                        for (ModuleInfo moduleInfo : customerModule) {
                            if(StringUtils.isNotBlank(moduleInfo.getGroupCode())){
                                Integer groupNum = groupSortMap.get(moduleInfo.getGroupCode());
                                if(groupNum == null || groupNum.compareTo(moduleInfo.getGroupSort())>0){
                                    groupSortMap.put(moduleInfo.getGroupCode(),moduleInfo.getGroupSort());
                                }
                                List<ModuleInfo> moduleInfoList = groupToItemModule.computeIfAbsent(moduleInfo.getGroupCode(), k -> new ArrayList<>());
                                moduleInfoList.add(moduleInfo);
                            }

                        }
                    } else {
                        groupSortMap = null;
                    }
                    // 组装分组编码对应的模块集合
                    Map<String,List<TenantModuleDto>> groupCodeToModuleMap = Maps.newLinkedHashMap();
                    value.stream().sorted(createGroupComparator(groupSortMap)).forEach(e->{
                        List<TenantModuleDto> tenantModuleList = groupCodeToModuleMap.computeIfAbsent(e.getGroupCode(), k -> new ArrayList<>());
                        tenantModuleList.add(e);
                    });
                    // 进行第二层分组
                    groupCodeToModuleMap.forEach((k,v)->{
                        if(CollectionUtils.isNotEmpty(v)){
                            ModuleOrderVo groupModule = new ModuleOrderVo(k, "GROUP_CODE", v.get(0).getGroupName(), new ArrayList<>());
                            groupList.add(groupModule);
                            handleItemModule(groupModule.getGroupList(),v,groupToItemModule.get(k));
                        }
                    });
                }
            }


        }
        return moduleOrderVos;
    }
    // 创建基于客户模块的排序器
    private Comparator<TenantModuleDto> createItemComparator(Map<String, Integer> itemSortMap) {
        if(MapUtils.isNotEmpty(itemSortMap)){
            return Comparator.comparingInt(module ->
                    MapUtils.getInteger(itemSortMap, module.getCode(), Integer.MAX_VALUE));
        }else {
            return Comparator.comparingInt(module ->
                    Utils.toInt(module.getSort(), Integer.MAX_VALUE));
        }
    }

    private Comparator<TenantModuleDto> createGroupComparator(Map<String, Integer> groupSortMap) {
        if(MapUtils.isNotEmpty(groupSortMap)){
            return Comparator.comparingInt(module ->
                    MapUtils.getInteger(groupSortMap, module.getGroupCode(), Integer.MAX_VALUE));
        }else {
            return Comparator.comparingInt(module ->
                    Utils.toInt(module.getGroupSort(), Integer.MAX_VALUE));
        }
    }


    private void handleItemModule(List<ModuleOrderVo> groupList,List<TenantModuleDto> value,List<ModuleInfo> customerModule){
        value.stream().sorted(createItemComparator(createItemSortMap(customerModule))).forEach(e->{
            groupList.add(new ModuleOrderVo(e.getCode(),"ITEM_CODE",e.getName(),null));
        });
    }

    private Map<String, Integer> createItemSortMap(List<ModuleInfo> customerModule){
        Map<String, Integer> itemSortMap = Maps.newHashMap();
        if(CollectionUtils.isNotEmpty(customerModule)){
            for (ModuleInfo moduleInfo : customerModule) {
                Integer itemNum = itemSortMap.get(moduleInfo.getCode());
                if(itemNum == null || itemNum.compareTo(moduleInfo.getItemSort())>0){
                    itemSortMap.put(moduleInfo.getCode(),moduleInfo.getItemSort());
                }
            }
        }
        return itemSortMap;
    }





    private List<ModuleInfo> convertToModuleInfo(List<ModuleOrderVo> moduleOrderVos,String moduleCode,  String tenantId){
        List<ModuleInfo> moduleInfoList = new ArrayList<>();
        int appSort = 0;
        for (ModuleOrderVo moduleOrderVo : moduleOrderVos) {
            appSort++;
            ModuleInfo moduleInfo = new ModuleInfo();
            moduleInfo.setAppCode(moduleOrderVo.getCode());
            moduleInfo.setAppSort(appSort);
            moduleInfo.setModuleCode(moduleCode);
            moduleInfo.setTenantId(tenantId);
            List<ModuleOrderVo> groupList = moduleOrderVo.getGroupList();
            if(CollectionUtils.isEmpty(groupList)){
                continue;
            }
            ModuleOrderVo first = groupList.get(0);
            String codeType = first.getCodeType();
            if("GROUP_CODE".equals(codeType)){
                // 说明有分组
                int groupSort = 0;
                for (ModuleOrderVo groupVo : groupList) {
                    groupSort++;
                    moduleInfo.setGroupCode(groupVo.getCode());
                    moduleInfo.setGroupSort(groupSort);
                    handleSaveItemModule(moduleInfo,moduleInfoList,groupVo.getGroupList());
                }
            }else {
                handleSaveItemModule(moduleInfo,moduleInfoList,groupList);
            }
        }
        return moduleInfoList;
    }


    private void handleSaveItemModule(ModuleInfo moduleInfo,List<ModuleInfo> moduleInfoList, List<ModuleOrderVo> itemList){
        if(CollectionUtils.isEmpty(itemList)){
            return;
        }
        int index = 0;
        for (ModuleOrderVo moduleOrderVo : itemList) {
            ModuleInfo item = new ModuleInfo();
            index++;
            BeanUtils.copyProperties(moduleInfo,item);
            item.setItemSort(index);
            item.setCode(moduleOrderVo.getCode());
            moduleInfoList.add(item);
        }
    }

}