package com.digiwin.athena.semc.service.news.impl;


import com.alibaba.fastjson.JSONArray;
import com.baomidou.mybatisplus.core.toolkit.StringUtils;
import com.digiwin.athena.semc.common.enums.BizAuthTypeEnum;
import com.digiwin.athena.semc.event.NewsAuthBizIdSetEvent;
import com.digiwin.athena.semc.proxy.eoc.service.model.UserDeptDTO;
import com.digiwin.athena.semc.proxy.iam.service.model.RoleDTO;
import com.digiwin.athena.semc.util.InterceptorIgnoreUtil;
import com.digiwin.athena.semc.util.Utils;
import com.google.common.collect.Lists;

import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.digiwin.athena.appcore.auth.AppAuthContextHolder;
import com.digiwin.athena.semc.common.Constants;
import com.digiwin.athena.semc.dto.news.SaveNewsAuthReq;
import com.digiwin.athena.semc.entity.news.NewsAnnouncementAut;
import com.digiwin.athena.semc.mapper.news.NewsAnnouncementAuthMapper;
import com.digiwin.athena.semc.proxy.eoc.service.EocService;
import com.digiwin.athena.semc.proxy.iam.service.IamService;
import com.digiwin.athena.semc.proxywrapper.IamUserService;
import com.digiwin.athena.semc.proxywrapper.dto.IamUserAuthInfo;
import com.digiwin.athena.semc.service.news.NewsAnnouncementAuthService;

import com.google.common.collect.Maps;
import net.sf.json.JSONObject;
import org.apache.commons.collections.CollectionUtils;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.util.*;
import java.util.stream.Collectors;

import javax.annotation.Resource;

import lombok.extern.slf4j.Slf4j;

/**
 * 新闻公告权限表(TNewsAnnouncementAuth)表服务实现类
 *
 * @author sungqz
 * @since 2024-05-14
 */
@Service
@Slf4j
public class NewsAnnouncementAuthServiceImpl extends ServiceImpl<NewsAnnouncementAuthMapper, NewsAnnouncementAut> implements NewsAnnouncementAuthService {
    @Resource
    private NewsAnnouncementAuthMapper newsAnnouncementAuthMapper;

    @Resource
    private IamService iamService;

    @Resource
    private EocService eocService;

    @Resource
    private IamUserService iamUserService;

    /**
     * 保存公告的权限
     *
     * @param saveNewsAuthReq 入参
     * @return 返回
     */
    @Override
    @Transactional
    public void saveNewsAuth(SaveNewsAuthReq saveNewsAuthReq) {
        // 先删除当前权限关系
        QueryWrapper<NewsAnnouncementAut> condition = new QueryWrapper<>();
        condition.eq("news_id", saveNewsAuthReq.getNewsId());
        newsAnnouncementAuthMapper.delete(condition);

        // 全员可见
        if (Constants.NEWS_ALL_AUTH_FLAG_YES.equals(saveNewsAuthReq.getAllAuthFlag())) {
            NewsAnnouncementAut newsAnnouncementAuth = new NewsAnnouncementAut();
            newsAnnouncementAuth.setNewsId(saveNewsAuthReq.getNewsId());
            newsAnnouncementAuth.setAllAuthFlag(Constants.NEWS_ALL_AUTH_FLAG_YES);
            this.save(newsAnnouncementAuth);
        } else {
            NewsAnnouncementAut newsAnnouncementAuth = new NewsAnnouncementAut();
            newsAnnouncementAuth.setNewsId(saveNewsAuthReq.getNewsId());
            newsAnnouncementAuth.setAuthId(saveNewsAuthReq.getAuthId());
            newsAnnouncementAuth.setAuthName(saveNewsAuthReq.getAuthName());
            newsAnnouncementAuth.setAuthType(saveNewsAuthReq.getAuthType());
            this.save(newsAnnouncementAuth);
        }
    }

    /**
     * 查询当前用户所属的公告id列表
     *
     * @return 返回
     */
    public List<NewsAnnouncementAut> queryAuthNewsList() {
        return queryAuthNewsListByNewsTypeIdList(null);
    }

    @Override
    public List<NewsAnnouncementAut> queryAuthNewsListByNewsTypeIdList(List<Long> newsTypeIdList) {
        String userId = AppAuthContextHolder.getContext().getAuthoredUser().getUserId();
        String userToken = AppAuthContextHolder.getContext().getAuthoredUser().getToken();
        String tenantId = AppAuthContextHolder.getContext().getAuthoredUser().getTenantId();
        IamUserAuthInfo iamUserAuthInfo = iamUserService.getUserAuthSids(userId, tenantId, userToken);
        // 查询用户所在组织
        List<Long> authIds = Lists.newArrayList();
        // 拼接组织条件
        if (iamUserAuthInfo != null && CollectionUtils.isNotEmpty(iamUserAuthInfo.getOrgSids())) {
            authIds.addAll(iamUserAuthInfo.getOrgSids());
        }
        // 查询用户所在角色
        // 拼接角色条件
        if (iamUserAuthInfo != null && CollectionUtils.isNotEmpty(iamUserAuthInfo.getRoleSids())) {
            authIds.addAll(iamUserAuthInfo.getRoleSids());
        }
        // 查询用户所在部门
        // 拼接部门条件
        if (iamUserAuthInfo != null && CollectionUtils.isNotEmpty(iamUserAuthInfo.getDeptSids())) {
            authIds.addAll(iamUserAuthInfo.getDeptSids());
        }
        // 拼接用户条件
        Long userSid = AppAuthContextHolder.getContext().getAuthoredUser().getSid();
        if (Objects.nonNull(userSid)) {
            authIds.add(userSid);
        }
        List<NewsAnnouncementAut> authList = Lists.newArrayList();
        if (CollectionUtils.isNotEmpty(authIds)) {
            Collections.sort(authIds);
            LambdaQueryWrapper<NewsAnnouncementAut> queryWrapper = new LambdaQueryWrapper<>();
            queryWrapper.in(NewsAnnouncementAut::getAuthId, authIds);
            queryWrapper.in(CollectionUtils.isNotEmpty(newsTypeIdList), NewsAnnouncementAut::getNewsTypeId, newsTypeIdList);
            queryWrapper.select(NewsAnnouncementAut::getNewsId);
            authList.addAll(newsAnnouncementAuthMapper.selectList(queryWrapper));

        }
        LambdaQueryWrapper<NewsAnnouncementAut> queryWrapper = new LambdaQueryWrapper<>();
        queryWrapper.eq(NewsAnnouncementAut::getAllAuthFlag, Constants.NEWS_ALL_AUTH_FLAG_YES);
        queryWrapper.in(CollectionUtils.isNotEmpty(newsTypeIdList), NewsAnnouncementAut::getNewsTypeId, newsTypeIdList);
        queryWrapper.select(NewsAnnouncementAut::getNewsId);
        List<NewsAnnouncementAut> allAuthList = newsAnnouncementAuthMapper.selectList(queryWrapper);
        authList.addAll(allAuthList);
        return authList;
    }

    /**
     * 设置公告权限表的权限业务id
     *
     * @param event 事件
     */
    @Override
    public void setNewsAuthBizId(NewsAuthBizIdSetEvent event) {
        // 外部用户
        List<NewsAnnouncementAut> announcementAutList = event.getAnnouncementAutList();
        if (CollectionUtils.isNotEmpty(announcementAutList)) {
            updateExternalUserId(announcementAutList);
            return;
        }

        // 查询当前租户下配置的公告权限对象
        String tenantId = Utils.getTenantId();
        LambdaQueryWrapper<NewsAnnouncementAut> queryWrapper = new LambdaQueryWrapper<>();
        queryWrapper.eq(NewsAnnouncementAut::getAllAuthFlag, Constants.NEWS_ALL_AUTH_FLAG_NO);
        List<NewsAnnouncementAut> newsAnnouncementAutList = newsAnnouncementAuthMapper.selectList(queryWrapper);

        // 组织列表，存在权限业务id为空时，则更新
        List<NewsAnnouncementAut> orgAuthList = newsAnnouncementAutList.stream().filter(x -> BizAuthTypeEnum.ORG.getValue().equals(x.getAuthType())
                && StringUtils.isBlank(x.getAuthBizId())).collect(Collectors.toList());
        if (CollectionUtils.isNotEmpty(orgAuthList)) {
            Map<Long, String> orgSidMap = Maps.newHashMap();
            JSONArray companyFactory = iamService.getCompanyFactory();
            for (Object o : companyFactory) {
                parseAuthIdMap(o, orgSidMap);
            }
            orgAuthList.forEach(x -> {
                x.setAuthBizId(orgSidMap.get(x.getAuthId()));
                x.setCreateTenantId(tenantId);
            });
            this.saveOrUpdateBatch(orgAuthList);
        }

        // 角色列表，存在权限业务id为空时，则更新
        List<NewsAnnouncementAut> roleAuthList = newsAnnouncementAutList.stream().filter(x -> BizAuthTypeEnum.ROLE.getValue().equals(x.getAuthType())
                && StringUtils.isBlank(x.getAuthBizId())).collect(Collectors.toList());
        if (CollectionUtils.isNotEmpty(roleAuthList)) {
            List<RoleDTO> roleList = iamService.queryAllRoleList();
            Map<Long, String> roleSidMap = roleList.stream().collect(Collectors.toMap(RoleDTO::getSid, RoleDTO::getId, (a, b) -> a));

            roleAuthList.forEach(x -> {
                x.setAuthBizId(roleSidMap.get(x.getAuthId()));
                x.setCreateTenantId(tenantId);
            });
            this.saveOrUpdateBatch(roleAuthList);
        }

        // 部门列表，存在权限业务id为空时，则更新
        List<NewsAnnouncementAut> deptAuthList = newsAnnouncementAutList.stream().filter(x -> BizAuthTypeEnum.DEPARTMENT.getValue().equals(x.getAuthType())
                && StringUtils.isBlank(x.getAuthBizId())).collect(Collectors.toList());
        if (CollectionUtils.isNotEmpty(deptAuthList)) {
            List<UserDeptDTO> deptList = eocService.queryTenantDeptList();
            Map<Long, String> deptSidMap = deptList.stream().collect(Collectors.toMap(UserDeptDTO::getSid, UserDeptDTO::getId, (a, b) -> a));

            deptAuthList.forEach(x -> {
                x.setAuthBizId(deptSidMap.get(x.getAuthId()));
                x.setCreateTenantId(tenantId);
            });
            this.saveOrUpdateBatch(deptAuthList);
        }

        // 用户列表，存在权限业务id为空时，则更新
        List<NewsAnnouncementAut> userAuthList = newsAnnouncementAutList.stream().filter(x -> Arrays.asList(BizAuthTypeEnum.USER.getValue(), BizAuthTypeEnum.EXTERNAL_USER.getValue()).contains(x.getAuthType())
                && StringUtils.isBlank(x.getAuthBizId())).collect(Collectors.toList());
        if (CollectionUtils.isNotEmpty(userAuthList)) {
            updateExternalUserId(userAuthList);
        }
    }

    /**
     * 更新外部用户id
     *
     * @param userAuthList 用户权限列表
     */
    private void updateExternalUserId(List<NewsAnnouncementAut> userAuthList) {
        List<RoleDTO> allUserList = iamService.queryEnabledUserList(); // 所有用户
        Map<Long, String> userSidMap = allUserList.stream().collect(Collectors.toMap(RoleDTO::getSid, RoleDTO::getId, (a, b) -> a));

        userAuthList.forEach(x -> {
            x.setAuthBizId(userSidMap.get(x.getAuthId()));
            x.setCreateTenantId(Utils.getTenantId());
        });
        InterceptorIgnoreUtil.handler(() -> this.saveOrUpdateBatch(userAuthList));
    }

    /**
     * 获取所有的组织id列表
     *
     * @param object
     * @param map
     */
    private void parseAuthIdMap(Object object, Map<Long, String> map) {
        JSONObject data = JSONObject.fromObject(object);
        map.put(Long.valueOf(data.get("key").toString()), data.get("id").toString());
        // 遍历子节点
        if (!data.containsKey("children")) {
            return;
        }
        net.sf.json.JSONArray children = data.getJSONArray("children");
        for (Object o : children) {
            parseAuthIdMap(o, map);
        }
    }
}
