package com.digiwin.athena.semc.service.news.impl;

import com.google.common.collect.Lists;

import com.alibaba.fastjson.JSON;
import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.conditions.update.LambdaUpdateWrapper;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.digiwin.athena.appcore.auth.AppAuthContextHolder;
import com.digiwin.athena.appcore.auth.domain.AuthoredUser;
import com.digiwin.athena.semc.common.Constants;
import com.digiwin.athena.semc.common.PageInfoResp;
import com.digiwin.athena.semc.common.enums.MQMessageTypeEnum;
import com.digiwin.athena.semc.dto.news.MyNewsFavoritePageReq;
import com.digiwin.athena.semc.dto.news.NewsCommonIdListReq;
import com.digiwin.athena.semc.dto.news.NewsCommonIdReq;
import com.digiwin.athena.semc.entity.news.NewsAnnouncement;
import com.digiwin.athena.semc.entity.news.NewsAnnouncementFavorite;
import com.digiwin.athena.semc.env.EnvProperties;
import com.digiwin.athena.semc.mapper.news.NewsAnnouncementFavoriteMapper;
import com.digiwin.athena.semc.mq.dto.MQMessageDTO;
import com.digiwin.athena.semc.mq.dto.NewsCancelFavoriteMessageDTO;
import com.digiwin.athena.semc.mq.dto.NewsFavoriteMessageDTO;
import com.digiwin.athena.semc.mq.sender.RabbitMessageSender;
import com.digiwin.athena.semc.proxy.iam.service.IamService;
import com.digiwin.athena.semc.service.cache.LockClient;
import com.digiwin.athena.semc.service.news.NewsAnnouncementFavoriteService;
import com.digiwin.athena.semc.service.news.NewsAnnouncementReadService;
import com.digiwin.athena.semc.service.news.NewsAnnouncementService;
import com.digiwin.athena.semc.util.DmcFileUtil;
import com.digiwin.athena.semc.util.FormatUtil;
import com.digiwin.athena.semc.util.InterceptorIgnoreUtil;
import com.digiwin.athena.semc.vo.file.FileInfoVO;
import com.digiwin.athena.semc.vo.news.MyNewsAnnouncementFavoriteVO;
import com.digiwin.athena.semc.vo.news.ValidateNewsVO;

import org.apache.commons.collections.MapUtils;
import org.apache.commons.lang3.ObjectUtils;
import org.apache.commons.lang3.StringUtils;
import org.redisson.api.RLock;
import org.springframework.stereotype.Service;

import java.util.Collections;
import java.util.Date;
import java.util.List;
import java.util.Map;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;

import cn.hutool.core.date.DateUtil;
import cn.hutool.core.util.IdUtil;
import cn.hutool.crypto.digest.DigestUtil;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;

@Service
@Slf4j
@RequiredArgsConstructor
public class NewsAnnouncementFavoriteServiceImpl extends ServiceImpl<NewsAnnouncementFavoriteMapper, NewsAnnouncementFavorite> implements NewsAnnouncementFavoriteService {

    private final NewsAnnouncementService newsAnnouncementService;
    private final LockClient lockClient;
    private final IamService iamService;
    private final RabbitMessageSender rabbitMessageSender;
    private final EnvProperties envProperties;
    private final DmcFileUtil dmcFileUtil;
    private final NewsAnnouncementReadService newsAnnouncementReadService;

    @Override
    public Integer favoriteNews(NewsCommonIdReq newsCommonIdReq) {
        AuthoredUser authoredUser = AppAuthContextHolder.getContext().getAuthoredUser();
        String lockKey = FormatUtil.format("newsFavorite:{}:{}:{}", newsCommonIdReq.getId(), authoredUser.getTenantId(), authoredUser.getSid());
        RLock rLock = null;
        try {
            rLock = lockClient.getLock(lockKey);
            boolean lockResult = rLock.tryLock(1, 5, TimeUnit.SECONDS);
            if (!lockResult) {
                log.info("未获取到锁");
                return 1;
            }
            NewsAnnouncementFavorite currentNewsFavorite = getCurrentNewsFavorite(newsCommonIdReq.getId(), authoredUser);
            boolean existNewsFavorite = currentNewsFavorite != null;
            if (existNewsFavorite) {
                log.info("已收藏");
                return 1;
            }
            //校验公告
            ValidateNewsVO validateNewsVO = newsAnnouncementService.validateNews(newsCommonIdReq.getId());
            if (!ValidateNewsVO.CODE_1000.equals(validateNewsVO.getCode())) {
                log.warn("当前公告不存在：{}，{}", newsCommonIdReq.getId(), validateNewsVO.getCode());
                return 2;
            }
            //执行收藏
            NewsAnnouncement newsAnnouncement = validateNewsVO.getNewsAnnouncement();
            NewsAnnouncementFavorite newsAnnouncementFavorite = new NewsAnnouncementFavorite();
            newsAnnouncementFavorite.setNewsId(newsAnnouncement.getId());
            newsAnnouncementFavorite.setThirdNewsId(newsAnnouncement.getThirdNewsId());
            newsAnnouncementFavorite.setThirdAppId(newsAnnouncement.getThirdAppId());
            newsAnnouncementFavorite.setThirdAppCode(newsAnnouncement.getThirdAppCode());
            newsAnnouncementFavorite.setCreateUserId(authoredUser.getUserId());
            newsAnnouncementFavorite.setCreateUserSid(authoredUser.getSid());
            newsAnnouncementFavorite.setCreateUserName(authoredUser.getUserName());
            newsAnnouncementFavorite.setThirdUserId("");
            // 三方公告
            if (ObjectUtils.isNotEmpty(newsAnnouncement.getThirdNewsId()) && Constants.ThirdNewsAnnouncementEnum.OUTSIDER_INFORMATION.getType().equals(newsAnnouncement.getNewsSource())) {
                String verifyUserId = iamService.queryMappingEmpId(newsAnnouncement.getThirdAppCode());
                if (StringUtils.isNotBlank(verifyUserId)) {
                    newsAnnouncementFavorite.setThirdUserId(verifyUserId);
                }
            }
            newsAnnouncementFavorite.setTenantId(authoredUser.getTenantId());
            newsAnnouncementFavorite.setCreateTime(new Date());
            newsAnnouncementFavorite.setModifyUserId(authoredUser.getUserId());
            newsAnnouncementFavorite.setModifyTime(new Date());
            boolean resultFlag = save(newsAnnouncementFavorite);
            if (!resultFlag) {
                log.warn("收藏失败");
                return 3;
            }
            MQMessageDTO messageDTO = new MQMessageDTO();
            messageDTO.setMessageId(IdUtil.randomUUID());
            NewsFavoriteMessageDTO newsFavoriteMessageDTO = new NewsFavoriteMessageDTO();
            newsFavoriteMessageDTO.setNewsId(newsAnnouncement.getId());
            messageDTO.setMessage(JSON.toJSONString(newsFavoriteMessageDTO));
            messageDTO.setMessageType(MQMessageTypeEnum.NEWS_FAVORITE.getCode());
            messageDTO.setRetryMaxCount(3);
            rabbitMessageSender.sendWorkCommonMsg(messageDTO);
            return 0;
        } catch (InterruptedException e) {
            Thread.currentThread().interrupt();
            log.warn("收藏异常：{}", e.getMessage());
            return 3;
        } finally {
            lockClient.unlock(rLock);
        }
    }

    private NewsAnnouncementFavorite getCurrentNewsFavorite(Long newsId, AuthoredUser authoredUser) {
        LambdaQueryWrapper<NewsAnnouncementFavorite> queryWrapper = new LambdaQueryWrapper<>();
        queryWrapper.eq(NewsAnnouncementFavorite::getNewsId, newsId);
        queryWrapper.eq(NewsAnnouncementFavorite::getCreateUserSid, authoredUser.getSid());
        return getOne(queryWrapper);
    }


    @Override
    public Integer cancelFavoriteNews(NewsCommonIdListReq newsCommonIdListReq) {
        AuthoredUser authoredUser = AppAuthContextHolder.getContext().getAuthoredUser();
        String idsMd5 = newsCommonIdListReq.getIdList().get(0).toString();
        if (newsCommonIdListReq.getIdList().size()>1) {
            Collections.sort(newsCommonIdListReq.getIdList());
            idsMd5 = DigestUtil.md5Hex(StringUtils.join(newsCommonIdListReq.getIdList(), ","));
        }
        String lockKey = FormatUtil.format("cancelFavoriteNews:{}:{}:{}", idsMd5, authoredUser.getTenantId(), authoredUser.getSid());
        RLock rLock = null;
        try {
            rLock = lockClient.getLock(lockKey);
            boolean lockResult = rLock.tryLock(1, 5, TimeUnit.SECONDS);
            if (!lockResult) {
                log.info("未获取到锁");
                return 1;
            }
            LambdaQueryWrapper<NewsAnnouncementFavorite> queryWrapper = new LambdaQueryWrapper<>();
            queryWrapper.in(NewsAnnouncementFavorite::getNewsId, newsCommonIdListReq.getIdList());
            queryWrapper.eq(NewsAnnouncementFavorite::getCreateUserSid, authoredUser.getSid());
            boolean removeFlag = remove(queryWrapper);
            if (!removeFlag) {
                log.info("删除收藏记录失败");
                return 1;
            }
            MQMessageDTO messageDTO = new MQMessageDTO();
            messageDTO.setMessageId(IdUtil.randomUUID());
            NewsCancelFavoriteMessageDTO newsCancelFavoriteMessageDTO = new NewsCancelFavoriteMessageDTO();
            newsCancelFavoriteMessageDTO.setNewsIdList(newsCommonIdListReq.getIdList());
            messageDTO.setMessage(JSON.toJSONString(newsCancelFavoriteMessageDTO));
            messageDTO.setMessageType(MQMessageTypeEnum.NEWS_CANCEL_FAVORITE.getCode());
            messageDTO.setRetryMaxCount(3);
            rabbitMessageSender.sendWorkCommonMsg(messageDTO);
            return 0;
        } catch (InterruptedException e) {
            Thread.currentThread().interrupt();
            log.warn("取消收藏异常：{}", e.getMessage());
            return 2;
        } finally {
            lockClient.unlock(rLock);
        }
    }

    @Override
    public PageInfoResp<MyNewsAnnouncementFavoriteVO> myFavoriteNewsList(MyNewsFavoritePageReq myNewsFavoritePageReq) {
        AuthoredUser authoredUser = AppAuthContextHolder.getContext().getAuthoredUser();
        LambdaQueryWrapper<NewsAnnouncementFavorite> queryWrapper = new LambdaQueryWrapper<>();
        queryWrapper.eq(NewsAnnouncementFavorite::getCreateUserSid, authoredUser.getSid());
        queryWrapper.orderByDesc(NewsAnnouncementFavorite::getId);
        queryWrapper.select(NewsAnnouncementFavorite::getId, NewsAnnouncementFavorite::getNewsId, NewsAnnouncementFavorite::getCreateTime);
        PageInfoResp<MyNewsAnnouncementFavoriteVO> pageInfoResp = new PageInfoResp<>();
        pageInfoResp.setPageSize(myNewsFavoritePageReq.getPageSize());
        pageInfoResp.setPageNo(myNewsFavoritePageReq.getPageNum());
        Page<NewsAnnouncementFavorite> page = page(new Page<>(myNewsFavoritePageReq.getPageNum(), myNewsFavoritePageReq.getPageSize()), queryWrapper);
        if (page==null || ObjectUtils.isEmpty(page.getRecords())) {
            return pageInfoResp;
        }
        List<NewsAnnouncementFavorite> newsFavoriteList = page.getRecords();
        List<String> coverImageIdList = Lists.newArrayList();
        List<MyNewsAnnouncementFavoriteVO> resultList = Lists.newArrayList();
        //已读数为空的公告（这个是为了兼容老历史数据，没刷入read_count的情况）
        List<Long> zeroNewsIdReadList = Lists.newArrayList();
        Map<Long, ValidateNewsVO> validateNewsMap = newsAnnouncementService.validateNews(newsFavoriteList.stream().map(NewsAnnouncementFavorite::getNewsId).collect(Collectors.toList()));
        for (NewsAnnouncementFavorite newsFavorite :newsFavoriteList) {
            ValidateNewsVO validateNewsVO = validateNewsMap.get(newsFavorite.getNewsId());
            MyNewsAnnouncementFavoriteVO myNewsAnnouncementFavoriteVO = new MyNewsAnnouncementFavoriteVO();
            myNewsAnnouncementFavoriteVO.setId(newsFavorite.getId());
            myNewsAnnouncementFavoriteVO.setFavoriteTime(DateUtil.formatDateTime(newsFavorite.getCreateTime()));
            myNewsAnnouncementFavoriteVO.setNewsStatusCode(validateNewsVO.getCode());
            myNewsAnnouncementFavoriteVO.setNewsId(newsFavorite.getNewsId());
            NewsAnnouncement newsAnnouncement = validateNewsVO.getNewsAnnouncement();
            if (newsAnnouncement!=null) {
                myNewsAnnouncementFavoriteVO.setNewsTitle(newsAnnouncement.getNewsTitle());
                myNewsAnnouncementFavoriteVO.setSummary(newsAnnouncement.getSummary());
                myNewsAnnouncementFavoriteVO.setNewsEffectiveTime(newsAnnouncement.getNewsEffectiveTime());
                myNewsAnnouncementFavoriteVO.setLikeCount(newsAnnouncement.getLikeCount());
                myNewsAnnouncementFavoriteVO.setReadCount(newsAnnouncement.getReadCount());
                if (newsAnnouncement.getReadCount()==null || newsAnnouncement.getReadCount()==0) {
                    zeroNewsIdReadList.add(newsFavorite.getNewsId());
                }
                // 移动端封面图需要拼接域名
                if (StringUtils.isNotBlank(newsAnnouncement.getDefaultImageIcon())) {
                    myNewsAnnouncementFavoriteVO.setDefaultImageIcon(envProperties.getSemcWebUrl() + newsAnnouncement.getDefaultImageIcon());
                }
                coverImageIdList.add(newsAnnouncement.getCoverImageId());
                myNewsAnnouncementFavoriteVO.setCoverImageId(newsAnnouncement.getCoverImageId());
            }
            resultList.add(myNewsAnnouncementFavoriteVO);
        }
        //获取封面图信息（性能考虑，批量查询）和兼容历史公告没有已读数的情况
        Map<String, Object> coverImageFileIdMap = dmcFileUtil.queryFileInfoList(coverImageIdList);
        Map<Long, Long> readCountMap = newsAnnouncementReadService.queryReadCount(zeroNewsIdReadList);
        for (MyNewsAnnouncementFavoriteVO x : resultList) {
            // 封面图
            if (StringUtils.isNotBlank(x.getCoverImageId()) && MapUtils.isNotEmpty(coverImageFileIdMap)) {
                List<FileInfoVO> fileList = newsAnnouncementService.buildDetailFileInfo(x.getCoverImageId().split(","), coverImageFileIdMap);
                x.setCoverImageInfo(fileList.get(0));
                x.setCoverImageId("");
            }
            //已读数
            if (x.getReadCount() == null || x.getReadCount() == 0) {
                Long readCount = readCountMap.getOrDefault(x.getNewsId(), 0L);
                x.setReadCount(readCount.intValue());
            }
        }
        pageInfoResp.setList(resultList);
        pageInfoResp.setTotalRecords((int) page.getTotal());
        pageInfoResp.setTotalPages((int) page.getPages());
        return pageInfoResp;
    }



    @Override
    public void handlerFavoriteNews(NewsFavoriteMessageDTO newsFavoriteMessageDTO) {
        updateFavoriteCount(newsFavoriteMessageDTO.getNewsId());
    }

    @Override
    public void handlerCancelFavoriteNews(NewsCancelFavoriteMessageDTO newsCancelFavoriteMessageDTO) {
        newsCancelFavoriteMessageDTO.getNewsIdList().forEach(this::updateFavoriteCount);
    }


    /**
     * 获取公告总收藏数，然后更新文章收藏总数
     */
    private void updateFavoriteCount(Long newsId) {
        LambdaQueryWrapper<NewsAnnouncementFavorite> queryWrapper = new LambdaQueryWrapper<>();
        queryWrapper.eq(NewsAnnouncementFavorite::getNewsId, newsId);
        Long size = InterceptorIgnoreUtil.handler(() -> count(queryWrapper));
        if (size != null && size > -1) {
            InterceptorIgnoreUtil.handler(() -> {
                newsAnnouncementService.update(new LambdaUpdateWrapper<NewsAnnouncement>()
                        .eq(NewsAnnouncement::getId, newsId)
                        .set(NewsAnnouncement::getFavoriteCount, size));
            });
        }
    }

}
