package com.digiwin.athena.semc.service.news.impl;

import com.alibaba.fastjson.JSON;
import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.conditions.update.LambdaUpdateWrapper;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.digiwin.athena.appcore.auth.AppAuthContextHolder;
import com.digiwin.athena.appcore.auth.domain.AuthoredUser;
import com.digiwin.athena.semc.common.Constants;
import com.digiwin.athena.semc.common.enums.MQMessageTypeEnum;
import com.digiwin.athena.semc.dto.news.NewLikeReqDTO;
import com.digiwin.athena.semc.entity.news.NewsAnnouncement;
import com.digiwin.athena.semc.entity.news.NewsAnnouncementLike;
import com.digiwin.athena.semc.mapper.news.NewsAnnouncementLikeMapper;
import com.digiwin.athena.semc.mq.dto.MQMessageDTO;
import com.digiwin.athena.semc.mq.dto.NewsLikeMessageDTO;
import com.digiwin.athena.semc.mq.sender.RabbitMessageSender;
import com.digiwin.athena.semc.proxy.iam.service.IamService;
import com.digiwin.athena.semc.service.cache.LockClient;
import com.digiwin.athena.semc.service.news.NewsAnnouncementLikeService;
import com.digiwin.athena.semc.service.news.NewsAnnouncementService;
import com.digiwin.athena.semc.util.FormatUtil;
import com.digiwin.athena.semc.util.InterceptorIgnoreUtil;
import com.digiwin.athena.semc.vo.news.ValidateNewsVO;

import org.apache.commons.lang3.ObjectUtils;
import org.apache.commons.lang3.StringUtils;
import org.redisson.api.RLock;
import org.springframework.stereotype.Service;

import java.util.Date;
import java.util.concurrent.TimeUnit;

import cn.hutool.core.util.IdUtil;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;

@Service
@Slf4j
@RequiredArgsConstructor
public class NewsAnnouncementLikeServiceImpl extends ServiceImpl<NewsAnnouncementLikeMapper, NewsAnnouncementLike> implements NewsAnnouncementLikeService {

    private final NewsAnnouncementService newsAnnouncementService;
    private final LockClient lockClient;
    private final IamService iamService;
    private final RabbitMessageSender rabbitMessageSender;


    @Override
    public Integer like(NewLikeReqDTO newLikeReqDTO, boolean likeFlag) {
        AuthoredUser authoredUser = AppAuthContextHolder.getContext().getAuthoredUser();
        String lockKey = FormatUtil.format("newsLike:{}:{}:{}:{}", newLikeReqDTO.getId(), authoredUser.getTenantId(), authoredUser.getSid(), likeFlag);
        RLock rLock = null;
        try {
            rLock = lockClient.getLock(lockKey);
            boolean lockResult = rLock.tryLock(1, 5, TimeUnit.SECONDS);
            if (!lockResult) {
                log.info("未获取到锁");
                return 1;
            }
            NewsAnnouncementLike currentNewsLike = getCurrentNewsLike(newLikeReqDTO.getId(), authoredUser);
            boolean existNewsLike = currentNewsLike != null;
            if (likeFlag) {
                if (existNewsLike) {
                    log.info("已点赞");
                    return 1;
                }
            } else {
                if (!existNewsLike) {
                    log.info("没有点赞记录");
                    return 1;
                }
            }
            //校验公告
            ValidateNewsVO validateNewsVO = newsAnnouncementService.validateNews(newLikeReqDTO.getId());
            if (!ValidateNewsVO.CODE_1000.equals(validateNewsVO.getCode())) {
                log.warn("当前公告不存在：{}，{}", newLikeReqDTO.getId(), validateNewsVO.getCode());
                return 2;
            }
            //执行点赞或者取消点赞
            boolean resultFlag;
            if (likeFlag) {
                NewsAnnouncement newsAnnouncement = validateNewsVO.getNewsAnnouncement();
                NewsAnnouncementLike newsAnnouncementLike = new NewsAnnouncementLike();
                newsAnnouncementLike.setNewsId(newsAnnouncement.getId());
                newsAnnouncementLike.setThirdNewsId(newsAnnouncement.getThirdNewsId());
                newsAnnouncementLike.setThirdAppId(newsAnnouncement.getThirdAppId());
                newsAnnouncementLike.setThirdAppCode(newsAnnouncement.getThirdAppCode());
                newsAnnouncementLike.setCreateUserId(authoredUser.getUserId());
                newsAnnouncementLike.setCreateUserSid(authoredUser.getSid());
                newsAnnouncementLike.setCreateUserName(authoredUser.getUserName());
                newsAnnouncementLike.setThirdUserId("");
                // 三方公告
                if (ObjectUtils.isNotEmpty(newsAnnouncement.getThirdNewsId()) && Constants.ThirdNewsAnnouncementEnum.OUTSIDER_INFORMATION.getType().equals(newsAnnouncement.getNewsSource())) {
                    String verifyUserId = iamService.queryMappingEmpId(newsAnnouncement.getThirdAppCode());
                    if (StringUtils.isNotBlank(verifyUserId)) {
                        newsAnnouncementLike.setThirdUserId(verifyUserId);
                    }
                }
                newsAnnouncementLike.setTenantId(authoredUser.getTenantId());
                newsAnnouncementLike.setCreateTime(new Date());
                newsAnnouncementLike.setModifyUserId(authoredUser.getUserId());
                newsAnnouncementLike.setModifyTime(new Date());
                resultFlag = save(newsAnnouncementLike);
            } else {
                resultFlag = InterceptorIgnoreUtil.handler(() -> removeById(currentNewsLike.getId()));
            }
            if (!resultFlag) {
                log.warn("点赞或取消失败");
                return 1;
            }
            MQMessageDTO messageDTO = new MQMessageDTO();
            messageDTO.setMessageId(IdUtil.randomUUID());
            NewsLikeMessageDTO newsLikeMessageDTO = new NewsLikeMessageDTO();
            newsLikeMessageDTO.setLikeFlag(likeFlag);
            newsLikeMessageDTO.setAuthoredUser(authoredUser);
            newsLikeMessageDTO.setNewsAnnouncement(validateNewsVO.getNewsAnnouncement());
            messageDTO.setMessage(JSON.toJSONString(newsLikeMessageDTO));
            messageDTO.setMessageType(MQMessageTypeEnum.NEWS_LIKE.getCode());
            messageDTO.setRetryMaxCount(3);
            rabbitMessageSender.sendWorkCommonMsg(messageDTO);
            return 0;
        } catch (InterruptedException e) {
            Thread.currentThread().interrupt();
            log.warn("点赞或取消异常：{}", e.getMessage());
            return 1;
        } finally {
            lockClient.unlock(rLock);
        }
    }

    @Override
    public void handlerLikeEvent(NewsLikeMessageDTO newsLikeMessageDTO) {
        NewsAnnouncement newsAnnouncement = newsLikeMessageDTO.getNewsAnnouncement();
        updateLikeCount(newsAnnouncement.getId());
    }


    private NewsAnnouncementLike getCurrentNewsLike(Long newsId, AuthoredUser authoredUser){
        LambdaQueryWrapper<NewsAnnouncementLike> queryWrapper = new LambdaQueryWrapper<>();
        queryWrapper.eq(NewsAnnouncementLike::getNewsId, newsId);
        queryWrapper.eq(NewsAnnouncementLike::getCreateUserSid, authoredUser.getSid());
        queryWrapper.eq(NewsAnnouncementLike::getTenantId, authoredUser.getTenantId());
        return InterceptorIgnoreUtil.handler(() -> getOne(queryWrapper));
    }


    /**
     * 获取公告总点赞数，然后更新文章点赞总数
     * @param newsId
     */
    private void updateLikeCount(Long newsId){
        LambdaQueryWrapper<NewsAnnouncementLike> queryWrapper = new LambdaQueryWrapper<>();
        queryWrapper.eq(NewsAnnouncementLike::getNewsId, newsId);
        Long size =InterceptorIgnoreUtil.handler(() -> count(queryWrapper));
        if (size!=null && size>-1) {
            InterceptorIgnoreUtil.handler(() -> {
                newsAnnouncementService.update(new LambdaUpdateWrapper<NewsAnnouncement>()
                        .eq(NewsAnnouncement::getId, newsId)
                        .set(NewsAnnouncement::getLikeCount, size));
            });
        }
    }


}