package com.digiwin.athena.semc.service.news.impl;

import com.google.common.collect.Maps;

import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.digiwin.athena.appcore.auth.AppAuthContextHolder;
import com.digiwin.athena.semc.dto.news.AnnouncementIdReq;
import com.digiwin.athena.semc.entity.news.NewsAnnouncementRead;
import com.digiwin.athena.semc.mapper.news.NewsAnnouncementMapper;
import com.digiwin.athena.semc.mapper.news.NewsAnnouncementReadMapper;
import com.digiwin.athena.semc.proxy.iam.service.IamService;
import com.digiwin.athena.semc.service.cache.RedisLock;
import com.digiwin.athena.semc.service.news.NewsAnnouncementReadService;
import com.digiwin.athena.semc.util.DateUtils;
import com.digiwin.athena.semc.util.InterceptorIgnoreUtil;

import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.ObjectUtils;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

import javax.annotation.Resource;

/**
 * 新闻公告已读表(TNewsAnnouncementRead)表服务实现类
 *
 * @author sungqz
 * @since 2024-05-15
 */
@Service
public class NewsAnnouncementReadServiceImpl extends ServiceImpl<NewsAnnouncementReadMapper, NewsAnnouncementRead> implements NewsAnnouncementReadService {

    @Resource
    private NewsAnnouncementReadMapper newsAnnouncementReadMapper;

    @Resource
    private NewsAnnouncementMapper newsAnnouncementMapper;

    @Resource
    private IamService iamService;

    /**
     * web端/移动端-读取公告，记录为已读公告
     *
     * @param announcementIdReq 入参
     */
    @Override
    @Transactional
    public Boolean insertNews(AnnouncementIdReq announcementIdReq) {
        String tenantId = AppAuthContextHolder.getContext().getAuthoredUser().getTenantId();
        String userId = AppAuthContextHolder.getContext().getAuthoredUser().getUserId();
        RedisLock redisLock = new RedisLock("insertNews-" + tenantId + "-" + userId);
        try {
            // 获取分布式锁，防止重复记录
            if (!redisLock.lock()) {
                return Boolean.TRUE;
            }
        } catch (InterruptedException e) {
            log.error("obtain distributed lock error.", e);
            return Boolean.FALSE;
        }
        // 业务处理
        Boolean respFlag;
        try {
            // 查询当前用户是否已读当前公告，如果未读过则记录为已读
            NewsAnnouncementRead readInfo = queryNewsAnnouncementReadInfo(announcementIdReq);
            if (ObjectUtils.isNotEmpty(readInfo)) {
                return Boolean.TRUE;
            }
            readInfo = new NewsAnnouncementRead();
            readInfo.setNewsId(announcementIdReq.getId());
            readInfo.setCreateUserId(userId);
            readInfo.setCreateUserSid(AppAuthContextHolder.getContext().getAuthoredUser().getSid());
            readInfo.setCreateUserName(AppAuthContextHolder.getContext().getAuthoredUser().getUserName());
            readInfo.setCreateTime(DateUtils.getNowTime(DateUtils.DATE_TIME_NORMAL_FORMATTER));
            readInfo.setTenantId(tenantId);
            readInfo.setDelFlag(0);
            // 三方公告
            if (ObjectUtils.isNotEmpty(announcementIdReq.getThirdNewsId())) {
                String verifyUserId = iamService.queryMappingEmpId(announcementIdReq.getThirdAppCode());
                readInfo.setThirdUserId(verifyUserId);
                readInfo.setThirdAppId(announcementIdReq.getThirdAppId());
                readInfo.setThirdAppCode(announcementIdReq.getThirdAppCode());
                readInfo.setThirdNewsId(announcementIdReq.getThirdNewsId());
            }
            newsAnnouncementReadMapper.insertNewsRead(readInfo);
            // 更新已读人员数量
            newsAnnouncementMapper.updateReadCount(announcementIdReq.getId());
            respFlag = Boolean.TRUE;
        } catch (Exception e) {
            log.error("insert news error.", e);
            respFlag = Boolean.FALSE;
        } finally {
            redisLock.unlock();
        }
        return respFlag;
    }

    @Override
    public Map<Long, Long> queryReadCount(List<Long> newsIdList) {
        if (CollectionUtils.isEmpty(newsIdList)) {
            return Maps.newHashMap();
        }
        LambdaQueryWrapper<NewsAnnouncementRead> queryWrapper = new LambdaQueryWrapper<>();
        queryWrapper.in(NewsAnnouncementRead::getNewsId,newsIdList);
        queryWrapper.select(NewsAnnouncementRead::getNewsId);
        List<NewsAnnouncementRead> readList = InterceptorIgnoreUtil.handler(() -> list(queryWrapper));
        if (CollectionUtils.isEmpty(readList)) {
            return Maps.newHashMap();
        }
        return readList.stream().collect(Collectors.groupingBy(NewsAnnouncementRead::getNewsId, Collectors.counting()));
    }

    /**
     * 查询当前用户是否已读指定id的公告
     *
     * @param announcementIdReq 请求参数
     * @return 返回
     */
    private NewsAnnouncementRead queryNewsAnnouncementReadInfo(AnnouncementIdReq announcementIdReq) {
        QueryWrapper<NewsAnnouncementRead> readCondition = new QueryWrapper<>();
        readCondition.eq("news_id", announcementIdReq.getId());
        readCondition.eq("create_user_id", AppAuthContextHolder.getContext().getAuthoredUser().getUserId());
        return newsAnnouncementReadMapper.selectOne(readCondition);
    }
}
