package com.digiwin.athena.semc.service.news.impl;

import cn.hutool.core.thread.ThreadUtil;
import cn.hutool.core.util.IdUtil;
import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.baomidou.mybatisplus.core.conditions.update.LambdaUpdateWrapper;
import com.baomidou.mybatisplus.core.conditions.update.UpdateWrapper;
import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.digiwin.athena.appcore.auth.AppAuthContextHolder;
import com.digiwin.athena.appcore.auth.domain.AuthoredUser;
import com.digiwin.athena.semc.common.Constants;
import com.digiwin.athena.semc.common.NewsTypeConstants;
import com.digiwin.athena.semc.common.PageInfoResp;
import com.digiwin.athena.semc.common.enums.*;
import com.digiwin.athena.semc.dto.news.*;
import com.digiwin.athena.semc.entity.applink.AppLinkDTO;
import com.digiwin.athena.semc.entity.news.*;
import com.digiwin.athena.semc.env.EnvProperties;
import com.digiwin.athena.semc.event.NewsAuthBizIdSetEvent;
import com.digiwin.athena.semc.event.NewsChangeEvent;
import com.digiwin.athena.semc.event.SendNoticeEvent;
import com.digiwin.athena.semc.event.dto.NewsChangeEventDTO;
import com.digiwin.athena.semc.event.dto.SendNoticeDTO;
import com.digiwin.athena.semc.mapper.news.*;
import com.digiwin.athena.semc.mq.dto.MQMessageDTO;
import com.digiwin.athena.semc.mq.sender.RabbitMessageSender;
import com.digiwin.athena.semc.proxy.aim.dto.MessageBatchUserDTO;
import com.digiwin.athena.semc.proxy.aim.dto.MessageDTO;
import com.digiwin.athena.semc.proxy.aim.service.AimService;
import com.digiwin.athena.semc.proxy.eoc.service.EocService;
import com.digiwin.athena.semc.proxy.eoc.service.model.UserDeptInfoDTO;
import com.digiwin.athena.semc.proxy.iam.service.IamService;
import com.digiwin.athena.semc.proxy.iam.service.model.OrgRoleUserDTO;
import com.digiwin.athena.semc.proxy.iam.service.model.RoleDTO;
import com.digiwin.athena.semc.proxy.iam.service.model.UserDTO;
import com.digiwin.athena.semc.proxy.iam.service.model.UserInfoDTO;
import com.digiwin.athena.semc.proxywrapper.IamUserService;
import com.digiwin.athena.semc.proxywrapper.dto.IamUserAuthInfo;
import com.digiwin.athena.semc.service.applink.AppLinkService;
import com.digiwin.athena.semc.service.cache.CommonConfigService;
import com.digiwin.athena.semc.service.news.NewsAnnouncementAuthService;
import com.digiwin.athena.semc.service.news.NewsAnnouncementService;
import com.digiwin.athena.semc.service.news.NewsAnnouncementTypeAuthService;
import com.digiwin.athena.semc.service.news.ThirdNewsAnnouncementService;
import com.digiwin.athena.semc.util.*;
import com.digiwin.athena.semc.vo.auth.AuthVO;
import com.digiwin.athena.semc.vo.file.FileInfoVO;
import com.digiwin.athena.semc.vo.news.NewsAnnouncementStatisticsVO;
import com.digiwin.athena.semc.vo.news.NewsAnnouncementVO;
import com.digiwin.athena.semc.vo.news.NewsTypeAuthVO;
import com.digiwin.athena.semc.vo.news.ValidateNewsVO;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.google.common.collect.Sets;
import io.github.linpeilie.Converter;
import io.vavr.Tuple2;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.collections.MapUtils;
import org.apache.commons.lang3.ObjectUtils;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.MDC;
import org.springframework.context.ApplicationEventPublisher;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import javax.annotation.Resource;
import java.time.LocalDateTime;
import java.time.format.DateTimeFormatter;
import java.util.*;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.Executor;
import java.util.function.Function;
import java.util.function.Supplier;
import java.util.stream.Collectors;


/**
 * 新闻公告表(TNewsAnnouncement)表服务实现类
 *
 * @author sungqz
 * @since 2024-05-13
 */
@Service
@Slf4j
public class NewsAnnouncementServiceImpl extends ServiceImpl<NewsAnnouncementMapper, NewsAnnouncement> implements NewsAnnouncementService {

    @Resource
    private NewsAnnouncementMapper newsAnnouncementMapper;

    @Resource
    private NewsAnnouncementTypeMapper newsAnnouncementTypeMapper;

    @Resource
    private NewsAnnouncementAuthMapper newsAnnouncementAuthMapper;

    @Resource
    private NewsAnnouncementAuthService newsAnnouncementAuthService;

    @Resource
    private NewsAnnouncementReadMapper newsAnnouncementReadMapper;

    @Resource
    private EnvProperties envProperties;

    @Resource
    private DmcFileUtil dmcFileUtil;

    @Resource
    private IamService iamService;

    @Resource
    private EocService eocService;

    @Resource
    private AimService aimService;

    @Resource
    private AppLinkService appLinkService;

    @Resource
    private ThirdNewsAnnouncementService thirdNewsAnnouncementService;

    @Resource
    private NewsAnnouncementTypeAuthService newsAnnouncementTypeAuthService;

    @Resource
    private Converter converter;

    @Resource
    private ApplicationEventPublisher eventPublisher;

    @Resource
    private IamUserService iamUserService;

    @Resource
    private NewsAnnouncementLikeMapper newsAnnouncementLikeMapper;

    @Resource
    private NewsAnnouncementFavoriteMapper newsAnnouncementFavoriteMapper;

    @Resource
    private RabbitMessageSender rabbitMessageSender;

    @Resource
    private CommonConfigService commonConfigService;

    @Resource(name = "semcAsyncExecutor")
    private Executor asyncTaskExecutor;

    /**
     * 公告置顶数量最大限制
     */
    public static final int MAX_TOPPED_COUNT = 3;

    /**
     * 分页查询公告列表
     *
     * @param queryNewsListPageReq 查询参数
     * @return 返回
     */
    @Override
    public PageInfoResp<NewsAnnouncementVO> queryNewsListByPage(QueryNewsListPageReq queryNewsListPageReq) {
        // 先判断公告是否过了截止日期
        QueryWrapper<NewsAnnouncement> condition = new QueryWrapper<>();
        condition.lt("news_end_time", DateUtils.getNowTime(null));
        condition.ne("news_status", Constants.NewsAnnouncementStatusEnum.EXPIRED.getFlag());

        // 更新失效的公告
        List<NewsAnnouncement> expiredList = newsAnnouncementMapper.selectList(condition);
        if (CollectionUtils.isNotEmpty(expiredList)) {
            expiredList.forEach(x -> {
                x.setNewsStatus(Constants.NewsAnnouncementStatusEnum.EXPIRED.getFlag());
                x.setToppedFlag(Constants.TOPPED_FLAG_NO);
                x.setToppedTime(null);
            });
            this.updateBatchById(expiredList);
        }

        condition.clear();
        // 公告标题
        if (StringUtils.isNotBlank(queryNewsListPageReq.getNewsTitle())) {
            condition.like("news_title", queryNewsListPageReq.getNewsTitle());
        }
        // 公告来源 0：一方公告 1：三方公告
        if (ObjectUtils.isNotEmpty(queryNewsListPageReq.getNewsSource())) {
            condition.eq("news_source", queryNewsListPageReq.getNewsSource());
        }
        // 公告来源系统
        if (StringUtils.isNotEmpty(queryNewsListPageReq.getAppSourceCode())) {
            // "雅典娜智驱平台"应用的公告全是一方公告
            if (VirtualApplicationEnum.ATHENA_AGILE_INTERACTION.getCode().equals(queryNewsListPageReq.getAppSourceCode())) {
                condition.eq("news_source", Constants.ThirdNewsAnnouncementEnum.INSIDER_INFORMATION.getType());
            } else {
                condition.eq("news_source", Constants.ThirdNewsAnnouncementEnum.OUTSIDER_INFORMATION.getType());
                condition.eq("third_app_code", queryNewsListPageReq.getAppSourceCode());
            }
        }
        // 公告类型id
        if (CollectionUtils.isNotEmpty(queryNewsListPageReq.getNewsTypeIdList())) {
            condition.in("news_type_id", queryNewsListPageReq.getNewsTypeIdList());
        } else {
            NewsTypeAuthVO newsTypeAuthVO = newsAnnouncementTypeAuthService.queryAuthRelNewsType(null);
            // 如果没有全部公告类型的权限，则只查询有权限的公告类型的公告
            if (Constants.ALL_AUTH_FLAG_NO.equals(newsTypeAuthVO.getAllNewsTypeFlag())) {
                List<Long> authorizedNewsTypeIdList = newsTypeAuthVO.getAuthorizedNewsTypeIdList();
                condition.in("news_type_id", authorizedNewsTypeIdList);
            }
        }

        // 状态
        if (ObjectUtils.isNotEmpty(queryNewsListPageReq.getNewsStatus())) {
            // "待生效"状态的公告为未到生效时间的公告，即生效时间大于当前时间的公告
            if (Constants.NewsAnnouncementStatusEnum.PENDING_EFFECTIVE.getFlag().equals(queryNewsListPageReq.getNewsStatus())) {
                condition.gt("news_effective_time", DateUtils.getNowTime(null));
            } else if (Constants.NewsAnnouncementStatusEnum.PUBLISHED.getFlag().equals(queryNewsListPageReq.getNewsStatus())) {
                condition.eq("news_status", queryNewsListPageReq.getNewsStatus());
                condition.le("news_effective_time", DateUtils.getNowTime(null));
                condition.ge("news_end_time", DateUtils.getNowTime(null));
            } else {
                condition.eq("news_status", queryNewsListPageReq.getNewsStatus());
            }
        }
        // 公告渠道
        if (ObjectUtils.isNotEmpty(queryNewsListPageReq.getNewsChannel())) {
            condition.and(x -> x.eq("news_channel", Constants.NewsChannelFlagEnum.ALL.getFlag())
                    .or().eq("news_channel", queryNewsListPageReq.getNewsChannel()));
            // 移动端渠道只查询一方公告
            if (Constants.NewsChannelFlagEnum.MOBILE.getFlag().equals(queryNewsListPageReq.getNewsChannel())) {
                condition.eq("news_source", Constants.ThirdNewsAnnouncementEnum.INSIDER_INFORMATION.getType());
            }
        }
        // 大于生效时间的开始时间
        if (StringUtils.isNotBlank(queryNewsListPageReq.getEffectiveTimeStart())) {
            condition.ge("news_effective_time", queryNewsListPageReq.getEffectiveTimeStart());
        }
        // 小于生效时间的结束时间
        if (StringUtils.isNotBlank(queryNewsListPageReq.getEffectiveTimeEnd())) {
            condition.le("news_effective_time", queryNewsListPageReq.getEffectiveTimeEnd());
        }
        // 大于截止时间的开始时间
        if (StringUtils.isNotBlank(queryNewsListPageReq.getEndTimeStart())) {
            condition.ge("news_end_time", queryNewsListPageReq.getEndTimeStart());
        }
        // 小于截止时间的结束时间
        if (StringUtils.isNotBlank(queryNewsListPageReq.getEndTimeEnd())) {
            condition.le("news_end_time", queryNewsListPageReq.getEndTimeEnd());
        }
        // 大于创建时间的开始时间
        if (StringUtils.isNotBlank(queryNewsListPageReq.getCreateTimeStart())) {
            condition.ge("create_time", queryNewsListPageReq.getCreateTimeStart());
        }
        // 小于创建时间的结束时间
        if (StringUtils.isNotBlank(queryNewsListPageReq.getCreateTimeEnd())) {
            condition.le("create_time", queryNewsListPageReq.getCreateTimeEnd());
        }
        // 置顶公告排在最前面
        condition.orderByDesc("topped_time").orderByDesc("create_time");
        condition.eq("tenant_id", AppAuthContextHolder.getContext().getAuthoredUser().getTenantId());

        // 分页查询列表
        condition.select("id", "news_title", "news_status", "news_effective_flag", "news_effective_time", "news_end_time", "news_type_id", "news_type_path", "summary", "news_channel", "default_image_icon", "cover_image_id", "topped_flag", "topped_time", "modified_flag", "file_id",
                "news_source", "third_news_id", "third_app_id", "third_app_code", "third_img_url", "third_user_ids", "third_news_html", "create_user_id", "create_time", "modify_user_id", "modify_time", "like_count");
        PageInfoResp<NewsAnnouncementVO> pageInfoResp = new PageInfoResp<>();
        Page<NewsAnnouncement> pageResult = newsAnnouncementMapper.selectPage(new Page<>(queryNewsListPageReq.getPageNum(), queryNewsListPageReq.getPageSize()), condition);
        if (pageResult == null || CollectionUtils.isEmpty(pageResult.getRecords())) {
            return pageInfoResp;
        }
        List<NewsAnnouncement> newsList = pageResult.getRecords();

        // 获取公告类型的名称
        Map<Long, NewsAnnouncementType> newsTypeMap = queryNewsTypeByNewsList(newsList);

        // 批量获取文件信息
        Map<String, Object> fileIdMap = queryFileInfoList(newsList);

        // 如果存在三方公告，则获取三方应用的名称
        Map<String, String> appCodeMap = Maps.newHashMap();
        boolean flag = newsList.stream().anyMatch(x -> Constants.ThirdNewsAnnouncementEnum.OUTSIDER_INFORMATION.getType().equals(x.getNewsSource()));
        if (flag) {
            List<AppLinkDTO> appLinkList = appLinkService.queryManageListSync(AppAuthContextHolder.getContext().getAuthoredUser());
            appCodeMap = appLinkList.stream().collect(Collectors.toMap(AppLinkDTO::getCode, AppLinkDTO::getName, (a, b) -> a));
        }

        List<NewsAnnouncementVO> newsAnnouncementVOList = Lists.newArrayList();
        for (NewsAnnouncement x : newsList) {
            NewsAnnouncementVO newsAnnouncementVO = converter.convert(x, NewsAnnouncementVO.class);
            // 公告类型的名称
            if (MapUtils.isNotEmpty(newsTypeMap) && ObjectUtils.isNotEmpty(newsTypeMap.get(x.getNewsTypeId()))) {
                NewsAnnouncementType newsAnnouncementType = newsTypeMap.get(x.getNewsTypeId());
                newsAnnouncementVO.setParentNewsTypeId(newsAnnouncementType.getParentId());
                newsAnnouncementVO.setNewsTypeName(getNewsTypeNameByNewPath(newsTypeMap, x));
            } else {
                buildOtherNewsType(newsAnnouncementVO);
            }
            // 构建附件
            if (StringUtils.isNotBlank(x.getFileId()) && MapUtils.isNotEmpty(fileIdMap)) {
                List<FileInfoVO> fileList = buildDetailFileInfo(x.getFileId().split(","), fileIdMap);
                newsAnnouncementVO.setFileList(fileList);
            }
            // 封面图
            if (StringUtils.isNotBlank(x.getCoverImageId()) && MapUtils.isNotEmpty(fileIdMap)) {
                List<FileInfoVO> fileList = buildDetailFileInfo(x.getCoverImageId().split(","), fileIdMap);
                newsAnnouncementVO.setCoverImageInfo(fileList.get(0));
            }
            // 三方应用名称
            if (Constants.ThirdNewsAnnouncementEnum.OUTSIDER_INFORMATION.getType().equals(x.getNewsSource())) {
                newsAnnouncementVO.setThirdAppName(appCodeMap.get(x.getThirdAppCode()));
            } else {
                newsAnnouncementVO.setThirdAppCode(VirtualApplicationEnum.ATHENA_AGILE_INTERACTION.getCode());
                newsAnnouncementVO.setThirdAppName(VirtualApplicationEnum.ATHENA_AGILE_INTERACTION.getName());
            }
            // 如果公告还未到生效时间，则显示"待生效"状态
            if (NewsAnnouncementPublishTypeEnum.NONE.getCode().equals(x.getPublishType())) {
                newsAnnouncementVO.setPublishId("");
                newsAnnouncementVO.setPublishName("");
            }
            newsAnnouncementVOList.add(newsAnnouncementVO);
        }

        // 构建公告的已读、未读人员列表
        buildNewsReadUserList(newsAnnouncementVOList, false, false);
        // 返回对象
        pageInfoResp.setTotalPages(Integer.parseInt(String.valueOf(pageResult.getPages())));
        pageInfoResp.setPageNo(queryNewsListPageReq.getPageNum());
        pageInfoResp.setPageSize(queryNewsListPageReq.getPageSize());
        pageInfoResp.setTotalRecords(Integer.parseInt(String.valueOf(pageResult.getTotal())));
        pageInfoResp.setList(newsAnnouncementVOList);
        return pageInfoResp;
    }

    /**
     * "其他"公告类型
     *
     * @param newsAnnouncementVO 公告
     */
    private void buildOtherNewsType(NewsAnnouncementVO newsAnnouncementVO) {
        newsAnnouncementVO.setNewsTypeId(4L);
        newsAnnouncementVO.setParentNewsTypeId(0L);
        newsAnnouncementVO.setNewsTypeName(NewsTypeConstants.OTHER_NEWS_TYPE_NAME);
    }

    /**
     * 构建公告的已读、未读人员列表
     *
     * @param newsAnnouncementVOList 公告返回列表
     * @param setReadUserListValue   是否设置已读人员列表集合数据
     * @param setUnReadUserListValue 是否设置未读人员列表集合数据
     */
    private void buildNewsReadUserList(List<NewsAnnouncementVO> newsAnnouncementVOList, boolean setReadUserListValue, boolean setUnReadUserListValue) {
        // 查询所有已读公告的人员列表
        List<Long> newsIdList = newsAnnouncementVOList.stream().map(NewsAnnouncementVO::getId).collect(Collectors.toList());
        Map<Long, List<NewsAnnouncementRead>> readNewsIdMap = Maps.newHashMap();
        if (CollectionUtils.isNotEmpty(newsIdList)) {
            List<NewsAnnouncementRead> readList = newsAnnouncementReadMapper.selectNewsAnnouncementReadList(newsIdList);
            readNewsIdMap = readList.stream().collect(Collectors.groupingBy(NewsAnnouncementRead::getNewsId));
        }

        List<RoleDTO> allUserList = iamService.queryEnabledUserList(); // 所有用户
        Map<String, String> userNameMap = allUserList.stream().collect(Collectors.toMap(RoleDTO::getId, RoleDTO::getName, (a, b) -> a));
        List<OrgRoleUserDTO> orgUserList = Lists.newArrayList(); // 所有组织下的用户
        List<OrgRoleUserDTO> roleUserList = Lists.newArrayList(); // 所有角色下的用户
        List<UserDeptInfoDTO> deptUserList = Lists.newArrayList(); // 所有部门下的用户
        Map<Long, List<NewsAnnouncementAut>> newsIdAuthMap = Maps.newHashMap(); // 每条公告的权限对象列表
        // 一方公告，查询公告权限表中关联的人员
        List<Long> onePartyIdList = newsAnnouncementVOList.stream().filter(x -> Constants.ThirdNewsAnnouncementEnum.INSIDER_INFORMATION.getType().equals(x.getNewsSource())).map(NewsAnnouncementVO::getId).collect(Collectors.toList());
        if (CollectionUtils.isNotEmpty(onePartyIdList)) {
            List<NewsAnnouncementAut> autList = newsAnnouncementAuthMapper.selectByNewsId(onePartyIdList);
            if (CollectionUtils.isNotEmpty(autList)) {
                // 所有组织下的用户
                List<Long> orgSidList = autList.stream().filter(x -> BizAuthTypeEnum.ORG.getValue().equals(x.getAuthType())).map(NewsAnnouncementAut::getAuthId).collect(Collectors.toList());
                orgUserList = iamService.queryUserByAllOrgSidList(orgSidList);

                // 所有角色下的用户
                List<Long> roleSidList = autList.stream().filter(x -> BizAuthTypeEnum.ROLE.getValue().equals(x.getAuthType())).map(NewsAnnouncementAut::getAuthId).collect(Collectors.toList());
                roleUserList = iamService.queryUserByAllRoleSidList(roleSidList);

                // 所有部门下的用户
                List<Long> deptSidList = autList.stream().filter(x -> BizAuthTypeEnum.DEPARTMENT.getValue().equals(x.getAuthType())).map(NewsAnnouncementAut::getAuthId).collect(Collectors.toList());
                deptUserList = eocService.queryUserByDeptSidList(deptSidList);

                // 权限对象以公告id分组
                newsIdAuthMap = autList.stream().collect(Collectors.groupingBy(NewsAnnouncementAut::getNewsId));

                // 权限表外部用户id赋值事件，为了刷历史数据
                // TODO: 2025/4/10  历史数据刷完删除该事件
                List<NewsAnnouncementAut> authList = autList.stream().filter(x -> BizAuthTypeEnum.EXTERNAL_USER.getValue().equals(x.getAuthType()) && StringUtils.isBlank(x.getAuthBizId())).collect(Collectors.toList());
                if (CollectionUtils.isNotEmpty(authList)) {
                    NewsAuthBizIdSetEvent newsAuthBizIdSetEvent = new NewsAuthBizIdSetEvent(this, authList);
                    eventPublisher.publishEvent(newsAuthBizIdSetEvent);
                }
            }
        }

        for (NewsAnnouncementVO x : newsAnnouncementVOList) {
            // 返回创建人和更新人名称
            x.setCreateUserName(userNameMap.get(x.getCreateUserId()));
            x.setModifyUserName(userNameMap.get(x.getModifyUserId()));

            List<UserDTO> readUserList = Lists.newArrayList();
            List<UserDTO> unReadUserList = Lists.newArrayList();
            List<UserDTO> shouldReadUserList = Lists.newArrayList();
            List<NewsAnnouncementRead> curNewsReadList = readNewsIdMap.get(x.getId());
            // 构建一方公告已读、未读、应读人员
            List<NewsAnnouncementAut> newsAuthList = newsIdAuthMap.get(x.getId());
            if (Constants.ThirdNewsAnnouncementEnum.INSIDER_INFORMATION.getType().equals(x.getNewsSource()) && CollectionUtils.isNotEmpty(newsAuthList)) {
                // 全员
                if (Constants.NEWS_ALL_AUTH_FLAG_YES.equals(newsAuthList.get(0).getAllAuthFlag())) {
                    shouldReadUserList = allUserList.stream().map(y -> {
                        UserDTO userDTO = new UserDTO();
                        userDTO.setSid(y.getSid());
                        userDTO.setId(y.getId());
                        userDTO.setName(y.getName());
                        return userDTO;
                    }).collect(Collectors.toList());

                    // 构建已读、未读人员列表
                    unReadUserList = filterReadUserList(readUserList, shouldReadUserList, curNewsReadList);
                } else {
                    // 组织
                    List<Long> orgSidList = newsAuthList.stream().filter(y -> BizAuthTypeEnum.ORG.getValue().equals(y.getAuthType())).map(NewsAnnouncementAut::getAuthId).collect(Collectors.toList());
                    List<OrgRoleUserDTO> orgList = orgUserList.stream().filter(y -> orgSidList.contains(y.getOrgSid())).collect(Collectors.toList());
                    // 角色
                    List<Long> roleSidList = newsAuthList.stream().filter(y -> BizAuthTypeEnum.ROLE.getValue().equals(y.getAuthType())).map(NewsAnnouncementAut::getAuthId).collect(Collectors.toList());
                    List<OrgRoleUserDTO> roleList = roleUserList.stream().filter(y -> roleSidList.contains(y.getRoleSid())).collect(Collectors.toList());
                    // 部门
                    List<Long> deptSidList = newsAuthList.stream().filter(y -> BizAuthTypeEnum.DEPARTMENT.getValue().equals(y.getAuthType())).map(NewsAnnouncementAut::getAuthId).collect(Collectors.toList());
                    List<UserDeptInfoDTO> deptList = deptUserList.stream().filter(y -> deptSidList.contains(y.getDeptSid())).collect(Collectors.toList());
                    // 用户
                    List<NewsAnnouncementAut> userList = newsAuthList.stream().filter(y -> BizAuthTypeEnum.USER.getValue().equals(y.getAuthType())
                            || BizAuthTypeEnum.EXTERNAL_USER.getValue().equals(y.getAuthType())).collect(Collectors.toList());

                    // 当前公告关联的所有用户
                    for (OrgRoleUserDTO orgRoleUserDTO : orgList) {
                        shouldReadUserList.addAll(orgRoleUserDTO.getUsers());
                    }
                    for (OrgRoleUserDTO orgRoleUserDTO : roleList) {
                        shouldReadUserList.addAll(orgRoleUserDTO.getUsers());
                    }
                    for (UserDeptInfoDTO userDeptInfoDTO : deptList) {
                        List<UserDTO> userDTOList = Lists.newArrayList();
                        userDeptInfoDTO.getEmps().forEach(empInfo -> {
                            if (ObjectUtils.isNotEmpty(empInfo.getEmpUserSid())) {
                                UserDTO userDTO = new UserDTO();
                                userDTO.setSid(empInfo.getEmpUserSid());
                                userDTO.setId(empInfo.getEmpUserId());
                                userDTO.setName(empInfo.getEmpUserName());
                                userDTOList.add(userDTO);
                            }
                        });
                        shouldReadUserList.addAll(userDTOList);
                    }
                    for (NewsAnnouncementAut newsAnnouncementAut : userList) {
                        UserDTO userDTO = new UserDTO();
                        userDTO.setSid(newsAnnouncementAut.getAuthId());
                        userDTO.setName(newsAnnouncementAut.getAuthName());
                        shouldReadUserList.add(userDTO);
                    }
                    shouldReadUserList = shouldReadUserList.stream().collect(Collectors.collectingAndThen(Collectors.toCollection(() ->
                            new TreeSet<>(Comparator.comparing(UserDTO::getSid))), ArrayList::new));
                    // 构建已读、未读人员列表
                    unReadUserList = filterReadUserList(readUserList, shouldReadUserList, curNewsReadList);
                }
            }
            // 构建三方公告已读、未读人员
            if (Constants.ThirdNewsAnnouncementEnum.OUTSIDER_INFORMATION.getType().equals(x.getNewsSource())) {
                // 应读人员列表
                shouldReadUserList = Arrays.stream(x.getThirdUserIds().split(",")).map(y -> {
                    UserDTO userDTO = new UserDTO();
                    userDTO.setId(y);
                    userDTO.setName(y);
                    return userDTO;
                }).collect(Collectors.toList());
                // 已读人员列表
                List<String> readSidList = Lists.newArrayList();
                if (CollectionUtils.isNotEmpty(curNewsReadList)) {
                    for (NewsAnnouncementRead announcementRead : curNewsReadList) {
                        UserDTO userDTO = new UserDTO();
                        userDTO.setId(announcementRead.getThirdUserId());
                        userDTO.setName(announcementRead.getThirdUserId());
                        readUserList.add(userDTO);
                        readSidList.add(announcementRead.getThirdUserId());
                    }
                }
                // 过滤出未读人员列表
                unReadUserList = shouldReadUserList.stream().filter(y -> !readSidList.contains(y.getId())).collect(Collectors.toList());
            }
            x.setReadCount(readUserList.size());
            x.setUnReadCount(unReadUserList.size());
            x.setShouldReadCount(shouldReadUserList.size());
            //全部应阅人员暂时没有场景展示
            //x.setShouldReadList(shouldReadUserList);
            if (setReadUserListValue) {
                x.setReadList(readUserList);
            }
            if (setUnReadUserListValue) {
                x.setUnReadList(unReadUserList);
            }
        }
    }

    /**
     * 构建已读、未读人员列表
     *
     * @param readUserList       已读人员列表
     * @param shouldReadUserList 未读人员列表
     * @param readList           公告已读表中的已读人员
     * @return 返回
     */
    private List<UserDTO> filterReadUserList(List<UserDTO> readUserList, List<UserDTO> shouldReadUserList, List<NewsAnnouncementRead> readList) {
        if (CollectionUtils.isEmpty(readList)) {
            return shouldReadUserList;
        }
        List<Long> readSidList = Lists.newArrayList();
        for (NewsAnnouncementRead announcementRead : readList) {
            UserDTO userDTO = new UserDTO();
            userDTO.setSid(announcementRead.getCreateUserSid());
            userDTO.setId(announcementRead.getCreateUserId());
            userDTO.setName(announcementRead.getCreateUserName());
            readUserList.add(userDTO);
            if (ObjectUtils.isNotEmpty(announcementRead.getCreateUserSid())) {
                readSidList.add(announcementRead.getCreateUserSid());
            }
        }
        // 过滤出未读人员列表
        shouldReadUserList = shouldReadUserList.stream().filter(y -> !readSidList.contains(y.getSid())).collect(Collectors.toList());
        return shouldReadUserList;
    }

    /**
     * 批量获取文件信息
     *
     * @param newsList 公告列表
     * @return 返回
     */
    private Map<String, Object> queryFileInfoList(List<NewsAnnouncement> newsList) {
        List<String> fileIdList = Lists.newArrayList();
        newsList.forEach(x -> {
            // 附件id
            if (StringUtils.isNotBlank(x.getFileId())) {
                fileIdList.addAll(Arrays.asList(x.getFileId().split(",")));
            }
            // 封面图文件id
            if (StringUtils.isNotBlank(x.getCoverImageId())) {
                fileIdList.add(x.getCoverImageId().trim());
            }
        });
        return dmcFileUtil.queryFileInfoList(fileIdList);
    }

    /**
     * 构建文件信息
     *
     * @param fileIdArray 文件id
     * @param fileIdMap   文件信息
     * @return 返回
     */
    @Override
    public List<FileInfoVO> buildDetailFileInfo(String[] fileIdArray, Map<String, Object> fileIdMap) {
        List<FileInfoVO> fileInfoVOList = Lists.newArrayList();
        for (String fileId : fileIdArray) {
            FileInfoVO fileInfoVO = new FileInfoVO();
            // 文件id
            fileInfoVO.setId(fileId.trim());
            fileInfoVOList.add(fileInfoVO);
            if (Objects.isNull(fileIdMap.get(fileId.trim()))) {
                continue;
            }
            LinkedHashMap file = (LinkedHashMap) fileIdMap.get(fileId.trim());
            // 文件名称
            fileInfoVO.setName(String.valueOf(file.get("fileName")));
            // 文件后缀
            fileInfoVO.setType(String.valueOf(file.get("extension")));
            // 文件链接
            fileInfoVO.setUrl(envProperties.getDmcUri() + Constants.DMC_FILE_PREVIEW_PATH + fileId.trim());
            // 移动端文件链接
            fileInfoVO.setMobileOnlineUrl(envProperties.getDmcUri() + Constants.DMC_MOBILE_FILE_PREVIEW_PATH + fileId.trim());
            // 文件大小
            fileInfoVO.setSize(String.valueOf(file.get("size")));
            // 创建时间
            fileInfoVO.setCreateDate(String.valueOf(file.get("createDate")));
        }
        return fileInfoVOList;
    }

    /**
     * 更新公告发布状态
     *
     * @param updateNewsStatusReq  入参
     * @param newsAnnouncementList 公告列表
     */
    @Override
    public void updatePublishStatus(UpdateNewsStatusReq updateNewsStatusReq, List<NewsAnnouncement> newsAnnouncementList) {
        UpdateWrapper<NewsAnnouncement> updateWrapper = new UpdateWrapper<>();
        updateWrapper.set("news_status", updateNewsStatusReq.getNewsStatus())
                .set("modify_user_id", AppAuthContextHolder.getContext().getAuthoredUser().getUserId())
                .set("modify_time", LocalDateTime.now());
        // 当状态变成已发布，则更新发布时间
        if (Constants.NewsAnnouncementStatusEnum.PUBLISHED.getFlag().equals(updateNewsStatusReq.getNewsStatus())) {
            updateWrapper.set("news_effective_time", LocalDateTime.now());
        }
        updateWrapper.in("id", updateNewsStatusReq.getIdList());
        this.update(updateWrapper);

        // 当状态变成已发布，判断是否发送工作提醒
        if (Constants.NewsAnnouncementStatusEnum.PUBLISHED.getFlag().equals(updateNewsStatusReq.getNewsStatus())) {
            List<NewsAnnouncement> needSendNoticeList = newsAnnouncementList.stream().filter(x -> Constants.NUMBER_ONE.equals(x.getNoticeFlag())
                    && Constants.NUMBER_ZERO.equals(x.getAlreadyNoticeFlag())).collect(Collectors.toList());
            if (CollectionUtils.isEmpty(needSendNoticeList)) {
                log.info("no news need send notice");
                return;
            }

            List<Long> newsIdList = needSendNoticeList.stream().map(NewsAnnouncement::getId).collect(Collectors.toList());
            this.update(new LambdaUpdateWrapper<NewsAnnouncement>()
                    .in(NewsAnnouncement::getId, newsIdList)
                    .set(NewsAnnouncement::getAlreadyNoticeFlag, Constants.NUMBER_ONE));

            // 查询公告关联的权限，然后异步发送工作提醒
            List<NewsAnnouncementAut> autList = newsAnnouncementAuthMapper.selectByNewsId(newsIdList);
            if (CollectionUtils.isEmpty(autList)) {
                log.info("all news auth list is empty. newsIdList:{}", newsIdList);
                return;
            }

            // 以公告分组
            Map<Long, List<NewsAnnouncementAut>> newsIdMap = autList.stream().collect(Collectors.groupingBy(NewsAnnouncementAut::getNewsId));
            for (NewsAnnouncement newsAnnouncement : needSendNoticeList) {
                List<NewsAnnouncementAut> newsAuthList = newsIdMap.get(newsAnnouncement.getId());
                if (CollectionUtils.isEmpty(newsAuthList)) {
                    log.info("this news auth list is empty. newsId:{}", newsAnnouncement.getId());
                    continue;
                }

                boolean flag = newsAuthList.stream().anyMatch(x -> Constants.NEWS_ALL_AUTH_FLAG_YES.equals(x.getAllAuthFlag()));
                Integer allAuthFlag = flag ? Constants.NEWS_ALL_AUTH_FLAG_YES : Constants.NEWS_ALL_AUTH_FLAG_NO;
                List<Long> orgSidList = newsAuthList.stream().filter(x -> BizAuthTypeEnum.ORG.getValue().equals(x.getAuthType())).map(NewsAnnouncementAut::getAuthId).collect(Collectors.toList());
                List<Long> roleSidList = newsAuthList.stream().filter(x -> BizAuthTypeEnum.ROLE.getValue().equals(x.getAuthType())).map(NewsAnnouncementAut::getAuthId).collect(Collectors.toList());
                List<Long> deptSidList = newsAuthList.stream().filter(x -> BizAuthTypeEnum.DEPARTMENT.getValue().equals(x.getAuthType())).map(NewsAnnouncementAut::getAuthId).collect(Collectors.toList());
                Map<Long, String> userSidMap = newsAuthList.stream().filter(x -> BizAuthTypeEnum.USER.getValue().equals(x.getAuthType())).collect(Collectors.toMap(NewsAnnouncementAut::getAuthId, NewsAnnouncementAut::getAuthBizId));
                Map<Long, NewsAnnouncementAut> externalSidMap = newsAuthList.stream().filter(x -> BizAuthTypeEnum.EXTERNAL_USER.getValue().equals(x.getAuthType())).collect(Collectors.toMap(NewsAnnouncementAut::getAuthId, Function.identity()));
                // 发送工作提醒事件
                sendNoticeEvent(newsAnnouncement.getId(), newsAnnouncement.getNewsTitle(), allAuthFlag, orgSidList, roleSidList, deptSidList, userSidMap, externalSidMap, null);
            }
        }
    }

    /**
     * 更新公告置顶状态
     *
     * @param id         主键id
     * @param toppedFlag 是否置顶 0:否 1:是
     */
    @Override
    @Transactional
    public void updateToppedStatus(Long id, Integer toppedFlag) {
        // 置顶
        if (toppedFlag.equals(Constants.TOPPED_FLAG_YES)) {
            // 查询当前租户的置顶消息是否达到三条，达到三条则将最早的一条取消置顶
            QueryWrapper<NewsAnnouncement> condition = new QueryWrapper<>();
            condition.eq("topped_flag", Constants.TOPPED_FLAG_YES);
            condition.orderByAsc("topped_time");
            List<NewsAnnouncement> toppedList = newsAnnouncementMapper.selectList(condition);
            if (toppedList.size() >= MAX_TOPPED_COUNT) {
                NewsAnnouncement newsAnnouncement = toppedList.get(0);
                newsAnnouncement.setToppedFlag(Constants.TOPPED_FLAG_NO);
                newsAnnouncement.setToppedTime(null);
                newsAnnouncementMapper.updateById(newsAnnouncement);
            }

            // 置顶当前公告
            UpdateWrapper<NewsAnnouncement> updateWrapper = new UpdateWrapper<>();
            updateWrapper.set("topped_flag", toppedFlag)
                    .set("topped_time", LocalDateTime.now())
                    .set("modify_user_id", AppAuthContextHolder.getContext().getAuthoredUser().getUserId())
                    .set("modify_time", LocalDateTime.now());
            updateWrapper.eq("id", id);
            this.update(updateWrapper);
        } else { // 取消置顶
            UpdateWrapper<NewsAnnouncement> updateWrapper = new UpdateWrapper<>();
            updateWrapper.set("topped_flag", toppedFlag)
                    .set("topped_time", null)
                    .set("modify_time", LocalDateTime.now());
            updateWrapper.eq("id", id);
            this.update(updateWrapper);
        }
    }

    /**
     * 删除公告
     *
     * @param idList 公告id列表
     */
    @Override
    @Transactional
    public void delNews(List<Long> idList) {
        // 删除公告
        newsAnnouncementMapper.deleteBatchIds(idList);
        // 删除公告权限
        newsAnnouncementAuthMapper.deleteByNewsIdList(idList);
        // 删除已读公告
        newsAnnouncementReadMapper.deleteByNewsIdList(idList);
        //删除点赞公告
        LambdaQueryWrapper<NewsAnnouncementLike> queryWrapper = new LambdaQueryWrapper<>();
        queryWrapper.in(NewsAnnouncementLike::getNewsId, idList);
        InterceptorIgnoreUtil.handler(() -> {
            newsAnnouncementLikeMapper.delete(queryWrapper);
        });
        //注意不能删除用户的收藏，需要用户自己删除

    }

    /**
     * 通过ID查询单条数据
     *
     * @param id 主键
     * @return 实例对象
     */
    @Override
    public NewsAnnouncementVO queryManagerNewsDetail(Long id) {
        NewsAnnouncement newsAnnouncement = newsAnnouncementMapper.selectById(id);
        if (ObjectUtils.isEmpty(newsAnnouncement)) {
            return null;
        }
        NewsAnnouncementVO newsAnnouncementVO = converter.convert(newsAnnouncement, NewsAnnouncementVO.class);
        // 公告类型名称
        Map<Long, NewsAnnouncementType> newsTypeMap = queryNewsTypeByNewsList(Lists.newArrayList(newsAnnouncement));
        NewsAnnouncementType newsAnnouncementType = newsTypeMap.get(newsAnnouncementVO.getNewsTypeId());
        if (ObjectUtils.isEmpty(newsAnnouncementType)) {
            buildOtherNewsType(newsAnnouncementVO);
        } else {
            newsAnnouncementVO.setParentNewsTypeId(newsAnnouncementType.getParentId());
            newsAnnouncementVO.setNewsTypeName(getNewsTypeNameByNewPath(newsTypeMap, newsAnnouncement));
        }

        // 查询该条公告关联的权限
        List<NewsAnnouncementAut> announcementAutList = newsAnnouncementAuthMapper.selectByNewsId(Collections.singletonList(id));
        if (CollectionUtils.isEmpty(announcementAutList)) {
            return newsAnnouncementVO;
        }
        // 全员可见
        if (Constants.NEWS_ALL_AUTH_FLAG_YES.equals(announcementAutList.get(0).getAllAuthFlag())) {
            newsAnnouncementVO.setAllAuthFlag(Constants.NEWS_ALL_AUTH_FLAG_YES);
        } else { // 部分人员可见
            newsAnnouncementVO.setAllAuthFlag(Constants.NEWS_ALL_AUTH_FLAG_NO);
            Map<Integer, List<NewsAnnouncementAut>> authTypeMap = announcementAutList.stream().collect(Collectors.groupingBy(NewsAnnouncementAut::getAuthType));
            // 根据权限分类
            for (Map.Entry<Integer, List<NewsAnnouncementAut>> entry : authTypeMap.entrySet()) {
                List<AuthVO> authDTOList = entry.getValue().stream().map(NewsAnnouncementAut -> converter.convert(NewsAnnouncementAut, AuthVO.class)).collect(Collectors.toList());
                if (BizAuthTypeEnum.ORG.getValue().equals(entry.getKey())) { // 设置组织权限
                    newsAnnouncementVO.setOrgList(authDTOList);
                } else if (BizAuthTypeEnum.ROLE.getValue().equals(entry.getKey())) { // 设置角色权限
                    newsAnnouncementVO.setRoleList(authDTOList);
                } else if (BizAuthTypeEnum.USER.getValue().equals(entry.getKey())) { // 设置用户权限
                    newsAnnouncementVO.setUserList(authDTOList);
                } else if (BizAuthTypeEnum.EXTERNAL_USER.getValue().equals(entry.getKey())) { // 设置外部用户权限
                    newsAnnouncementVO.setExternalUserList(authDTOList);
                } else if (BizAuthTypeEnum.DEPARTMENT.getValue().equals(entry.getKey())) { // 设置部门权限
                    newsAnnouncementVO.setDeptList(authDTOList);
                }
            }
        }
        if (NewsAnnouncementPublishTypeEnum.NONE.getCode().equals(newsAnnouncementVO.getPublishType())) {
            newsAnnouncementVO.setPublishName("");
            newsAnnouncementVO.setPublishId("");
        }
        return newsAnnouncementVO;
    }

    /**
     * 保存公告
     *
     * @param saveAnnouncementReq 入参
     * @return 返回
     */
    @Override
    @Transactional
    public void saveNews(SaveAnnouncementReq saveAnnouncementReq) {
        // 变更前的公告类型
        Long oldNewsTypeId = null;
        DataChangeTypeEnum dataChangeTypeEnum;
        boolean noticeFlag = Boolean.FALSE;
        // 新增
        if (ObjectUtils.isEmpty(saveAnnouncementReq.getId())) {
            dataChangeTypeEnum = DataChangeTypeEnum.ADD;
            NewsAnnouncement newsAnnouncement = new NewsAnnouncement();
            buildNewAnnouncement(saveAnnouncementReq, newsAnnouncement);
            // 是否发送工作提醒
            if (Constants.VALID_STATUS_ENABLE.equals(saveAnnouncementReq.getNewsStatus()) && Constants.NUMBER_ONE.equals(saveAnnouncementReq.getNoticeFlag())) {
                noticeFlag = true;
            }
            newsAnnouncementMapper.insert(newsAnnouncement);
            saveAnnouncementReq.setId(newsAnnouncement.getId());
        } else {
            dataChangeTypeEnum = DataChangeTypeEnum.EDIT;
            NewsAnnouncement newsAnnouncement = newsAnnouncementMapper.selectById(saveAnnouncementReq.getId());
            oldNewsTypeId = newsAnnouncement.getNewsTypeId();
            // 标题、类型、公告渠道、生效时间、截止时间
            if (!newsAnnouncement.getNewsTitle().equals(saveAnnouncementReq.getNewsTitle())
                    || !newsAnnouncement.getNewsTypeId().equals(saveAnnouncementReq.getNewsTypeId())
                    || !newsAnnouncement.getNewsChannel().equals(saveAnnouncementReq.getNewsChannel())
                    || !newsAnnouncement.getNewsEndTime().equals(saveAnnouncementReq.getNewsEndTime())) {
                newsAnnouncement.setModifiedFlag(Constants.MODIFIED_FLAG_YES);
            }
            // 附件有变化
            if (CollectionUtils.isEmpty(saveAnnouncementReq.getFileIdList())) {
                if (StringUtils.isNotBlank(newsAnnouncement.getFileId())) {
                    newsAnnouncement.setModifiedFlag(Constants.MODIFIED_FLAG_YES);
                }
            } else {
                if (StringUtils.isBlank(newsAnnouncement.getFileId())
                        || !(String.join(",", saveAnnouncementReq.getFileIdList()).equals(newsAnnouncement.getFileId()))) {
                    newsAnnouncement.setModifiedFlag(Constants.MODIFIED_FLAG_YES);
                }
            }
            // 正文
            if (!newsAnnouncement.getNewsContent().contentEquals(saveAnnouncementReq.getNewsContent())) {
                newsAnnouncement.setModifiedFlag(Constants.MODIFIED_FLAG_YES);
            }

            // 如果公告已经过了截止时间，且编辑了截止时间，则清空该公告的已读信息
            if (newsAnnouncement.getNewsEndTime().compareTo(DateUtils.getNowTime(null)) < 0
                    && !newsAnnouncement.getNewsEndTime().equals(saveAnnouncementReq.getNewsEndTime())) {
                newsAnnouncementReadMapper.deleteByNewsIdList(Collections.singletonList(saveAnnouncementReq.getId()));
            }

            buildNewAnnouncement(saveAnnouncementReq, newsAnnouncement);
            // 如果需要发送工作提醒且没有发过，则发送
            if (Constants.NUMBER_ONE.equals(saveAnnouncementReq.getNoticeFlag()) && Constants.NUMBER_ZERO.equals(newsAnnouncement.getAlreadyNoticeFlag())) {
                noticeFlag = true;
            }
            newsAnnouncementMapper.updateById(newsAnnouncement);
        }

        // 保存公告权限
        List<UserDTO> externalUserList = saveNewsAuthInner(saveAnnouncementReq);

        // 事务提交后，发送事件
        Long finalOldNewsTypeId = oldNewsTypeId;
        boolean flag = noticeFlag;
        TransactionUtils.executeAfterCommit(() -> {
            // 异步更新公告类型关联的公告总数
            NewsChangeEventDTO eventDTO = new NewsChangeEventDTO();
            eventDTO.setDataChangeType(dataChangeTypeEnum);
            eventDTO.setOldNewsTypeIdList(Collections.singletonList(finalOldNewsTypeId));
            eventDTO.setCurrentNewsTypeId(saveAnnouncementReq.getNewsTypeId());
            NewsChangeEvent newsChangeEvent = new NewsChangeEvent(this, eventDTO);
            eventPublisher.publishEvent(newsChangeEvent);

            // 异步发送工作提醒
            if (flag) {
                List<Long> orgSidList = Lists.newArrayList();
                List<Long> roleSidList = Lists.newArrayList();
                List<Long> deptSidList = Lists.newArrayList();
                Map<Long, String> userSidMap = Maps.newHashMap();
                if (CollectionUtils.isNotEmpty(saveAnnouncementReq.getOrgList())) {
                    orgSidList = saveAnnouncementReq.getOrgList().stream().map(AuthVO::getAuthId).collect(Collectors.toList());
                }
                if (CollectionUtils.isNotEmpty(saveAnnouncementReq.getRoleList())) {
                    roleSidList = saveAnnouncementReq.getRoleList().stream().map(AuthVO::getAuthId).collect(Collectors.toList());
                }
                if (CollectionUtils.isNotEmpty(saveAnnouncementReq.getDeptList())) {
                    deptSidList = saveAnnouncementReq.getDeptList().stream().map(AuthVO::getAuthId).collect(Collectors.toList());
                }
                if (CollectionUtils.isNotEmpty(saveAnnouncementReq.getUserList())) {
                    userSidMap = saveAnnouncementReq.getUserList().stream().collect(Collectors.toMap(AuthVO::getAuthId, AuthVO::getAuthBizId));
                }
                // 发送工作提醒事件
                sendNoticeEvent(saveAnnouncementReq.getId(), saveAnnouncementReq.getNewsTitle(), saveAnnouncementReq.getAllAuthFlag(), orgSidList, roleSidList, deptSidList, userSidMap, null, externalUserList);
            }
        });
    }


    /**
     * 发送工作提醒事件
     *
     * @param id             公告id
     * @param newsTitle      公告标题
     * @param allAuthFlag    是否全员可见 0:否 1:是
     * @param orgSidList     组织sid
     * @param roleSidList    角色sid
     * @param deptSidList    部门sid
     * @param userSidMap     用户sid
     * @param externalSidMap 外部用户sid
     */
    private void sendNoticeEvent(Long id, String newsTitle, Integer allAuthFlag, List<Long> orgSidList,
                                 List<Long> roleSidList, List<Long> deptSidList, Map<Long, String> userSidMap, Map<Long, NewsAnnouncementAut> externalSidMap, List<UserDTO> externalUserList) {
        SendNoticeDTO sendNoticeDTO = new SendNoticeDTO();
        sendNoticeDTO.setNewsId(id);
        sendNoticeDTO.setNewsTitle(newsTitle);
        sendNoticeDTO.setAllAuthFlag(allAuthFlag);
        sendNoticeDTO.setOrgSidList(orgSidList);
        sendNoticeDTO.setRoleSidList(roleSidList);
        sendNoticeDTO.setDeptSidList(deptSidList);
        sendNoticeDTO.setUserSidMap(userSidMap);
        sendNoticeDTO.setExternalUserSidMap(externalSidMap);
        sendNoticeDTO.setExternalUserList(externalUserList);

        // 发送事件
        SendNoticeEvent sendNoticeEvent = new SendNoticeEvent(this, sendNoticeDTO);
        eventPublisher.publishEvent(sendNoticeEvent);

        // 发送完成，更新为已发送过工作提醒
        this.update(new LambdaUpdateWrapper<NewsAnnouncement>()
                .eq(NewsAnnouncement::getId, id)
                .set(NewsAnnouncement::getAlreadyNoticeFlag, Constants.NUMBER_ONE));
    }


    /**
     * 保存公告权限
     *
     * @param saveAnnouncementReq 请求入参
     */
    private List<UserDTO> saveNewsAuthInner(SaveAnnouncementReq saveAnnouncementReq) {
        // 先删除当前权限关系
        newsAnnouncementAuthMapper.deleteByNewsId(saveAnnouncementReq.getId());

        List<UserDTO> externalUserList = Lists.newArrayList();
        String tenantId = Utils.getTenantId();
        // 全员可见
        if (Constants.NEWS_ALL_AUTH_FLAG_YES.equals(saveAnnouncementReq.getAllAuthFlag())) {
            NewsAnnouncementAut newsAnnouncementAuth = new NewsAnnouncementAut();
            newsAnnouncementAuth.setNewsId(saveAnnouncementReq.getId());
            newsAnnouncementAuth.setNewsTypeId(saveAnnouncementReq.getNewsTypeId());
            newsAnnouncementAuth.setAllAuthFlag(Constants.NEWS_ALL_AUTH_FLAG_YES);
            newsAnnouncementAuth.setCreateTenantId(tenantId);
            newsAnnouncementAuthMapper.insert(newsAnnouncementAuth);
        } else {
            List<AuthVO> authList = new ArrayList<>();
            authList.addAll(CollectionUtils.isEmpty(saveAnnouncementReq.getOrgList()) ? Lists.newArrayList() : saveAnnouncementReq.getOrgList());
            authList.addAll(CollectionUtils.isEmpty(saveAnnouncementReq.getRoleList()) ? Lists.newArrayList() : saveAnnouncementReq.getRoleList());
            authList.addAll(CollectionUtils.isEmpty(saveAnnouncementReq.getDeptList()) ? Lists.newArrayList() : saveAnnouncementReq.getDeptList());
            authList.addAll(CollectionUtils.isEmpty(saveAnnouncementReq.getUserList()) ? Lists.newArrayList() : saveAnnouncementReq.getUserList());
            authList.addAll(CollectionUtils.isEmpty(saveAnnouncementReq.getExternalUserList()) ? Lists.newArrayList() : saveAnnouncementReq.getExternalUserList());
            if (CollectionUtils.isNotEmpty(authList)) {
                List<NewsAnnouncementAut> newsAuthList = new ArrayList<>();
                authList.forEach(x -> {
                    NewsAnnouncementAut newsAuth = converter.convert(x, NewsAnnouncementAut.class);
                    // 如果是外部用户，需要保存外部用户的个人租户
                    if (BizAuthTypeEnum.EXTERNAL_USER.getValue().equals(x.getAuthType())) {
                        Map<String, Object> map = getPersonalTenantId(x.getAuthId());
                        if (MapUtils.isNotEmpty(map) && ObjectUtils.isNotEmpty(map.get("tenantId"))) {
                            newsAuth.setTenantId(String.valueOf(map.get("tenantId")));
                            // 外部用户信息，用于后续发送公告工作提醒
                            UserDTO userDTO = new UserDTO();
                            userDTO.setTenantId(String.valueOf(map.get("tenantId")));
                            userDTO.setId(String.valueOf(map.get("userId")));
                            externalUserList.add(userDTO);
                        }
                    }
                    newsAuth.setCreateTenantId(tenantId);
                    newsAuth.setNewsId(saveAnnouncementReq.getId());
                    newsAuth.setNewsTypeId(saveAnnouncementReq.getNewsTypeId());
                    newsAuthList.add(newsAuth);
                });
                newsAnnouncementAuthService.saveBatch(newsAuthList);
            }
        }
        return externalUserList;
    }

    /**
     * 如果是外部用户，需要保存外部用户的个人租户
     *
     * @param sid 用户sid
     * @return
     */
    public Map<String, Object> getPersonalTenantId(Long sid) {
        // 如果是外部用户，需要保存外部用户的个人租户
        UserInfoDTO userInfoDTO = iamService.queryUser(null, sid, AppAuthContextHolder.getContext().getAuthoredUser().getToken());
        //获取是否已存在个人租户和用户
        AuthoredUser authoredUser = new AuthoredUser();
        authoredUser.setToken(AppAuthContextHolder.getContext().getAuthoredUser().getToken());
        return iamService.getTenantAuth(authoredUser, userInfoDTO.getEmail(), userInfoDTO.getTelephone(), "Athena", 0);
    }

    /**
     * 构建对象
     *
     * @param saveAnnouncementReq 请求入参
     * @param newsAnnouncement    对象
     */
    private void buildNewAnnouncement(SaveAnnouncementReq saveAnnouncementReq, NewsAnnouncement newsAnnouncement) {
        newsAnnouncement.setPublishType(saveAnnouncementReq.getPublishType());
        newsAnnouncement.setPublishId(saveAnnouncementReq.getPublishId());
        newsAnnouncement.setPublishName(saveAnnouncementReq.getPublishName());
        if (NewsAnnouncementPublishTypeEnum.NONE.getCode().equals(saveAnnouncementReq.getPublishType())) {
            newsAnnouncement.setPublishName("");
            newsAnnouncement.setPublishId("");
        }
        newsAnnouncement.setNewsTitle(saveAnnouncementReq.getNewsTitle());
        newsAnnouncement.setNewsTypeId(saveAnnouncementReq.getNewsTypeId());
        newsAnnouncement.setNewsTypePath(NewsTypeConstants.NEWS_TYPE_FIRST_LEVEL_PARENT_ID.equals(saveAnnouncementReq.getParentNewsTypeId())
                ? String.valueOf(saveAnnouncementReq.getNewsTypeId()) : saveAnnouncementReq.getParentNewsTypeId() + "-" + saveAnnouncementReq.getNewsTypeId());
        newsAnnouncement.setNewsStatus(saveAnnouncementReq.getNewsStatus());
        newsAnnouncement.setNewsEffectiveTime(saveAnnouncementReq.getNewsEffectiveTime());
        newsAnnouncement.setNewsEndTime(saveAnnouncementReq.getNewsEndTime());
        newsAnnouncement.setSummary(saveAnnouncementReq.getSummary());
        newsAnnouncement.setNewsChannel(saveAnnouncementReq.getNewsChannel());
        newsAnnouncement.setDefaultImageIcon(saveAnnouncementReq.getDefaultImageIcon());
        newsAnnouncement.setCoverImageId(saveAnnouncementReq.getCoverImageId());
        newsAnnouncement.setNoticeFlag(saveAnnouncementReq.getNoticeFlag());
        newsAnnouncement.setNewsContent(saveAnnouncementReq.getNewsContent());
        if (CollectionUtils.isNotEmpty(saveAnnouncementReq.getFileIdList())) {
            List<String> fileList = saveAnnouncementReq.getFileIdList().stream().filter(ObjectUtils::isNotEmpty).collect(Collectors.toList());
            newsAnnouncement.setFileId(String.join(",", fileList));
        } else {
            newsAnnouncement.setFileId("");
        }
    }


    /**
     * 根据公告列表查询对应的公告类型map
     *
     * @param newsAnnouncementList
     * @return
     */
    private Map<Long, NewsAnnouncementType> queryNewsTypeByNewsList(List<NewsAnnouncement> newsAnnouncementList) {
        //公告分类
        Set<Long> newsTypeIdSet = Sets.newHashSet();
        for (NewsAnnouncement x : newsAnnouncementList) {
            if (StringUtils.isNotBlank(x.getNewsTypePath())) {
                newsTypeIdSet.addAll(Arrays.stream(x.getNewsTypePath().split("-")).map(Long::parseLong).collect(Collectors.toList()));
            } else {
                newsTypeIdSet.add(x.getNewsTypeId());
            }
        }
        List<NewsAnnouncementType> newsAnnouncementTypes = InterceptorIgnoreUtil.handler(() -> newsAnnouncementTypeMapper.selectBatchIds(new ArrayList<>(newsTypeIdSet)));
        return newsAnnouncementTypes.stream().collect(Collectors.toMap(NewsAnnouncementType::getId, Function.identity(), (a, b) -> a));
    }


    /**
     * 根据公告路径获取公告分类名称
     *
     * @param newsTypeMap
     * @param newsAnnouncement
     * @return
     */
    private String getNewsTypeNameByNewPath(Map<Long, NewsAnnouncementType> newsTypeMap, NewsAnnouncement newsAnnouncement) {
        String newsTypePath = StringUtils.isNotBlank(newsAnnouncement.getNewsTypePath()) ? newsAnnouncement.getNewsTypePath() : newsAnnouncement.getNewsTypeId().toString();
        StringBuilder newsTypeName = new StringBuilder();
        String[] split = newsTypePath.split("-");
        for (int i = 0; i < split.length; i++) {
            long newsTypeId = Long.parseLong(split[i]);
            NewsAnnouncementType x = newsTypeMap.get(newsTypeId);
            String tempTypeName = ObjectUtils.isNotEmpty(x) && StringUtils.isNotBlank(x.getName()) ? x.getName() : NewsTypeConstants.OTHER_NEWS_TYPE_NAME;
            newsTypeName.append(tempTypeName).append("/");
        }
        if (newsTypeName.length() > 0) {
            newsTypeName.deleteCharAt(newsTypeName.length() - 1);
        }
        return newsTypeName.toString();
    }


    /**
     * web端-前台查询用户公告列表
     *
     * @return 返回
     */
    @Override
    public PageInfoResp<NewsAnnouncementVO> queryUserNewsList(WebQueryNewsListPageReq webQueryNewsListPageReq) {
        List<Long> newsTypeIdList = webQueryNewsListPageReq.getNewsTypeIdList();
        if (webQueryNewsListPageReq.getPageNum() == null) {
            webQueryNewsListPageReq.setPageNum(1);
        }
        if (webQueryNewsListPageReq.getPageSize() == null) {
            webQueryNewsListPageReq.setPageSize(10);
        }
        PageInfoResp<NewsAnnouncementVO> pageInfoResp = new PageInfoResp<>();
        pageInfoResp.setPageSize(webQueryNewsListPageReq.getPageSize());
        pageInfoResp.setPageNo(webQueryNewsListPageReq.getPageNum());
        //查询用户三方公告
        List<NewsAnnouncement> thirdNewsList = queryThirdNewsList(newsTypeIdList);
        List<NewsAnnouncementAut> autList = newsAnnouncementAuthService.queryAuthNewsListByNewsTypeIdList(newsTypeIdList);
        if (CollectionUtils.isEmpty(autList) && CollectionUtils.isEmpty(thirdNewsList)) {
            return pageInfoResp;
        }
        String nowTime = DateUtils.getNowTime();
        // 公告id去重
        List<Long> newsIdDistinctList = autList.stream().map(NewsAnnouncementAut::getNewsId).distinct().collect(Collectors.toList());
        if (CollectionUtils.isNotEmpty(thirdNewsList)) {
            newsIdDistinctList.addAll(thirdNewsList.stream().map(NewsAnnouncement::getId).distinct().collect(Collectors.toList()));
        }
        // 查询用户的公告列表
        LambdaQueryWrapper<NewsAnnouncement> queryWrapper = Wrappers.lambdaQuery();
        queryWrapper.in(NewsAnnouncement::getNewsChannel, Lists.newArrayList(Constants.NewsChannelFlagEnum.ALL.getFlag(), Constants.NewsChannelFlagEnum.WEB.getFlag()));
        queryWrapper.eq(NewsAnnouncement::getNewsStatus, Constants.NewsAnnouncementStatusEnum.PUBLISHED.getFlag());
        queryWrapper.lt(NewsAnnouncement::getNewsEffectiveTime, nowTime);
        queryWrapper.gt(NewsAnnouncement::getNewsEndTime, nowTime);
        queryWrapper.in(CollectionUtils.isNotEmpty(newsTypeIdList), NewsAnnouncement::getNewsTypeId, newsTypeIdList);
        queryWrapper.in(NewsAnnouncement::getId, newsIdDistinctList);
        queryWrapper.select(NewsAnnouncement.class, info -> !info.getColumn().equals("news_content"));
        queryWrapper.orderByDesc(NewsAnnouncement::getToppedTime);
        queryWrapper.orderByDesc(NewsAnnouncement::getNewsEffectiveTime);
        queryWrapper.orderByDesc(NewsAnnouncement::getId);
        IPage<NewsAnnouncement> newsAnnouncementPage = InterceptorIgnoreUtil.handler((Supplier<IPage<NewsAnnouncement>>) () -> newsAnnouncementMapper.selectPage(new Page<>(webQueryNewsListPageReq.getPageNum(), webQueryNewsListPageReq.getPageSize()), queryWrapper));
        List<NewsAnnouncement> newsAnnouncementList = newsAnnouncementPage.getRecords();
        if (CollectionUtils.isEmpty(newsAnnouncementList)) {
            return pageInfoResp;
        }

        // 批量获取文件信息
        Map<String, Object> fileIdMap = queryFileInfoList(newsAnnouncementList);
        // 查询已读公告记录及公告类型名称
        List<Long> newIdList = newsAnnouncementList.stream().map(NewsAnnouncement::getId).collect(Collectors.toList());
        List<Long> readNewsIdList = Lists.newArrayList();
        Map<Long, NewsAnnouncementType> newsTypeMap = Maps.newHashMap();
        if (CollectionUtils.isNotEmpty(newIdList)) {
            List<NewsAnnouncementRead> readList = newsAnnouncementReadMapper.selectNewsAnnouncementReadList(newIdList);
            readNewsIdList = readList.stream().filter(x -> Utils.getUserId().equals(x.getCreateUserId())).map(NewsAnnouncementRead::getNewsId).collect(Collectors.toList());
            //公告分类
            newsTypeMap = queryNewsTypeByNewsList(newsAnnouncementList);
        }
        List<NewsAnnouncementVO> newsAnnouncementVOList = Lists.newArrayList();
        //获取有点赞的公告id
        Map<Long, Boolean> likeNewsIdMap = getNewsLikeMap(newsAnnouncementList);
        for (NewsAnnouncement x : newsAnnouncementList) {
            NewsAnnouncementVO newsAnnouncementVO = converter.convert(x, NewsAnnouncementVO.class);
            // 构建附件
            if (StringUtils.isNotBlank(x.getFileId()) && MapUtils.isNotEmpty(fileIdMap)) {
                List<FileInfoVO> fileList = buildDetailFileInfo(x.getFileId().split(","), fileIdMap);
                newsAnnouncementVO.setFileList(fileList);
            }
            // 封面图
            if (StringUtils.isNotBlank(x.getCoverImageId()) && MapUtils.isNotEmpty(fileIdMap)) {
                List<FileInfoVO> fileList = buildDetailFileInfo(x.getCoverImageId().split(","), fileIdMap);
                newsAnnouncementVO.setCoverImageInfo(fileList.get(0));
            }
            // 封面图需要拼接域名
            if (StringUtils.isNotBlank(x.getDefaultImageIcon())) {
                newsAnnouncementVO.setDefaultImageIcon(envProperties.getSemcWebUrl() + x.getDefaultImageIcon());
            }
            // 公告类型的名称
            newsAnnouncementVO.setNewsTypeName(getNewsTypeNameByNewPath(newsTypeMap, x));
            // 是否已读
            newsAnnouncementVO.setReadFlag(CollectionUtils.isNotEmpty(readNewsIdList) && readNewsIdList.contains(x.getId()));
            newsAnnouncementVO.setLikeFlag(Boolean.TRUE.equals(likeNewsIdMap.get(x.getId())));
            newsAnnouncementVOList.add(newsAnnouncementVO);
        }
        // 构建公告的已读、应读人员数量
        buildNewsReadUserList(newsAnnouncementVOList, false, false);
        pageInfoResp.setList(newsAnnouncementVOList);
        pageInfoResp.setTotalRecords((int) newsAnnouncementPage.getTotal());
        pageInfoResp.setTotalPages((int) newsAnnouncementPage.getPages());
        return pageInfoResp;
    }

    /**
     * 查询三方公告
     */
    @Override
    public List<NewsAnnouncement> queryThirdNewsList(List<Long> newsTypeIdList) {
        LambdaQueryWrapper<NewsAnnouncement> queryWrapper = new LambdaQueryWrapper<>();
        queryWrapper.eq(NewsAnnouncement::getTenantId, Utils.getTenantId());
        queryWrapper.eq(NewsAnnouncement::getNewsStatus, Constants.NewsAnnouncementStatusEnum.PUBLISHED.getFlag());
        queryWrapper.gt(NewsAnnouncement::getNewsEndTime, DateUtils.getNowTime());
        queryWrapper.eq(NewsAnnouncement::getNewsSource, Constants.ThirdNewsAnnouncementEnum.OUTSIDER_INFORMATION.getType());
        if (CollectionUtils.isNotEmpty(newsTypeIdList)) {
            queryWrapper.in(NewsAnnouncement::getNewsTypeId, newsTypeIdList);
        }
        queryWrapper.select(NewsAnnouncement.class, info -> !info.getColumn().equals("news_content"));
        List<NewsAnnouncement> thirdNewsList = InterceptorIgnoreUtil.handler(() -> newsAnnouncementMapper.selectList(queryWrapper));
        if (CollectionUtils.isNotEmpty(thirdNewsList)) {
            List<NewsAnnouncement> newsAnnouncements = thirdNewsAnnouncementService.buildThirdNewSWithSSO(thirdNewsList);
            return newsAnnouncements;
        }
        return Lists.newArrayList();
    }

    /**
     * web端/移动端-查询用户端公告详情
     *
     * @param newsAnnouncement 公告
     * @return 返回
     */
    public NewsAnnouncementVO queryUserNewsDetail(NewsAnnouncement newsAnnouncement, Integer callSource) {
        boolean isMobile = Integer.valueOf(0).equals(callSource) || callSource == null;
        NewsAnnouncementVO newsAnnouncementVO = converter.convert(newsAnnouncement, NewsAnnouncementVO.class);
        // 公告类型名称
        Map<Long, NewsAnnouncementType> newsTypeMap = queryNewsTypeByNewsList(Lists.newArrayList(newsAnnouncement));
        newsAnnouncementVO.setNewsTypeName(getNewsTypeNameByNewPath(newsTypeMap, newsAnnouncement));
        // 批量获取文件信息
        Map<String, Object> fileIdMap = queryFileInfoList(Collections.singletonList(newsAnnouncement));
        // 构建附件
        if (StringUtils.isNotBlank(newsAnnouncement.getFileId()) && MapUtils.isNotEmpty(fileIdMap)) {
            List<FileInfoVO> fileList = buildDetailFileInfo(newsAnnouncement.getFileId().split(","), fileIdMap);
            newsAnnouncementVO.setFileList(fileList);
        }
        // 封面图
        if (StringUtils.isNotBlank(newsAnnouncement.getCoverImageId())) {
            List<FileInfoVO> fileList = buildDetailFileInfo(newsAnnouncement.getCoverImageId().split(","), fileIdMap);
            newsAnnouncementVO.setCoverImageInfo(fileList.get(0));
        }
        // 构建公告的已读、未读人员列表
        buildNewsReadUserList(Collections.singletonList(newsAnnouncementVO), isMobile, false);
        //公告是否点赞
        newsAnnouncementVO.setLikeFlag(false);
        if (newsAnnouncementVO.getLikeCount() != null && newsAnnouncementVO.getLikeCount() > 0) {
            AuthoredUser authoredUser = AppAuthContextHolder.getContext().getAuthoredUser();
            LambdaQueryWrapper<NewsAnnouncementLike> queryWrapper = new LambdaQueryWrapper<>();
            queryWrapper.eq(NewsAnnouncementLike::getNewsId, newsAnnouncement.getId());
            queryWrapper.eq(NewsAnnouncementLike::getCreateUserSid, authoredUser.getSid());
            Long count = newsAnnouncementLikeMapper.selectCount(queryWrapper);
            newsAnnouncementVO.setLikeFlag(count != null && count > 0);
            if (isMobile) {
                newsAnnouncementVO.setLikeList(getNewLikeUserList(newsAnnouncement.getId())._1);
            }
        }
        newsAnnouncementVO.setFavoriteFlag(false);
        //公告是否收藏
        if (newsAnnouncementVO.getFavoriteCount() != null && newsAnnouncementVO.getFavoriteCount() > 0) {
            AuthoredUser authoredUser = AppAuthContextHolder.getContext().getAuthoredUser();
            LambdaQueryWrapper<NewsAnnouncementFavorite> queryWrapper = new LambdaQueryWrapper<>();
            queryWrapper.eq(NewsAnnouncementFavorite::getNewsId, newsAnnouncement.getId());
            queryWrapper.eq(NewsAnnouncementFavorite::getCreateUserSid, authoredUser.getSid());
            Long count = newsAnnouncementFavoriteMapper.selectCount(queryWrapper);
            newsAnnouncementVO.setFavoriteFlag(count != null && count > 0);
        }
        return newsAnnouncementVO;
    }

    private Tuple2<List<UserDTO>, List<NewsAnnouncementLike>> getNewLikeUserList(Long newsId) {
        LambdaQueryWrapper<NewsAnnouncementLike> queryWrapper = new LambdaQueryWrapper<>();
        queryWrapper.eq(NewsAnnouncementLike::getNewsId, newsId);
        queryWrapper.orderByDesc(NewsAnnouncementLike::getId);
        List<NewsAnnouncementLike> newsAnnouncementLikeList = InterceptorIgnoreUtil.handler(() -> newsAnnouncementLikeMapper.selectList(queryWrapper));
        List<UserDTO> likeUserList = newsAnnouncementLikeList.stream().map(newsAnnouncementLike -> {
            UserDTO userDTO = new UserDTO();
            if (StringUtils.isNotBlank(newsAnnouncementLike.getThirdNewsId()) && StringUtils.isNotBlank(newsAnnouncementLike.getThirdUserId())) {
                userDTO.setId(newsAnnouncementLike.getThirdUserId());
                userDTO.setName(newsAnnouncementLike.getThirdUserId());
            } else {
                userDTO.setId(newsAnnouncementLike.getCreateUserId());
                userDTO.setName(newsAnnouncementLike.getCreateUserName());
            }
            userDTO.setSid(newsAnnouncementLike.getCreateUserSid());

            return userDTO;
        }).collect(Collectors.toList());
        return new Tuple2<>(likeUserList, newsAnnouncementLikeList);
    }


    /**
     * 查询用户端公告详情，不包含附件
     *
     * @param id 主键id
     * @return 返回
     */
    @Override
    public NewsAnnouncement queryUserNewsById(Long id) {
        return newsAnnouncementMapper.selectNewsAnnouncementById(id);
    }

    @Override
    public ValidateNewsVO validateNews(Long id) {
        // 1000 代表存在
        LambdaQueryWrapper<NewsAnnouncement> queryWrapper = new LambdaQueryWrapper<>();
        queryWrapper.eq(NewsAnnouncement::getId, id);
        queryWrapper.select(NewsAnnouncement.class, info -> !info.getColumn().equals("news_content"));
        NewsAnnouncement newsAnnouncement = InterceptorIgnoreUtil.handler(() -> newsAnnouncementMapper.selectOne(queryWrapper));
        ValidateNewsVO validateNewsVO = new ValidateNewsVO();
        validateNewsVO.setCode(ValidateNewsVO.CODE_1000);
        // 1001 代表不存在
        if (ObjectUtils.isEmpty(newsAnnouncement)) {
            validateNewsVO.setCode(ValidateNewsVO.CODE_1001);
            return validateNewsVO;
        }
        validateNewsVO.setNewsAnnouncement(newsAnnouncement);
        // 1002 代表失效或者过期
        if (Arrays.asList(Constants.NewsAnnouncementStatusEnum.UNPUBLISHED.getFlag(), Constants.NewsAnnouncementStatusEnum.EXPIRED.getFlag()).contains(newsAnnouncement.getNewsStatus())
                || LocalDateTime.now().compareTo(LocalDateTime.parse(newsAnnouncement.getNewsEndTime(), DateTimeFormatter.ofPattern(DateUtils.DATE_TIME_NORMAL_FORMATTER))) > 0) {
            validateNewsVO.setCode(ValidateNewsVO.CODE_1002);
            return validateNewsVO;
        }
        if (Constants.ThirdNewsAnnouncementEnum.OUTSIDER_INFORMATION.getType().equals(newsAnnouncement.getNewsSource())) {
            return validateNewsVO;
        }
        validateNewsVO.setNewsAnnouncement(newsAnnouncement);
        // 1003 代表没有权限
        LambdaQueryWrapper<NewsAnnouncementAut> authQueryWrapper = new LambdaQueryWrapper<>();
        authQueryWrapper.eq(NewsAnnouncementAut::getNewsId, id);
        authQueryWrapper.select(NewsAnnouncementAut::getAllAuthFlag, NewsAnnouncementAut::getAuthId);
        List<NewsAnnouncementAut> authList = newsAnnouncementAuthMapper.selectList(authQueryWrapper);
        boolean anyMatch = authList.stream().anyMatch(m -> Constants.NEWS_ALL_AUTH_FLAG_YES.equals(m.getAllAuthFlag()));
        if (anyMatch) {
            return validateNewsVO;
        }
        AuthoredUser authoredUser = AppAuthContextHolder.getContext().getAuthoredUser();
        Long userSid = authoredUser.getSid();
        anyMatch = authList.stream().anyMatch(m -> userSid.equals(m.getAuthId()));
        if (anyMatch) {
            return validateNewsVO;
        }
        IamUserAuthInfo iamUserAuthInfo = iamUserService.getUserAuthSids(authoredUser.getUserId(), authoredUser.getTenantId(), authoredUser.getToken());
        if (iamUserAuthInfo == null) {
            //没有权限
            validateNewsVO.setCode(ValidateNewsVO.CODE_1003);
            return validateNewsVO;
        }
        if (iamUserAuthInfo.getRoleSids() == null) {
            iamUserAuthInfo.setRoleSids(Lists.newArrayList());
        }
        if (iamUserAuthInfo.getOrgSids() == null) {
            iamUserAuthInfo.setOrgSids(Lists.newArrayList());
        }
        if (iamUserAuthInfo.getDeptSids() == null) {
            iamUserAuthInfo.setDeptSids(Lists.newArrayList());
        }
        for (NewsAnnouncementAut newsAnnouncementAut : authList) {
            anyMatch = iamUserAuthInfo.getRoleSids().stream().anyMatch(m -> m.equals(newsAnnouncementAut.getAuthId()));
            if (anyMatch) {
                return validateNewsVO;
            }
            anyMatch = iamUserAuthInfo.getOrgSids().stream().anyMatch(m -> m.equals(newsAnnouncementAut.getAuthId()));
            if (anyMatch) {
                return validateNewsVO;
            }
            anyMatch = iamUserAuthInfo.getDeptSids().stream().anyMatch(m -> m.equals(newsAnnouncementAut.getAuthId()));
            if (anyMatch) {
                return validateNewsVO;
            }
        }
        //没有权限
        validateNewsVO.setCode(ValidateNewsVO.CODE_1003);
        return validateNewsVO;
    }


    @Override
    public Map<Long, ValidateNewsVO> validateNews(List<Long> idList) {
        // 1000 代表存在
        LambdaQueryWrapper<NewsAnnouncement> queryWrapper = new LambdaQueryWrapper<>();
        queryWrapper.in(NewsAnnouncement::getId, idList);
        queryWrapper.select(NewsAnnouncement.class, info -> !info.getColumn().equals("news_content"));
        List<NewsAnnouncement> newsList = InterceptorIgnoreUtil.handler(() -> newsAnnouncementMapper.selectList(queryWrapper));
        if (CollectionUtils.isEmpty(newsList)) {
            return Maps.newHashMap();
        }
        Map<Long, ValidateNewsVO> resultMap = Maps.newHashMap();
        Map<Long, NewsAnnouncement> newsMap = newsList.stream().collect(Collectors.toMap(NewsAnnouncement::getId, Function.identity(), (a, b) -> a));
        //查询公告权限
        LambdaQueryWrapper<NewsAnnouncementAut> authQueryWrapper = new LambdaQueryWrapper<>();
        authQueryWrapper.in(NewsAnnouncementAut::getNewsId, idList);
        authQueryWrapper.select(NewsAnnouncementAut::getAllAuthFlag, NewsAnnouncementAut::getAuthId, NewsAnnouncementAut::getNewsId);
        List<NewsAnnouncementAut> authAllList = newsAnnouncementAuthMapper.selectList(authQueryWrapper);
        Map<Long, List<NewsAnnouncementAut>> authMap = Maps.newHashMap();
        if (CollectionUtils.isNotEmpty(authAllList)) {
            authMap = authAllList.stream().collect(Collectors.groupingBy(NewsAnnouncementAut::getNewsId));
        }
        AuthoredUser authoredUser = AppAuthContextHolder.getContext().getAuthoredUser();
        IamUserAuthInfo iamUserAuthInfo = iamUserService.getUserAuthSids(authoredUser.getUserId(), authoredUser.getTenantId(), authoredUser.getToken());
        if (iamUserAuthInfo != null) {
            if (iamUserAuthInfo.getRoleSids() == null) {
                iamUserAuthInfo.setRoleSids(Lists.newArrayList());
            }
            if (iamUserAuthInfo.getOrgSids() == null) {
                iamUserAuthInfo.setOrgSids(Lists.newArrayList());
            }
            if (iamUserAuthInfo.getDeptSids() == null) {
                iamUserAuthInfo.setDeptSids(Lists.newArrayList());
            }
        }
        for (Long newId : idList) {
            NewsAnnouncement newsAnnouncement = newsMap.get(newId);
            ValidateNewsVO validateNewsVO = new ValidateNewsVO();
            validateNewsVO.setNewsAnnouncement(newsAnnouncement);
            validateNewsVO.setCode(ValidateNewsVO.CODE_1000);
            // 1001 代表不存在
            if (newsAnnouncement == null) {
                validateNewsVO.setCode(ValidateNewsVO.CODE_1001);
                resultMap.put(newId, validateNewsVO);
                continue;
            }
            // 1002 代表失效或者过期
            if (Arrays.asList(Constants.NewsAnnouncementStatusEnum.UNPUBLISHED.getFlag(), Constants.NewsAnnouncementStatusEnum.EXPIRED.getFlag()).contains(newsAnnouncement.getNewsStatus())
                    || LocalDateTime.now().compareTo(LocalDateTime.parse(newsAnnouncement.getNewsEndTime(), DateTimeFormatter.ofPattern(DateUtils.DATE_TIME_NORMAL_FORMATTER))) > 0) {
                validateNewsVO.setCode(ValidateNewsVO.CODE_1002);
                resultMap.put(newId, validateNewsVO);
                continue;
            }
            if (Constants.ThirdNewsAnnouncementEnum.OUTSIDER_INFORMATION.getType().equals(newsAnnouncement.getNewsSource())) {
                resultMap.put(newId, validateNewsVO);
                continue;
            }
            // 1003 代表没有权限
            List<NewsAnnouncementAut> authList = authMap.getOrDefault(newId, Lists.newArrayList());
            boolean anyMatch = authList.stream().anyMatch(m -> Constants.NEWS_ALL_AUTH_FLAG_YES.equals(m.getAllAuthFlag()));
            if (anyMatch) {
                resultMap.put(newId, validateNewsVO);
                continue;
            }
            Long userSid = authoredUser.getSid();
            anyMatch = authList.stream().anyMatch(m -> userSid.equals(m.getAuthId()));
            if (anyMatch) {
                resultMap.put(newId, validateNewsVO);
                continue;
            }
            if (iamUserAuthInfo == null) {
                //没有权限
                validateNewsVO.setCode(ValidateNewsVO.CODE_1003);
                resultMap.put(newId, validateNewsVO);
                continue;
            }
            boolean isBreak = false;
            for (NewsAnnouncementAut newsAnnouncementAut : authList) {
                anyMatch = iamUserAuthInfo.getRoleSids().stream().anyMatch(m -> m.equals(newsAnnouncementAut.getAuthId()));
                if (anyMatch) {
                    resultMap.put(newId, validateNewsVO);
                    isBreak = true;
                    break;
                }
                anyMatch = iamUserAuthInfo.getOrgSids().stream().anyMatch(m -> m.equals(newsAnnouncementAut.getAuthId()));
                if (anyMatch) {
                    resultMap.put(newId, validateNewsVO);
                    isBreak = true;
                    break;
                }
                anyMatch = iamUserAuthInfo.getDeptSids().stream().anyMatch(m -> m.equals(newsAnnouncementAut.getAuthId()));
                if (anyMatch) {
                    resultMap.put(newId, validateNewsVO);
                    isBreak = true;
                    break;
                }
            }
            if (isBreak) {
                continue;
            }
            //没有权限
            validateNewsVO.setCode(ValidateNewsVO.CODE_1003);
            resultMap.put(newId, validateNewsVO);
        }
        return resultMap;
    }

    /**
     * web端-查询用户的未读公告数
     *
     * @return 返回
     */
    @Override
    public Integer queryNewsUnReadCount() {
        List<NewsAnnouncementAut> autList = newsAnnouncementAuthService.queryAuthNewsList();
        if (CollectionUtils.isEmpty(autList)) {
            return 0;
        }
        // 公告id去重
        List<Long> newsIdDistinctList = autList.stream().map(NewsAnnouncementAut::getNewsId).distinct().collect(Collectors.toList());
        // 查询用户的公告列表
        List<NewsAnnouncement> newsAnnouncementList = newsAnnouncementMapper.selectNewsAnnouncementIdList(Constants.NewsAnnouncementStatusEnum.PUBLISHED.getFlag(),
                newsIdDistinctList, null, DateUtils.getNowTime(null));
        // 查询用户的三方公告列表
        newsAnnouncementList.addAll(queryThirdNewsList(null));
        if (CollectionUtils.isEmpty(newsAnnouncementList)) {
            return 0;
        }
        List<Long> idList = newsAnnouncementList.stream().map(NewsAnnouncement::getId).collect(Collectors.toList());
        // 查询已读公告记录
        LambdaQueryWrapper<NewsAnnouncementRead> readCondition = new LambdaQueryWrapper<>();
        readCondition.in(NewsAnnouncementRead::getNewsId, idList);
        readCondition.eq(NewsAnnouncementRead::getCreateUserId, AppAuthContextHolder.getContext().getAuthoredUser().getUserId());
        Long readCount = newsAnnouncementReadMapper.selectCount(readCondition);
        if (readCount == null || readCount <= 0) {
            return newsAnnouncementList.size();
        }
        return newsAnnouncementList.size() - readCount.intValue();
    }

    /**
     * 查询公告系统来源
     *
     * @return 返回
     */
    @Override
    public List<JSONObject> queryNewsAppList() {
        List<JSONObject> list = Lists.newArrayList();

        // 查询当前租户下公告列表
        QueryWrapper<NewsAnnouncement> condition = new QueryWrapper<>();
        condition.select("id", "news_title", "news_source", "third_news_id", "third_app_id", "third_app_code");
        condition.orderByDesc("topped_time").orderByDesc("create_time");
        condition.eq("tenant_id", AppAuthContextHolder.getContext().getAuthoredUser().getTenantId());
        List<NewsAnnouncement> newsAnnouncementList = newsAnnouncementMapper.selectList(condition);
        if (CollectionUtils.isEmpty(newsAnnouncementList)) {
            return list;
        }

        // 一方公告
        boolean flag = newsAnnouncementList.stream().anyMatch(x -> Constants.ThirdNewsAnnouncementEnum.INSIDER_INFORMATION.getType().equals(x.getNewsSource()));
        if (flag) {
            JSONObject jsonObject = new JSONObject();
            jsonObject.put("appCode", VirtualApplicationEnum.ATHENA_AGILE_INTERACTION.getCode());
            jsonObject.put("appName", VirtualApplicationEnum.ATHENA_AGILE_INTERACTION.getName());
            list.add(jsonObject);
        }

        // 过滤出三方公告列表
        List<String> thirdNewsAppCodeList = newsAnnouncementList.stream().filter(x -> Constants.ThirdNewsAnnouncementEnum.OUTSIDER_INFORMATION.getType().equals(x.getNewsSource()) && StringUtils.isNotBlank(x.getThirdAppCode()))
                .map(NewsAnnouncement::getThirdAppCode).distinct().collect(Collectors.toList());
        if (CollectionUtils.isNotEmpty(thirdNewsAppCodeList)) {
            List<AppLinkDTO> appLinkList = appLinkService.queryManageListSync(AppAuthContextHolder.getContext().getAuthoredUser());
            Map<String, String> appCodeMap = appLinkList.stream().collect(Collectors.toMap(AppLinkDTO::getCode, AppLinkDTO::getName, (a, b) -> a));
            for (String appCode : thirdNewsAppCodeList) {
                JSONObject jsonObject = new JSONObject();
                jsonObject.put("appCode", appCode);
                jsonObject.put("appName", appCodeMap.get(appCode));
                list.add(jsonObject);
            }
        }
        return list;
    }

    /**
     * 移动端-前台分页查询用户公告列表
     *
     * @param queryMobileNewsListPageReq 请求参数
     * @return 查询结果
     */
    @Override
    public PageInfoResp<NewsAnnouncementVO> queryMobileNewsListPage(QueryMobileNewsListPageReq queryMobileNewsListPageReq) {
        PageInfoResp<NewsAnnouncementVO> pageInfoResp = new PageInfoResp<>();
        List<Long> newsTypeIdList = queryMobileNewsListPageReq.getNewsTypeIdList();
        if (queryMobileNewsListPageReq.getPageNum() == null) {
            queryMobileNewsListPageReq.setPageNum(1);
        }
        if (queryMobileNewsListPageReq.getPageSize() == null) {
            queryMobileNewsListPageReq.setPageSize(10);
        }
        //兼容老版本，如果末级为空，则根据一级查询所有二级
        if (CollectionUtils.isEmpty(newsTypeIdList) && null != queryMobileNewsListPageReq.getNewsTypeId()) {
            newsTypeIdList = new ArrayList<>();
            //查询二级公告类型
            List<NewsAnnouncementType> announcementTypeList = newsAnnouncementTypeMapper.queryByParentId(queryMobileNewsListPageReq.getNewsTypeId());
            if (CollectionUtils.isEmpty(announcementTypeList)) {
                newsTypeIdList.add(queryMobileNewsListPageReq.getNewsTypeId());
            } else {
                List<Long> chirenTypeIdList = announcementTypeList.stream().map(NewsAnnouncementType::getId).distinct().collect(Collectors.toList());
                newsTypeIdList.addAll(chirenTypeIdList);
            }
        }
        List<NewsAnnouncementAut> autList = newsAnnouncementAuthService.queryAuthNewsListByNewsTypeIdList(newsTypeIdList);
        if (CollectionUtils.isEmpty(autList)) {
            return pageInfoResp;
        }
        // 公告id去重
        List<Long> newsIdDistinctList = autList.stream().map(NewsAnnouncementAut::getNewsId).distinct().collect(Collectors.toList());
        // 查询用户公告列表总数
        String currentTime = DateUtils.getNowTime(null);
        Integer count = newsAnnouncementMapper.queryMobileUserNewsListCount(newsIdDistinctList, queryMobileNewsListPageReq, Utils.getTenantId(), currentTime);
        if (count == 0) {
            return pageInfoResp;
        }
        // 分页查询用户公告列表
        Integer startNum = (queryMobileNewsListPageReq.getPageNum() - 1) * queryMobileNewsListPageReq.getPageSize();
        List<NewsAnnouncement> newsAnnouncementList = newsAnnouncementMapper.queryMobileUserNewsListPage(newsIdDistinctList, queryMobileNewsListPageReq, startNum, queryMobileNewsListPageReq.getPageSize(), Utils.getTenantId(), currentTime);

        // 查询已读公告记录
        List<Long> newIdList = newsAnnouncementList.stream().map(NewsAnnouncement::getId).collect(Collectors.toList());
        //当前用户已读的
        List<Long> newsIdList = Lists.newArrayList();
        Map<Long, List<NewsAnnouncementRead>> newsReadMap = Maps.newHashMap();
        if (CollectionUtils.isNotEmpty(newIdList)) {
            List<NewsAnnouncementRead> readList = newsAnnouncementReadMapper.selectNewsAnnouncementReadList(newIdList);
            newsReadMap = readList.stream().collect(Collectors.groupingBy(NewsAnnouncementRead::getNewsId));
            newsIdList = readList.stream().filter(x -> Utils.getUserId().equals(x.getCreateUserId())).map(NewsAnnouncementRead::getNewsId).collect(Collectors.toList());
        }
        //获取有点赞的公告id
        Map<Long, Boolean> likeNewsIdMap = getNewsLikeMap(newsAnnouncementList);
        // 批量获取文件信息
        Map<String, Object> fileIdMap = queryFileInfoList(newsAnnouncementList);

        // 构建返回对象
        List<NewsAnnouncementVO> newsAnnouncementVOList = Lists.newArrayList();
        for (NewsAnnouncement x : newsAnnouncementList) {
            NewsAnnouncementVO newsAnnouncementVO = converter.convert(x, NewsAnnouncementVO.class);
            // 移动端封面图需要拼接域名
            if (StringUtils.isNotBlank(x.getDefaultImageIcon())) {
                newsAnnouncementVO.setDefaultImageIcon(envProperties.getSemcWebUrl() + x.getDefaultImageIcon());
            }

            // 封面图
            if (StringUtils.isNotBlank(x.getCoverImageId()) && MapUtils.isNotEmpty(fileIdMap)) {
                List<FileInfoVO> fileList = buildDetailFileInfo(x.getCoverImageId().split(","), fileIdMap);
                newsAnnouncementVO.setCoverImageInfo(fileList.get(0));
            }
            // 是否已读
            newsAnnouncementVO.setReadFlag(CollectionUtils.isNotEmpty(newsIdList) && newsIdList.contains(x.getId()));
            List<NewsAnnouncementRead> newsAnnouncementReadList = newsReadMap.get(x.getId());
            newsAnnouncementVO.setReadCount(CollectionUtils.isNotEmpty(newsAnnouncementReadList) ? newsAnnouncementReadList.size() : 0);
            if (NewsAnnouncementPublishTypeEnum.NONE.getCode().equals(x.getPublishType())) {
                newsAnnouncementVO.setPublishId("");
                newsAnnouncementVO.setPublishName("");
            }
            newsAnnouncementVO.setLikeFlag(Boolean.TRUE.equals(likeNewsIdMap.get(x.getId())));
            newsAnnouncementVOList.add(newsAnnouncementVO);
        }
        int totalPages = count % queryMobileNewsListPageReq.getPageSize() == 0 ? count / queryMobileNewsListPageReq.getPageSize()
                : count / queryMobileNewsListPageReq.getPageSize() + 1;
        pageInfoResp.setTotalPages(totalPages);
        pageInfoResp.setPageNo(queryMobileNewsListPageReq.getPageNum());
        pageInfoResp.setPageSize(queryMobileNewsListPageReq.getPageSize());
        pageInfoResp.setTotalRecords(count);
        pageInfoResp.setList(newsAnnouncementVOList);
        return pageInfoResp;
    }

    private Map<Long, Boolean> getNewsLikeMap(List<NewsAnnouncement> newsAnnouncementList) {
        //获取有点赞的公告id
        List<Long> likeNewsIdList = newsAnnouncementList.stream().filter(m -> m.getLikeCount() != null && m.getLikeCount() > 0).map(NewsAnnouncement::getId).collect(Collectors.toList());
        if (CollectionUtils.isEmpty(likeNewsIdList)) {
            return Maps.newHashMap();
        }
        AuthoredUser authoredUser = AppAuthContextHolder.getContext().getAuthoredUser();
        LambdaQueryWrapper<NewsAnnouncementLike> queryWrapper = new LambdaQueryWrapper<>();
        queryWrapper.in(NewsAnnouncementLike::getNewsId, likeNewsIdList);
        queryWrapper.eq(NewsAnnouncementLike::getCreateUserSid, authoredUser.getSid());
        queryWrapper.select(NewsAnnouncementLike::getNewsId);
        List<NewsAnnouncementLike> newsAnnouncementLikeList = newsAnnouncementLikeMapper.selectList(queryWrapper);
        return newsAnnouncementLikeList.stream().collect(Collectors.toMap(NewsAnnouncementLike::getNewsId, n -> Boolean.TRUE));

    }

    /**
     * 查询当前租户移动端公告列表
     *
     * @return 查询结果
     */
    @Override
    public List<NewsAnnouncement> queryNewsListByCondition(QueryNewsListPageReq queryNewsListPageReq) {
        String currentTime = DateUtils.getNowTime(null);
        String tenantId = AppAuthContextHolder.getContext().getAuthoredUser().getTenantId();
        return newsAnnouncementMapper.queryNewsListByCondition(currentTime, tenantId, queryNewsListPageReq.getNewsChannelList(), queryNewsListPageReq.getNewsSource());
    }

    @Override
    public NewsAnnouncementStatisticsVO queryManagerNewsStatistics(NewsStatisticsReqDTO newsStatisticsReqDTO) {
        Long id = newsStatisticsReqDTO.getId();
        NewsAnnouncement newsAnnouncement = newsAnnouncementMapper.selectById(id);
        if (ObjectUtils.isEmpty(newsAnnouncement)) {
            return null;
        }
        List<NewsAnnouncementVO> newsAnnouncementVOList = Lists.newArrayList(converter.convert(newsAnnouncement, NewsAnnouncementVO.class));
        // 构建公告的已读、未读人员列表
        buildNewsReadUserList(newsAnnouncementVOList, true, true);
        NewsAnnouncementVO newsAnnouncementVO = newsAnnouncementVOList.get(0);
        NewsAnnouncementStatisticsVO newsAnnouncementStatisticsVO = new NewsAnnouncementStatisticsVO();
        newsAnnouncementStatisticsVO.setId(id);
        newsAnnouncementStatisticsVO.setReadCount(newsAnnouncementVO.getReadCount());
        newsAnnouncementStatisticsVO.setReadList(newsAnnouncementVO.getReadList());
        newsAnnouncementStatisticsVO.setUnReadCount(newsAnnouncementVO.getUnReadCount());
        newsAnnouncementStatisticsVO.setUnReadList(newsAnnouncementVO.getUnReadList());
        newsAnnouncementStatisticsVO.setLikeCount(newsAnnouncementVO.getLikeCount());
        newsAnnouncementStatisticsVO.setLikeList(Lists.newArrayList());
        if (newsAnnouncementVO.getLikeCount() != null && newsAnnouncementVO.getLikeCount() > 0) {
            Tuple2<List<UserDTO>, List<NewsAnnouncementLike>> tuple2 = getNewLikeUserList(newsAnnouncement.getId());
            newsAnnouncementStatisticsVO.setLikeList(tuple2._1);
        }
        newsAnnouncementStatisticsVO.setLikeCount(CollectionUtils.isNotEmpty(newsAnnouncementStatisticsVO.getLikeList()) ? newsAnnouncementStatisticsVO.getLikeList().size() : 0);
        return newsAnnouncementStatisticsVO;
    }

    /**
     * 发送工作提醒
     *
     * @param event 发送工作提醒事件
     */
    @Override
    public void sendNewsNotice(SendNoticeEvent event) {
        SendNoticeDTO sendNoticeDTO = event.getSendNoticeDTO();
        List<String> allUserIdList = Lists.newArrayList();
        List<UserDTO> externalUserList = Lists.newArrayList();

        // 全员可见
        if (Constants.NEWS_ALL_AUTH_FLAG_YES.equals(sendNoticeDTO.getAllAuthFlag())) {
            List<RoleDTO> roleList = iamService.queryEnabledUserList();
            // 内部用户id列表
            allUserIdList = roleList.stream().filter(x -> Constants.NUMBER_ZERO.equals(x.getUserType())).map(RoleDTO::getId).collect(Collectors.toList());
            // 外部用户，需要查询外部用户的个人租户和用户id
            boolean flag = commonConfigService.querySendNewsNoticeOuterFlag();
            if (flag) {
                List<Long> externalUserIdList = roleList.stream().filter(x -> Constants.NUMBER_ONE.equals(x.getUserType())).map(RoleDTO::getSid).collect(Collectors.toList());
                externalUserList = queryExternalUserList(externalUserIdList);
            }
        } else {
            // 组织下所有用户
            if (CollectionUtils.isNotEmpty(sendNoticeDTO.getOrgSidList())) {
                List<OrgRoleUserDTO> orgUserList = iamService.queryUserByAllOrgSidList(sendNoticeDTO.getOrgSidList());
                for (OrgRoleUserDTO orgRoleUserDTO : orgUserList) {
                    List<String> userIdList = orgRoleUserDTO.getUsers().stream().map(UserDTO::getId).collect(Collectors.toList());
                    allUserIdList.addAll(userIdList);
                }
            }
            // 角色下所有用户
            if (CollectionUtils.isNotEmpty(sendNoticeDTO.getRoleSidList())) {
                List<OrgRoleUserDTO> roleUserList = iamService.queryUserByAllRoleSidList(sendNoticeDTO.getRoleSidList());
                for (OrgRoleUserDTO orgRoleUserDTO : roleUserList) {
                    List<String> userIdList = orgRoleUserDTO.getUsers().stream().map(UserDTO::getId).collect(Collectors.toList());
                    allUserIdList.addAll(userIdList);
                }
            }
            // 部门下所有用户
            if (CollectionUtils.isNotEmpty(sendNoticeDTO.getDeptSidList())) {
                List<UserDeptInfoDTO> deptUserList = eocService.queryUserByDeptSidList(sendNoticeDTO.getDeptSidList());
                for (UserDeptInfoDTO userDeptInfoDTO : deptUserList) {
                    List<String> userIdList = userDeptInfoDTO.getEmps().stream().map(y -> y.getEmpUserId()).collect(Collectors.toList());
                    allUserIdList.addAll(userIdList);
                }
            }
            // 用户列表
            if (MapUtils.isNotEmpty(sendNoticeDTO.getUserSidMap())) {
                List<Long> sidList = Lists.newArrayList();
                for (Map.Entry<Long, String> entry : sendNoticeDTO.getUserSidMap().entrySet()) {
                    if (ObjectUtils.isEmpty(entry.getValue())) {
                        sidList.add(entry.getKey());
                    } else {
                        allUserIdList.add(entry.getValue());
                    }
                }
                if (CollectionUtils.isNotEmpty(sidList)) {
                    CompletableFuture[] completableFutures = new CompletableFuture[sidList.size()];
                    for (int i = 0; i < sidList.size(); i++) {
                        Long sid = sidList.get(i);
                        completableFutures[i] = CompletableFuture.supplyAsync(() -> iamService.queryUser(null, sid, Utils.getUserToken()), asyncTaskExecutor);
                    }
                    CompletableFuture.allOf(completableFutures).join();
                    for (CompletableFuture<UserInfoDTO> completableFuture : completableFutures) {
                        try {
                            UserInfoDTO userInfoDTO = completableFuture.get();
                            allUserIdList.add(userInfoDTO.getId());
                        } catch (Exception e) {
                            log.error("iamService | query user error：{}", e.getMessage(), e);
                        }
                    }
                }
            }
            // 外部用户列表，需要查询外部用户的个人租户和用户id
            if (MapUtils.isNotEmpty(sendNoticeDTO.getExternalUserSidMap())) {
                boolean flag = commonConfigService.querySendNewsNoticeOuterFlag();
                if (flag) {
                    List<Long> sidList = Lists.newArrayList();
                    for (Map.Entry<Long, NewsAnnouncementAut> entry : sendNoticeDTO.getExternalUserSidMap().entrySet()) {
                        if (ObjectUtils.isEmpty(entry.getValue()) && StringUtils.isNotBlank(entry.getValue().getAuthBizId())) {
                            UserDTO userDTO = new UserDTO();
                            userDTO.setTenantId(entry.getValue().getTenantId());
                            userDTO.setId(entry.getValue().getAuthBizId());
                            externalUserList.add(userDTO);
                        } else {
                            sidList.add(entry.getKey());
                        }
                    }
                    if (CollectionUtils.isNotEmpty(sidList)) {
                        List<UserDTO> userList = queryExternalUserList(sidList);
                        externalUserList.addAll(userList);
                    }
                }
            }
            // 外部用户列表(新增/编辑公告的场景已经查询出外部用户信息)
            if (CollectionUtils.isNotEmpty(sendNoticeDTO.getExternalUserList())) {
                externalUserList = sendNoticeDTO.getExternalUserList();
            }
        }

        // 构建消息体，发送工作提醒
        Long mewsId = sendNoticeDTO.getNewsId();
        String newsTitle = sendNoticeDTO.getNewsTitle();
        if (CollectionUtils.isNotEmpty(allUserIdList)) {
            allUserIdList = allUserIdList.stream().distinct().collect(Collectors.toList());
            sendNewsNoticeMQ(Utils.getTenantId(), mewsId, newsTitle, allUserIdList);
        }
        if (CollectionUtils.isNotEmpty(externalUserList)) {
            Map<String, List<UserDTO>> tenantIdMap = externalUserList.stream().collect(Collectors.groupingBy(UserDTO::getTenantId));
            tenantIdMap.forEach((k, v) -> {
                List<String> userIdList = v.stream().map(UserDTO::getId).distinct().collect(Collectors.toList());
                sendNewsNoticeMQ(k, mewsId, newsTitle, userIdList);
            });
        }
    }

    /**
     * 发送工作提醒
     *
     * @param tenantId      租户id
     * @param newsId        公告id
     * @param newsTitle     公告标题
     * @param allUserIdList 用户列表
     */
    private void sendNewsNoticeMQ(String tenantId, Long newsId, String newsTitle, List<String> allUserIdList) {
        MessageBatchUserDTO messageBatchUserDTO = new MessageBatchUserDTO();
        messageBatchUserDTO.setTenantId(tenantId);
        MessageDTO message = new MessageDTO();
        message.setType("news");
        message.setSource("semc");
        message.setNoticeOnlineUser(Boolean.TRUE);
        message.setNoticeMobileApp(Boolean.TRUE);
        message.setSendDate(LocalDateTime.now());
        message.setCreateDate(LocalDateTime.now());
        JSONObject jsonObject = new JSONObject();
        jsonObject.put("id", newsId);
        jsonObject.put("msg", newsTitle);
        jsonObject.put("title", "");
        message.setContent(jsonObject);
        messageBatchUserDTO.setMessage(message);

        // 用户分批发送消息
        MDC.put("userToken", Utils.getUserToken());
        // 获取发送的数量
        List<List<String>> partList = Lists.partition(allUserIdList, commonConfigService.querySendNewsNoticeUserCount());
        for (List<String> userIdList : partList) {
            MQMessageDTO messageDTO = new MQMessageDTO();
            messageDTO.setMessageId(IdUtil.randomUUID());
            messageBatchUserDTO.setUserIdList(userIdList);
            messageDTO.setMessage(JSON.toJSONString(messageBatchUserDTO));
            messageDTO.setMessageType(MQMessageTypeEnum.NEWS_SEND_NOTICE.getCode());
            messageDTO.setRetryMaxCount(3);
            rabbitMessageSender.sendWorkCommonMsg(messageDTO);
            ThreadUtil.sleep(100);
        }
    }

    /**
     * 查询外部用户信息
     *
     * @param externalUserIdList 外部用户sid列表
     * @return
     */
    private List<UserDTO> queryExternalUserList(List<Long> externalUserIdList) {
        List<UserDTO> externalUserList = Lists.newArrayList();
        if (CollectionUtils.isNotEmpty(externalUserIdList)) {
            CompletableFuture[] completableFutures = new CompletableFuture[externalUserIdList.size()];
            for (int i = 0; i < externalUserIdList.size(); i++) {
                long sid = externalUserIdList.get(i);
                completableFutures[i] = CompletableFuture.supplyAsync(() -> getPersonalTenantId(sid), asyncTaskExecutor);
            }
            CompletableFuture.allOf(completableFutures).join();
            for (CompletableFuture<Map<String, Object>> completableFuture : completableFutures) {
                try {
                    Map<String, Object> map = completableFuture.get();
                    if (MapUtils.isNotEmpty(map) && ObjectUtils.isNotEmpty(map.get("tenantId")) && ObjectUtils.isNotEmpty(map.get("userId"))) {
                        UserDTO userDTO = new UserDTO();
                        userDTO.setTenantId(String.valueOf(map.get("tenantId")));
                        userDTO.setId(String.valueOf(map.get("userId")));
                        externalUserList.add(userDTO);
                    }
                } catch (Exception e) {
                    log.error("query personal tenantId error：{}", e.getMessage(), e);
                }
            }
        }
        return externalUserList;
    }
}
