package com.digiwin.athena.semc.service.news.impl;

import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.baomidou.mybatisplus.core.toolkit.CollectionUtils;
import com.baomidou.mybatisplus.core.toolkit.ObjectUtils;
import com.baomidou.mybatisplus.core.toolkit.StringUtils;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.digiwin.athena.appcore.auth.AppAuthContextHolder;
import com.digiwin.athena.semc.common.Constants;
import com.digiwin.athena.semc.common.NewsTypeConstants;
import com.digiwin.athena.semc.common.RedisConstant;
import com.digiwin.athena.semc.dto.news.SaveNewsTypeAuthReq;
import com.digiwin.athena.semc.entity.news.NewsAnnouncementTypeAuth;
import com.digiwin.athena.semc.mapper.news.NewsAnnouncementTypeAuthMapper;
import com.digiwin.athena.semc.proxy.iam.service.IamService;
import com.digiwin.athena.semc.proxy.iam.service.model.RoleDTO;
import com.digiwin.athena.semc.proxywrapper.IamUserService;
import com.digiwin.athena.semc.proxywrapper.dto.IamUserAuthInfo;
import com.digiwin.athena.semc.service.cache.CommonConfigService;
import com.digiwin.athena.semc.service.cache.ICacheService;
import com.digiwin.athena.semc.service.news.NewsAnnouncementTypeAuthService;
import com.digiwin.athena.semc.util.FormatUtil;
import com.digiwin.athena.semc.util.Utils;
import com.digiwin.athena.semc.vo.news.NewsTypeAuthVO;
import com.google.common.base.Joiner;
import com.google.common.collect.Lists;
import io.vavr.Tuple2;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import javax.annotation.Resource;
import java.time.Duration;
import java.util.Collections;
import java.util.List;
import java.util.stream.Collectors;

/**
 * 公告类型权限表  表服务实现类
 *
 * @author sungqz
 * @since 2025-01-10
 */
@Service
public class NewsAnnouncementTypeAuthServiceImpl extends ServiceImpl<NewsAnnouncementTypeAuthMapper, NewsAnnouncementTypeAuth> implements NewsAnnouncementTypeAuthService {

    @Resource
    private NewsAnnouncementTypeAuthMapper newsAnnouncementTypeAuthMapper;

    @Resource
    private IamUserService iamUserService;

    @Resource
    private IamService iamService;

    @Resource
    private CommonConfigService commonConfigService;

    @Resource
    private ICacheService cacheService;

    /**
     * 管理后台-查询权限对象关联的公告类型
     *
     * @param authId 权限id
     * @return 返回
     */
    @Override
    public NewsTypeAuthVO queryAuthRelNewsType(String authId) {
        NewsTypeAuthVO newsTypeAuthVO = new NewsTypeAuthVO();
        List<NewsAnnouncementTypeAuth> newsAnnouncementTypeAuthList;
        if (StringUtils.isNotBlank(authId)) {
            LambdaQueryWrapper<NewsAnnouncementTypeAuth> condition = new LambdaQueryWrapper<NewsAnnouncementTypeAuth>()
                    .eq(NewsAnnouncementTypeAuth::getAuthId, authId);
            newsAnnouncementTypeAuthList = newsAnnouncementTypeAuthMapper.selectList(condition);
        } else {
            // 查询当前用户是否是超管
            boolean flag = querySuperAdminFlag();
            if (flag) {
                newsTypeAuthVO.setAllNewsTypeFlag(Constants.ALL_AUTH_FLAG_YES);
                return newsTypeAuthVO;
            }
            // 不是超管，进一步查询有权限的公告类型
            newsAnnouncementTypeAuthList = queryUserNewsTypeList();
        }
        // 构建返回对象
        return buildNewsTypeAuthVO(newsAnnouncementTypeAuthList);
    }

    /**
     * 获取当前用户是否是超管
     *
     * @return 返回 true:是 false:否
     */
    private boolean querySuperAdminFlag() {
        boolean flag;
        // 是否查询缓存开关配置
        Tuple2<Boolean, Integer> superAdminSwitchTuple2 = commonConfigService.querySuperAdminFlagSwitch();
        // 开关打开，查询缓存
        if (Boolean.TRUE.equals(superAdminSwitchTuple2._1)) {
            String cacheKey = FormatUtil.format(RedisConstant.SUPER_ADMIN_FLAG, AppAuthContextHolder.getContext().getAuthoredUser().getTenantId(), AppAuthContextHolder.getContext().getAuthoredUser().getUserId());
            String cacheValue = cacheService.getValue(cacheKey);
            if (StringUtils.isNotBlank(cacheValue)) {
                flag = Boolean.parseBoolean(cacheValue);
            } else {
                List<RoleDTO> roleDTOList = iamService.queryUserRoles(Utils.getUserId(), Utils.getUserToken());
                flag = roleDTOList.stream().anyMatch(x -> x.getId().equals(Constants.SUPER_ADMIN));
                cacheService.cache(cacheKey, Boolean.toString(flag), Duration.ofSeconds(superAdminSwitchTuple2._2));
            }
        } else {
            List<RoleDTO> roleDTOList = iamService.queryUserRoles(Utils.getUserId(), Utils.getUserToken());
            flag = roleDTOList.stream().anyMatch(x -> x.getId().equals(Constants.SUPER_ADMIN));
        }
        return flag;
    }

    /**
     * 构建权限返回对象
     *
     * @param newsAnnouncementTypeAuthList 请求入参
     * @return 返回
     */
    private NewsTypeAuthVO buildNewsTypeAuthVO(List<NewsAnnouncementTypeAuth> newsAnnouncementTypeAuthList) {
        NewsTypeAuthVO newsTypeAuthVO = new NewsTypeAuthVO();
        if (CollectionUtils.isEmpty(newsAnnouncementTypeAuthList)) {
            // 默认所有人都有"其他"公告类型的权限
            newsTypeAuthVO.setAuthorizedNewsTypeIdList(Collections.singletonList(NewsTypeConstants.NEWS_TYPE_OTHER_ID));
            newsTypeAuthVO.setAllNewsTypeFlag(Constants.ALL_AUTH_FLAG_NO);
            return newsTypeAuthVO;
        }

        // 判断是否有全部公告类型的权限
        boolean flag = newsAnnouncementTypeAuthList.stream().anyMatch(x -> Constants.ALL_AUTH_FLAG_YES.equals(x.getAllNewsTypeFlag()));
        if (flag) {
            newsTypeAuthVO.setAllNewsTypeFlag(Constants.ALL_AUTH_FLAG_YES);
        } else {
            List<Long> newsTypeIdList = newsAnnouncementTypeAuthList.stream().map(NewsAnnouncementTypeAuth::getNewsTypeId).collect(Collectors.toList());
            // 默认所有人都有"其他"公告类型的权限
            if (!newsTypeIdList.contains(NewsTypeConstants.NEWS_TYPE_OTHER_ID)) {
                newsTypeIdList.add(NewsTypeConstants.NEWS_TYPE_OTHER_ID);
            }
            newsTypeAuthVO.setAuthorizedNewsTypeIdList(newsTypeIdList);
            newsTypeAuthVO.setAllNewsTypeFlag(Constants.ALL_AUTH_FLAG_NO);
        }
        return newsTypeAuthVO;
    }

    /**
     * 管理后台-保存权限对象关联的公告类型
     *
     * @param saveNewsTypeAuthReq 请求入参
     */
    @Override
    @Transactional
    public void saveNewsTypeAuth(SaveNewsTypeAuthReq saveNewsTypeAuthReq) {
        // 先删除当前权限关系
        QueryWrapper<NewsAnnouncementTypeAuth> condition = new QueryWrapper<>();
        condition.eq("auth_id", saveNewsTypeAuthReq.getAuthId());
        newsAnnouncementTypeAuthMapper.delete(condition);

        // 新增权限关系
        if (Constants.ALL_AUTH_FLAG_YES.equals(saveNewsTypeAuthReq.getAllNewsTypeFlag())) {
            NewsAnnouncementTypeAuth newsAnnouncementTypeAuth = new NewsAnnouncementTypeAuth();
            newsAnnouncementTypeAuth.setAuthId(saveNewsTypeAuthReq.getAuthId());
            newsAnnouncementTypeAuth.setAuthName(saveNewsTypeAuthReq.getAuthName());
            newsAnnouncementTypeAuth.setAuthType(saveNewsTypeAuthReq.getAuthType());
            newsAnnouncementTypeAuth.setAllNewsTypeFlag(Constants.ALL_AUTH_FLAG_YES);
            this.save(newsAnnouncementTypeAuth);
        } else {
            if (CollectionUtils.isEmpty(saveNewsTypeAuthReq.getNewsTypeIdList())) {
                return;
            }
            List<NewsAnnouncementTypeAuth> newsAnnouncementTypeAuthListL = saveNewsTypeAuthReq.getNewsTypeIdList().stream().map(x -> {
                NewsAnnouncementTypeAuth newsAnnouncementTypeAuth = new NewsAnnouncementTypeAuth();
                newsAnnouncementTypeAuth.setNewsTypeId(x);
                newsAnnouncementTypeAuth.setAuthId(saveNewsTypeAuthReq.getAuthId());
                newsAnnouncementTypeAuth.setAuthName(saveNewsTypeAuthReq.getAuthName());
                newsAnnouncementTypeAuth.setAuthType(saveNewsTypeAuthReq.getAuthType());
                return newsAnnouncementTypeAuth;
            }).collect(Collectors.toList());
            // 500个一组分批存储，防止一次数据过大
            List<List<NewsAnnouncementTypeAuth>> partList = Lists.partition(newsAnnouncementTypeAuthListL, 500);
            for (List<NewsAnnouncementTypeAuth> tempList : partList) {
                this.saveBatch(tempList);
            }
        }
    }

    /**
     * 查询当前登录用户关联的公告类型
     *
     * @return 返回
     */
    private List<NewsAnnouncementTypeAuth> queryUserNewsTypeList() {
        // 查询当前用户有权限的公告类型
        String userId = AppAuthContextHolder.getContext().getAuthoredUser().getUserId();
        String userToken = AppAuthContextHolder.getContext().getAuthoredUser().getToken();
        String tenantId = AppAuthContextHolder.getContext().getAuthoredUser().getTenantId();
        IamUserAuthInfo iamUserAuthInfo = iamUserService.getUserAuthSids(userId, tenantId, userToken);

        List<Long> authIdList = Lists.newArrayList();
        if (iamUserAuthInfo != null && CollectionUtils.isNotEmpty(iamUserAuthInfo.getOrgSids())) {
            authIdList.addAll(iamUserAuthInfo.getOrgSids());
        }
        // 拼接角色条件
        if (iamUserAuthInfo != null && CollectionUtils.isNotEmpty(iamUserAuthInfo.getRoleSids())) {
            authIdList.addAll(iamUserAuthInfo.getRoleSids());
        }
        // 拼接部门条件
        if (iamUserAuthInfo != null && CollectionUtils.isNotEmpty(iamUserAuthInfo.getDeptSids())) {
            authIdList.addAll(iamUserAuthInfo.getDeptSids());
        }
        // 拼接用户条件
        Long userSid = AppAuthContextHolder.getContext().getAuthoredUser().getSid();
        if (ObjectUtils.isNotEmpty(userSid)) {
            authIdList.add(userSid);
        }
        List<NewsAnnouncementTypeAuth> newsAnnouncementTypeAuthList = Lists.newArrayList();
        if (CollectionUtils.isNotEmpty(authIdList)) {
            Collections.sort(authIdList);
            // 拼接组织条件
            StringBuilder authCondition = new StringBuilder("(auth_id in(" + Joiner.on(",").join(authIdList) + "))");
            QueryWrapper<NewsAnnouncementTypeAuth> condition = new QueryWrapper<>();
            condition.apply("(" + authCondition + ")");
            condition.select("all_news_type_flag", "news_type_id");
            newsAnnouncementTypeAuthList = newsAnnouncementTypeAuthMapper.selectList(condition);
        }
        return newsAnnouncementTypeAuthList;
    }
}
