package com.digiwin.athena.semc.service.news.impl;

import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.conditions.update.LambdaUpdateWrapper;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.digiwin.athena.appcore.auth.AppAuthContextHolder;
import com.digiwin.athena.semc.common.Constants;
import com.digiwin.athena.semc.common.NewsTypeConstants;
import com.digiwin.athena.semc.common.enums.DataChangeTypeEnum;
import com.digiwin.athena.semc.dto.news.SaveNewsTypeReqDTO;
import com.digiwin.athena.semc.entity.news.NewsAnnouncement;
import com.digiwin.athena.semc.entity.news.NewsAnnouncementType;
import com.digiwin.athena.semc.entity.news.NewsAnnouncementTypeAuth;
import com.digiwin.athena.semc.event.NewsChangeEvent;
import com.digiwin.athena.semc.mapper.news.NewsAnnouncementAuthMapper;
import com.digiwin.athena.semc.mapper.news.NewsAnnouncementMapper;
import com.digiwin.athena.semc.mapper.news.NewsAnnouncementTypeAuthMapper;
import com.digiwin.athena.semc.mapper.news.NewsAnnouncementTypeMapper;
import com.digiwin.athena.semc.service.news.NewsAnnouncementTypeService;
import com.digiwin.athena.semc.util.InterceptorIgnoreUtil;

import org.apache.commons.lang3.ObjectUtils;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.util.List;

import javax.annotation.Resource;

import io.github.linpeilie.Converter;

/**
 * 新闻公告类型表(TNewsAnnouncementType)表服务实现类
 *
 * @author sungqz
 * @since 2024-05-13
 */
@Service
public class NewsAnnouncementTypeServiceImpl extends ServiceImpl<NewsAnnouncementTypeMapper, NewsAnnouncementType> implements NewsAnnouncementTypeService {
    @Resource
    private NewsAnnouncementTypeMapper newsAnnouncementTypeMapper;

    @Resource
    private NewsAnnouncementMapper newsAnnouncementMapper;

    @Resource
    private NewsAnnouncementAuthMapper newsAnnouncementAuthMapper;

    @Resource
    private NewsAnnouncementTypeAuthMapper newsAnnouncementTypeAuthMapper;

    @Resource
    private Converter converter;

    /**
     * 查询公告类型
     *
     * @param id 公告类型id
     * @return 返回
     */
    @Override
    public List<NewsAnnouncementType> queryNewsTypeListByParentId(Long id) {
        return newsAnnouncementTypeMapper.queryNewsTypeListByParentId(AppAuthContextHolder.getContext().getAuthoredUser().getTenantId(), id);
    }

    @Override
    public boolean existByTypeId(Long id) {
        return InterceptorIgnoreUtil.handler(() -> {
            NewsAnnouncementType type = getById(id);
            return type != null;
        });
    }

    /**
     * 删除自定义类型
     *
     * @param newsAnnouncementType 公共类型
     */
    @Override
    @Transactional
    public void delNewsType(NewsAnnouncementType newsAnnouncementType) {
        // 删除公告类型
        newsAnnouncementTypeMapper.deleteById(newsAnnouncementType.getId());
        // 一级公告类型，需要删除该公告类型的子级类型
        if (NewsTypeConstants.NEWS_TYPE_FIRST_LEVEL_PARENT_ID.equals(newsAnnouncementType.getParentId())) {
            LambdaQueryWrapper<NewsAnnouncementType> wrapper = new LambdaQueryWrapper<>();
            wrapper.eq(NewsAnnouncementType::getParentId, newsAnnouncementType.getId());
            newsAnnouncementTypeMapper.delete(wrapper);
        }
        // 引用该公告类型的公告变为"其他"类型
        newsAnnouncementMapper.updateNewsTypeById(newsAnnouncementType.getId(), AppAuthContextHolder.getContext().getAuthoredUser().getTenantId());
        // 公告权限表中的公告类型变为"其他"类型
        newsAnnouncementAuthMapper.updateNewsTypeById(newsAnnouncementType.getId(), AppAuthContextHolder.getContext().getAuthoredUser().getTenantId());
        // 删除公告类型权限表中该公告类型，默认属于"其他"类型
        LambdaQueryWrapper<NewsAnnouncementTypeAuth> deleteWrapper = new LambdaQueryWrapper<>();
        deleteWrapper.eq(NewsAnnouncementTypeAuth::getNewsTypeId, newsAnnouncementType.getId());
        newsAnnouncementTypeAuthMapper.delete(deleteWrapper);
    }

    /**
     * 保存公告类型
     *
     * @param saveNewsTypeReqDTO 请求入参
     * @return 返回
     */
    @Override
    public Boolean saveNewsType(SaveNewsTypeReqDTO saveNewsTypeReqDTO) {
        NewsAnnouncementType newsAnnouncementType = converter.convert(saveNewsTypeReqDTO, NewsAnnouncementType.class);
        newsAnnouncementType.setLevel(NewsTypeConstants.NEWS_TYPE_FIRST_LEVEL_PARENT_ID.equals(saveNewsTypeReqDTO.getParentId()) ? NewsTypeConstants.NEWS_TYPE_FIRST_LEVEL : NewsTypeConstants.NEWS_TYPE_SECOND_LEVEL);
        return saveOrUpdate(newsAnnouncementType);
    }

    /**
     * 保存公告类型关联的公告数
     *
     * @param event 公告类型变更事件
     */
    public void saveNewsCount(NewsChangeEvent event) {
        if (DataChangeTypeEnum.ADD.equals(event.getEventDTO().getDataChangeType())
                || DataChangeTypeEnum.EDIT.equals(event.getEventDTO().getDataChangeType())) {
            // 更新对应的公告类型关联的公告数量
            Long oldNewsTypeId = event.getEventDTO().getOldNewsTypeIdList().get(0);
            Long currentNewsTypeId = event.getEventDTO().getCurrentNewsTypeId();
            updateNewsTypeCount(currentNewsTypeId);
            if (ObjectUtils.isNotEmpty(oldNewsTypeId) && !oldNewsTypeId.equals(currentNewsTypeId)) {
                updateNewsTypeCount(oldNewsTypeId);
            }
        }
        // 删除公告
        if (DataChangeTypeEnum.DELETE.equals(event.getEventDTO().getDataChangeType())) {
            List<Long> delNewsTypeIdList = event.getEventDTO().getOldNewsTypeIdList();
            delNewsTypeIdList.forEach(this::updateNewsTypeCount);
        }
    }

    /**
     * 更新公告类型关联的公告数量
     *
     * @param newsTypeId 公告类型
     */
    private void updateNewsTypeCount(Long newsTypeId) {
        InterceptorIgnoreUtil.handler(() -> {
            // 查询当前公告类型关联的公告
            LambdaQueryWrapper<NewsAnnouncement> condition = new LambdaQueryWrapper<>();
            condition.eq(NewsAnnouncement::getNewsTypeId, newsTypeId);
            condition.select(NewsAnnouncement::getId, NewsAnnouncement::getNewsStatus);
            List<NewsAnnouncement> newsAnnouncementList = newsAnnouncementMapper.selectList(condition);

            // 更新
            Integer totalCount = newsAnnouncementList.size();
            Long publishedCount = newsAnnouncementList.stream().filter(x -> Constants.NewsAnnouncementStatusEnum.PUBLISHED.getFlag().equals(x.getNewsStatus())).count();
            LambdaUpdateWrapper<NewsAnnouncementType> updateWrapper = new LambdaUpdateWrapper<>();
            updateWrapper.set(NewsAnnouncementType::getNewsCount, totalCount)
                    .set(NewsAnnouncementType::getPublishedNewsCount, publishedCount);
            updateWrapper.eq(NewsAnnouncementType::getId, newsTypeId);
            this.update(updateWrapper);
        });
    }
}
