package com.digiwin.athena.semc.service.news.impl;

import com.google.common.collect.Lists;

import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.digiwin.athena.semc.common.Constants;
import com.digiwin.athena.semc.common.PageInfoResp;
import com.digiwin.athena.semc.common.enums.ApplicationTypeEnum;
import com.digiwin.athena.semc.dto.news.ThirdNewsAnnouncementReq;
import com.digiwin.athena.semc.entity.applink.AppLinkDTO;
import com.digiwin.athena.semc.entity.news.NewsAnnouncement;
import com.digiwin.athena.semc.entity.portal.PreinstalledApplication;
import com.digiwin.athena.semc.entity.portal.PreinstalledApplicationInstance;
import com.digiwin.athena.semc.entity.sso.ErpSsoInfo;
import com.digiwin.athena.semc.entity.sso.ThirdSsoInfo;
import com.digiwin.athena.semc.mapper.news.NewsAnnouncementMapper;
import com.digiwin.athena.semc.mapper.portal.PreinstalledApplicationInstanceMapper;
import com.digiwin.athena.semc.mapper.portal.PreinstalledApplicationMapper;
import com.digiwin.athena.semc.mapper.sso.ErpSsoInfoMapper;
import com.digiwin.athena.semc.mapper.sso.ThirdSsoInfoMapper;
import com.digiwin.athena.semc.proxy.iam.service.IamService;
import com.digiwin.athena.semc.service.news.ThirdNewsAnnouncementService;
import com.digiwin.athena.semc.util.DateUtils;
import com.digiwin.athena.semc.vo.news.NewsAnnouncementVO;

import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.ObjectUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.stereotype.Service;

import java.time.LocalDateTime;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.function.Function;
import java.util.stream.Collectors;

import javax.annotation.Resource;

import io.github.linpeilie.Converter;
import io.vavr.Tuple3;
import lombok.extern.slf4j.Slf4j;

@Service
@Slf4j
public class ThirdNewsAnnouncementServiceImpl implements ThirdNewsAnnouncementService {

    private final String END_TIME = "2099-12-31 23:59:59";

    @Resource
    private NewsAnnouncementMapper newsAnnouncementMapper;

    @Resource
    private ThirdSsoInfoMapper thirdSsoInfoMapper;

    @Resource
    private ErpSsoInfoMapper erpSsoInfoMapper;

    @Resource
    private PreinstalledApplicationMapper preinstalledApplicationMapper;

    @Resource
    private IamService iamService;

    @Resource
    private PreinstalledApplicationInstanceMapper instanceMapper;

    @Resource
    private Converter converter;
    /**
     * 校验数据是否合法
     *
     * @param thirdNewsAnnouncementReqs
     * @return
     */
    @Override
    public String checkNewsDateIsValid(List<ThirdNewsAnnouncementReq> thirdNewsAnnouncementReqs) {
        String error = "";
        for (ThirdNewsAnnouncementReq newsReq : thirdNewsAnnouncementReqs) {
            //校验摘要大小
            if(StringUtils.isNotBlank(newsReq.getSummary()) && newsReq.getSummary().length()>300)
            {
                error ="摘要字数已超过规定范围";
                return error;
            }
            // 时间格式校验
            if (ObjectUtils.isEmpty(newsReq.getNewsEndTime())) {
                newsReq.setNewsEndTime(END_TIME);
                continue;
            }
            if (!DateUtils.validateDate(newsReq.getNewsEndTime(), DateUtils.DATE_TIME_NORMAL_FORMATTER)) {
                error = "截止时间格式不正确";
                return error;
            }
            // 截止日期必须大于当前时间+24h
            String tomorrow = DateUtils.addDate(DateUtils.getNowTime(null), DateUtils.DATE_TIME_NORMAL_FORMATTER, 1, DateUtils.DAY);
            if (newsReq.getNewsEndTime().compareTo(tomorrow) < 0) {
                error = "截止日期必须大于当前时间+24h";
                return error;
            }
        }
        return error;
    }

    @Override
    public void saveNewsList(List<ThirdNewsAnnouncementReq> thirdNewsAnnouncementReqs,String appId,String appCode,String tenantId, ApplicationTypeEnum applicationType) {
        if (CollectionUtils.isEmpty(thirdNewsAnnouncementReqs)) {
            return;
        }
        for (ThirdNewsAnnouncementReq thirdNewsAnnouncementReq : thirdNewsAnnouncementReqs) {
            NewsAnnouncement newsAnnouncement = new NewsAnnouncement();
            newsAnnouncement.setThirdAppCode(appCode);
            newsAnnouncement.setTenantId(tenantId);
            newsAnnouncement.setThirdAppId(appId);
            buildNewsAnnouncement(newsAnnouncement, thirdNewsAnnouncementReq,applicationType);
            //根据传入的thirdNewsId判断数据库中是否存在此条数据
            String thirdNewsId = thirdNewsAnnouncementReq.getThirdNewsId();
            NewsAnnouncement newsAnnouncementSql = newsAnnouncementMapper.selectByIdAndCode(thirdNewsId, appCode,tenantId);
            if (ObjectUtils.isEmpty(newsAnnouncementSql)) {
                newsAnnouncementMapper.thirdInsert(newsAnnouncement);
            } else {
                newsAnnouncement.setId(newsAnnouncementSql.getId());
                newsAnnouncementMapper.updateNewsAnnouncementById(newsAnnouncement);
            }
        }

    }

    @Override
    public String checkNewsIsValid(List<ThirdNewsAnnouncementReq> thirdNewsAnnouncementReqs) {
        String error = "";
        if(thirdNewsAnnouncementReqs.size()>500 || thirdNewsAnnouncementReqs.size()==0){
            return error = "单次最少传送1条公告,最多只能传送500条公告";
        }
        for (ThirdNewsAnnouncementReq thirdNewsAnnouncementReq : thirdNewsAnnouncementReqs) {
            if (ObjectUtils.isEmpty(thirdNewsAnnouncementReq.getThirdNewsId())) {
                return error = "三方公告id不能为空";
            }
            if (StringUtils.isBlank(thirdNewsAnnouncementReq.getNewsTitle()) || thirdNewsAnnouncementReq.getNewsTitle().length() >= 50) {
                return error = "公告标题规格不符合";
            }
            if (CollectionUtils.isEmpty(thirdNewsAnnouncementReq.getUserIds())) {
                return error = "未传入通知人员";
            }
            if(thirdNewsAnnouncementReq.getUserIds().size()>100){
                return error = "通知人员最多为100人";
            }
//            if (ObjectUtils.isEmpty(thirdNewsAnnouncementReq.getUrl())) {
//                return error = "三方网址不能为空";
//            }
            if(StringUtils.isNotBlank(thirdNewsAnnouncementReq.getSummary()) && thirdNewsAnnouncementReq.getSummary().length()>300)
            {
               return error ="摘要字数已超过规定范围";
            }
            if (!Arrays.asList(0, 1).contains(thirdNewsAnnouncementReq.getNewsStatus())) {
                return error = "公告状态不合法";
            }
            if (ObjectUtils.isEmpty(thirdNewsAnnouncementReq.getNewsTypeId()) || !Arrays.asList(1L, 2L, 3L, 4L).contains(thirdNewsAnnouncementReq.getNewsTypeId())) {
                return error = "公告类型错误";
            }
            // 时间格式校验
            if (ObjectUtils.isEmpty(thirdNewsAnnouncementReq.getNewsEndTime())) {
                thirdNewsAnnouncementReq.setNewsEndTime(END_TIME);
                continue;
            }
            if (!DateUtils.validateDate(thirdNewsAnnouncementReq.getNewsEndTime(), DateUtils.DATE_TIME_NORMAL_FORMATTER)) {
                return  error = "截止时间格式不正确";
            }
            // 截止日期必须大于当前时间+24h
            String tomorrow = DateUtils.addDate(DateUtils.getNowTime(null), DateUtils.DATE_TIME_NORMAL_FORMATTER, 1, DateUtils.DAY);
            if (thirdNewsAnnouncementReq.getNewsEndTime().compareTo(tomorrow) < 0) {
                return error = "截止日期必须大于当前时间+24h";
            }
        }
        return error;
    }

    @Override
    public PageInfoResp<NewsAnnouncementVO> queryThirdNewsListByPage(QueryWrapper<NewsAnnouncement> condition, Integer pageSize, Integer pageNum) {
        PageInfoResp<NewsAnnouncementVO> newsAnnouncementVOPageInfoResp = new PageInfoResp<>();
        newsAnnouncementVOPageInfoResp.setPageNo(pageNum);
        newsAnnouncementVOPageInfoResp.setPageSize(pageSize);
        Page<NewsAnnouncement> newsAnnouncementPage = new Page<>();
        if (ObjectUtils.isEmpty(condition)) {
            condition.eq("news_source", 1).eq("news_status", 1);
        }
        newsAnnouncementPage = newsAnnouncementMapper.selectPage(new Page<>(pageSize, pageNum), condition);
        List<NewsAnnouncement> records = newsAnnouncementPage.getRecords();
        List<NewsAnnouncementVO> result = new ArrayList<>();
        records.forEach(x -> {
            NewsAnnouncementVO newsAnnouncementVO =converter.convert(x, NewsAnnouncementVO.class);
            result.add(newsAnnouncementVO);
        });
        newsAnnouncementVOPageInfoResp.setPageSize(pageSize);
        newsAnnouncementVOPageInfoResp.setPageNo(pageNum);
        newsAnnouncementVOPageInfoResp.setTotalPages(Integer.parseInt(String.valueOf(newsAnnouncementPage.getPages())));
        newsAnnouncementVOPageInfoResp.setTotalRecords(Integer.parseInt(String.valueOf(newsAnnouncementPage.getTotal())));
        newsAnnouncementVOPageInfoResp.setList(result);
        return newsAnnouncementVOPageInfoResp;
    }

    @Override
    public Tuple3<String, String, ApplicationTypeEnum> checkAppCode(String appId, String appCode) {
        List<ThirdSsoInfo> thirdSsoInfos = thirdSsoInfoMapper.queryByAppCode(appId,appCode);
        List<ThirdSsoInfo> validListThird = thirdSsoInfos.stream().filter(x -> Constants.VALID_STATUS_ENABLE.equals(x.getValidStatus())).collect(Collectors.toList());
        if (CollectionUtils.isNotEmpty(validListThird)) {
            return new Tuple3<>("", validListThird.get(0).getTenantId(), ApplicationTypeEnum.BS_APPLICATION);
        }
        List<ErpSsoInfo> erpSsoInfos = erpSsoInfoMapper.queryByAppCode(appId,appCode);
        List<ErpSsoInfo> validListErp = erpSsoInfos.stream().filter(x -> Constants.VALID_STATUS_ENABLE.equals(x.getValidStatus())).collect(Collectors.toList());
        if (CollectionUtils.isNotEmpty(validListErp)) {
            return new Tuple3<>("", validListErp.get(0).getTenantId(), ApplicationTypeEnum.CS_APPLICATION);
        }
        List<PreinstalledApplication> preinstalledApplications = preinstalledApplicationMapper.getAllByAppCode(appCode);
        List<PreinstalledApplicationInstance> preinstalledApplicationInstances = new ArrayList<>();
        for (PreinstalledApplication preinstalledApplication : preinstalledApplications) {
            List<PreinstalledApplicationInstance> instances = instanceMapper.selectByAppId(appId, preinstalledApplication.getId());
            if(CollectionUtils.isNotEmpty(instances)){
                preinstalledApplicationInstances.addAll(instances);
            }
        }
        List<PreinstalledApplicationInstance> applications = preinstalledApplicationInstances.stream().filter(x -> Constants.VALID_STATUS_ENABLE.equals(x.getStatus())).collect(Collectors.toList());
        if (CollectionUtils.isNotEmpty(applications)) {
            return new Tuple3<>("", applications.get(0).getTenantId(), ApplicationTypeEnum.PRESET_APPLICATION);
        }
        return new Tuple3<>("当前应用不在配置中或未开启，无权推送公告","",null);
    }

    @Override
    public List<NewsAnnouncement> getThirdNewsList(NewsAnnouncement newsAnnouncement) {
        newsAnnouncement.setNewsSource(Constants.ThirdNewsAnnouncementEnum.OUTSIDER_INFORMATION.getType());
        List<NewsAnnouncement> list = newsAnnouncementMapper.selectAllNews(newsAnnouncement);
        return list;
    }


    /**
     * 三方公告组装SSO，和校验前台能否看到
     * @param newsAnnouncements
     * @return
     */
    @Override
    public List<NewsAnnouncement> buildThirdNewSWithSSO(List<NewsAnnouncement> newsAnnouncements) {
        //查询所有的sso配置
        List<AppLinkDTO> appLinkDTOS = Lists.newArrayList();
        Map<Integer, List<NewsAnnouncement>> thirdAppTypeNewsMap = newsAnnouncements.stream().collect(Collectors.groupingBy(NewsAnnouncement::getThirdAppType, Collectors.toList()));
        for (Map.Entry<Integer, List<NewsAnnouncement>> entry : thirdAppTypeNewsMap.entrySet()) {
            Integer appType = entry.getKey();
            List<String> thirdAppCodeList = entry.getValue().stream().map(NewsAnnouncement::getThirdAppCode).distinct().collect(Collectors.toList());
            if (CollectionUtils.isEmpty(thirdAppCodeList)) {
                continue;
            }
            //BS
            if (ApplicationTypeEnum.BS_APPLICATION.getType().equals(appType)) {
                LambdaQueryWrapper<ThirdSsoInfo> queryWrapper = new LambdaQueryWrapper<>();
                queryWrapper.in(ThirdSsoInfo::getAppCode, thirdAppCodeList);
                queryWrapper.eq(ThirdSsoInfo::getValidStatus, Constants.VALID_STATUS_ENABLE);
                queryWrapper.select(ThirdSsoInfo::getAppId);
                List<ThirdSsoInfo> list = thirdSsoInfoMapper.selectList(queryWrapper);
                list.forEach(m->{
                    AppLinkDTO appLinkDTO = new AppLinkDTO();
                    appLinkDTO.setApplicationAppId(m.getAppId());
                    appLinkDTOS.add(appLinkDTO);
                });
            }
            //CS
            else if (ApplicationTypeEnum.CS_APPLICATION.getType().equals(appType)) {
                LambdaQueryWrapper<ErpSsoInfo> queryWrapper = new LambdaQueryWrapper<>();
                queryWrapper.in(ErpSsoInfo::getCode, thirdAppCodeList);
                queryWrapper.eq(ErpSsoInfo::getValidStatus, Constants.VALID_STATUS_ENABLE);
                queryWrapper.select(ErpSsoInfo::getAppId);
                List<ErpSsoInfo> list = erpSsoInfoMapper.selectList(queryWrapper);
                list.forEach(m -> {
                    AppLinkDTO appLinkDTO = new AppLinkDTO();
                    appLinkDTO.setApplicationAppId(m.getAppId());
                    appLinkDTOS.add(appLinkDTO);
                });

            }
            //内置
            else if (ApplicationTypeEnum.PRESET_APPLICATION.getType().equals(appType)) {
                LambdaQueryWrapper<PreinstalledApplication> queryWrapper = new LambdaQueryWrapper<>();
                queryWrapper.in(PreinstalledApplication::getApplicationCode, thirdAppCodeList);
                queryWrapper.select(PreinstalledApplication::getId);
                List<PreinstalledApplication> preinstalledApplicationList = preinstalledApplicationMapper.selectList(queryWrapper);
                if (CollectionUtils.isNotEmpty(preinstalledApplicationList)) {
                    List<Long> preinstalledApplicationIdList = preinstalledApplicationList.stream().map(PreinstalledApplication::getId).collect(Collectors.toList());
                    LambdaQueryWrapper<PreinstalledApplicationInstance> instanceQueryWrapper = new LambdaQueryWrapper<>();
                    instanceQueryWrapper.in(PreinstalledApplicationInstance::getPreinstalledApplicationId, preinstalledApplicationIdList);
                    instanceQueryWrapper.eq(PreinstalledApplicationInstance::getStatus, Constants.VALID_STATUS_ENABLE);
                    instanceQueryWrapper.select(PreinstalledApplicationInstance::getAppId);
                    List<PreinstalledApplicationInstance> list = instanceMapper.selectList(instanceQueryWrapper);
                    list.forEach(m -> {
                        AppLinkDTO appLinkDTO = new AppLinkDTO();
                        appLinkDTO.setApplicationAppId(m.getAppId());
                        appLinkDTOS.add(appLinkDTO);
                    });
                }
            }
        }
        List<NewsAnnouncement> result = new ArrayList<>();
        //三方应用的账号  key为appId，value为三方账号对应的账号
        Map<String,String> thirdUserMap = new HashMap<>();
        Map<String, AppLinkDTO> appLinkDTOMap = appLinkDTOS.stream().filter(x -> !ObjectUtils.isEmpty(x.getApplicationAppId())).collect(Collectors.toMap(AppLinkDTO::getApplicationAppId, Function.identity()));
        for (NewsAnnouncement newsAnnouncement : newsAnnouncements) {
                if(appLinkDTOMap.containsKey(newsAnnouncement.getThirdAppId()) && ObjectUtils.isNotEmpty(appLinkDTOMap.get(newsAnnouncement.getThirdAppId()))){
                    if(!thirdUserMap.containsKey(newsAnnouncement.getThirdAppId()))
                    {
                        //获取当前租户下登录的IAM云端账号绑定的三方员工账号
                        String thirdUser = iamService.queryMappingEmpId(newsAnnouncement.getThirdAppCode());
                        if(StringUtils.isNotBlank(thirdUser)){
                            thirdUserMap.put(newsAnnouncement.getThirdAppId(),thirdUser);
                        } else {
                            continue;
                        }
                    }
                    if(newsAnnouncement.getThirdUserIds().contains(thirdUserMap.get(newsAnnouncement.getThirdAppId()))){
                        result.add(newsAnnouncement);
                    }
                }

        }
        return result;
    }

    private void buildNewsAnnouncement(NewsAnnouncement newsAnnouncement, ThirdNewsAnnouncementReq thirdNewsAnnouncementReq, ApplicationTypeEnum applicationType) {
        newsAnnouncement.setThirdNewsId(thirdNewsAnnouncementReq.getThirdNewsId());
        newsAnnouncement.setNewsTitle(thirdNewsAnnouncementReq.getNewsTitle());
        newsAnnouncement.setNewsStatus(thirdNewsAnnouncementReq.getNewsStatus() == null ? Constants.NewsAnnouncementStatusEnum.PUBLISHED.getFlag() : thirdNewsAnnouncementReq.getNewsStatus());
        newsAnnouncement.setNewsEndTime(thirdNewsAnnouncementReq.getNewsEndTime() == null ? "2099-12-31 23:59:59" : thirdNewsAnnouncementReq.getNewsEndTime());
        newsAnnouncement.setNewsTypeId(thirdNewsAnnouncementReq.getNewsTypeId());
        newsAnnouncement.setSummary(thirdNewsAnnouncementReq.getSummary());
        newsAnnouncement.setNewsContent(null);
        newsAnnouncement.setThirdUrl(thirdNewsAnnouncementReq.getUrl());
        newsAnnouncement.setThirdUserIds(String.join(",", thirdNewsAnnouncementReq.getUserIds()));
        newsAnnouncement.setNewsSource(Constants.ThirdNewsAnnouncementEnum.OUTSIDER_INFORMATION.getType());
        String dateTime = String.valueOf(LocalDateTime.now());
        newsAnnouncement.setCreateTime(dateTime);
        newsAnnouncement.setNewsEffectiveTime(dateTime);
        newsAnnouncement.setModifyTime(dateTime);
        newsAnnouncement.setThirdImgUrl(thirdNewsAnnouncementReq.getThirdImgUrl());
        newsAnnouncement.setThirdNewsHtml(thirdNewsAnnouncementReq.getThirdNewsHtml());
        newsAnnouncement.setThirdAppType(0);
        if (applicationType!=null) {
            newsAnnouncement.setThirdAppType(applicationType.getType());
        }

    }
}

