package com.digiwin.athena.semc.service.open.impl;

import com.digiwin.athena.appcore.exception.BusinessException;
import com.digiwin.athena.semc.common.enums.ApplicationTypeEnum;
import com.digiwin.athena.semc.dto.erpsso.AuthorizeDTO;
import com.digiwin.athena.semc.dto.erpsso.QuerySsoListReq;
import com.digiwin.athena.semc.dto.open.AppAccountDTO;
import com.digiwin.athena.semc.dto.open.AppJobAccountVO;
import com.digiwin.athena.semc.dto.open.AppJobUrlReq;
import com.digiwin.athena.semc.dto.open.AppJobVO;
import com.digiwin.athena.semc.entity.applink.AppLinkDTO;
import com.digiwin.athena.semc.entity.bench.SyncJobInfo;
import com.digiwin.athena.semc.entity.portal.LabelSystemData;
import com.digiwin.athena.semc.entity.portal.PreinstalledApplication;
import com.digiwin.athena.semc.entity.sso.ErpSsoInfo;
import com.digiwin.athena.semc.entity.sso.ThirdSsoInfo;
import com.digiwin.athena.semc.mapper.bench.SyncJobInfoMapper;
import com.digiwin.athena.semc.mapper.portal.LabelSystemDataMapper;
import com.digiwin.athena.semc.mapper.sso.ErpSsoInfoMapper;
import com.digiwin.athena.semc.mapper.sso.ThirdSsoInfoMapper;
import com.digiwin.athena.semc.proxy.iam.service.IamService;
import com.digiwin.athena.semc.proxy.km.service.model.AppJobDTO;
import com.digiwin.athena.semc.service.open.OpenAppJobService;
import com.digiwin.athena.semc.service.portal.IPreinstalledApplicationService;
import com.digiwin.athena.semc.service.portal.LabelSystemClickService;
import com.digiwin.athena.semc.service.portal.LabelSystemDataService;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.BooleanUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.stereotype.Service;

import javax.annotation.Resource;
import java.util.*;
import java.util.function.Function;
import java.util.stream.Collectors;

/**
 * @author 13816
 */
@Service
public class OpenAppJobServiceImpl implements OpenAppJobService {

    @Resource
    private ThirdSsoInfoMapper thirdSsoInfoMapper;

    @Resource
    private ErpSsoInfoMapper erpSsoInfoMapper;

    @Resource
    private SyncJobInfoMapper syncJobInfoMapper;

    @Resource
    private IPreinstalledApplicationService preinstalledApplicationService;

    @Resource
    private LabelSystemClickService labelSystemClickService;

    @Resource
    private LabelSystemDataMapper labelSystemDataMapper;

    @Resource
    private IamService iamService;

    @Resource
    private LabelSystemDataService labelSystemDataService;

    /**
     * 查询应用作业账号信息
     * 根据提供的应用链接信息列表，筛选并查询不同类型应用的作业账号信息
     *
     * @param appCodeList 应用链接信息列表，包含应用的代码和数据源类型
     * @return 返回一个列表，包含应用作业账号的信息
     */
    @Override
    public List<AppJobAccountVO> queryAppJobAccount(List<AppLinkDTO> appCodeList) {
        // 判断输入列表是否为空，如果为空则直接返回空列表
        if(CollectionUtils.isEmpty(appCodeList)){
            return Collections.emptyList();
        }

        // 过滤出具有有效主ID和数据源的应用链接
        List<AppLinkDTO> collect = appCodeList.stream().filter(item -> StringUtils.isNotBlank(item.getPrimaryId()) && Objects.nonNull(item.getDataSource())).collect(Collectors.toList());

        // 初始化用于存储应用作业账号信息的列表
        List<AppJobAccountVO> appJobAccounts = new ArrayList<>();
        // 初始化三个列表，分别存储CS、BS和预设应用的主ID
        List<Long> csList=new ArrayList<>();
        List<Long> bsList = new ArrayList<>();
        List<Long> preList = new ArrayList<>();

        // 根据应用的数据源类型，将主ID添加到相应的列表中
        collect.forEach(item->{
            if(ApplicationTypeEnum.CS_APPLICATION.getType().equals(item.getDataSource())){
                csList.add(Long.parseLong(item.getPrimaryId()));
            }else if(ApplicationTypeEnum.BS_APPLICATION.getType().equals(item.getDataSource())){
                bsList.add(Long.parseLong(item.getPrimaryId()));
            }else if(ApplicationTypeEnum.PRESET_APPLICATION.getType().equals(item.getDataSource())){
                preList.add(Long.parseLong(item.getPrimaryId()));
            }
        });

        // 查询CS应用的作业账号信息
        csJobAccounts(csList,appJobAccounts);
        // 如果BS应用列表不为空，则查询BS应用的作业账号信息
        bsJobAccounts(bsList,appJobAccounts);
        // 查询预设应用的作业账号信息
        preJobAccounts(preList,appJobAccounts);
        // 返回包含所有类型应用作业账号信息的列表
        return appJobAccounts;
    }

    /**
     * 根据CS应用列表和所有作业账号列表，查询并添加CS应用的作业账号信息到结果列表中
     *
     * @param csList CS应用ID列表，用于查询CS应用的作业账号信息
     * @param allAccounts 所有作业账号列表，用于存储包括CS应用作业账号在内的所有作业账号信息
     */
    private void csJobAccounts(List<Long> csList,List<AppJobAccountVO> allAccounts){
        // 如果CS应用列表不为空，则查询CS应用的作业账号信息
        if(CollectionUtils.isNotEmpty(csList)){
            QuerySsoListReq csReq = new QuerySsoListReq();
            csReq.setIdList(csList);
            List<ErpSsoInfo> erpInfoList = erpSsoInfoMapper.querySsoListByScope(csReq);
            // 如果查询结果不为空，则进一步处理以获取作业账号信息
            if(CollectionUtils.isNotEmpty(erpInfoList)){
                Map<String, ErpSsoInfo> ssoInfoMap = erpInfoList.stream().filter(item -> StringUtils.isNotBlank(item.getAppId())).collect(Collectors.toMap(ErpSsoInfo::getAppId, Function.identity(), (a, b) -> a));
                List<AppJobAccountVO> csAccounts = queryAppJobAccount(ssoInfoMap.keySet());
                // 如果获取到CS应用的作业账号信息，则补充应用代码和名称，并添加到结果列表中
                if(CollectionUtils.isNotEmpty(csAccounts)){
                    csAccounts.forEach(item->{
                        ErpSsoInfo erpSsoInfo = ssoInfoMap.get(item.getAppId());
                        if(Objects.nonNull(erpSsoInfo)){
                            item.setAppCode(erpSsoInfo.getCode());
                            item.setAppName(erpSsoInfo.getName());
                        }
                    });
                    allAccounts.addAll(csAccounts);
                }
            }
        }
    }

    /**
     * 根据给定的业务系统ID列表和所有应用作业账号列表，查询并补充业务系统的作业账号信息
     * 此方法首先检查业务系统ID列表是否非空，然后查询对应的第三方SSO信息，并进一步获取作业账号信息
     * 最后，将获取到的作业账号信息补充应用代码和名称，并添加到结果列表中
     *
     * @param bsList 业务系统ID列表，用于查询第三方SSO信息
     * @param allAccounts 所有应用作业账号列表，用于存储查询到的作业账号信息
     */
    private void bsJobAccounts(List<Long> bsList,List<AppJobAccountVO> allAccounts){
        // 检查业务系统ID列表是否非空
        if(CollectionUtils.isNotEmpty(bsList)){
            // 准备查询第三方SSO信息的请求对象，并设置需要查询的业务系统ID列表
            QuerySsoListReq bsReq = new QuerySsoListReq();
            bsReq.setIdList(bsList);
            List<ThirdSsoInfo> thirdSsoInfoList = thirdSsoInfoMapper.queryThirdListByScope(bsReq);
            // 如果查询结果不为空，则进一步处理以获取作业账号信息
            if(CollectionUtils.isNotEmpty(thirdSsoInfoList)){
                // 将查询到的第三方SSO信息转换为Map，以便后续使用应用ID进行快速查找
                Map<String, ThirdSsoInfo> thirdSsoInfoMap = thirdSsoInfoList.stream().filter(item -> StringUtils.isNotBlank(item.getAppId())).collect(Collectors.toMap(ThirdSsoInfo::getAppId, Function.identity(), (a, b) -> a));
                List<AppJobAccountVO> bsAccounts = queryAppJobAccount(thirdSsoInfoMap.keySet());
                // 如果获取到BS应用的作业账号信息，则补充应用代码和名称，并添加到结果列表中
                if(CollectionUtils.isNotEmpty(bsAccounts)){
                    bsAccounts.forEach(item->{
                        ThirdSsoInfo thirdSsoInfo = thirdSsoInfoMap.get(item.getAppId());
                        if(Objects.nonNull(thirdSsoInfo)){
                            item.setAppCode(thirdSsoInfo.getAppCode());
                            item.setAppName(thirdSsoInfo.getAppName());
                        }
                    });
                    allAccounts.addAll(bsAccounts);
                }
            }
        }
    }

    /**
     * 处理预设应用的作业账号信息
     * 此方法旨在从预设应用列表中查询并处理相关的作业账号信息，将其整合到所有作业账号列表中
     *
     * @param preList 预设应用ID列表，用于查询预设应用的作业账号信息
     * @param allAccounts 所有作业账号信息列表，用于存储包括预设应用在内的所有作业账号信息
     */
    private void preJobAccounts(List<Long> preList,List<AppJobAccountVO> allAccounts){
        // 如果预设应用列表不为空，则查询预设应用的作业账号信息
        if(CollectionUtils.isNotEmpty(preList)){
            List<PreinstalledApplication> preinstalledApplications = preinstalledApplicationService.queryPreinstalledApplicationList(preList);
            // 如果查询结果不为空，则进一步处理以获取作业账号信息
            if(CollectionUtils.isNotEmpty(preinstalledApplications)){
                // 将查询结果转换为Map，以便后续高效查询
                Map<String, PreinstalledApplication> preMap = preinstalledApplications.stream().filter(item -> StringUtils.isNotBlank(item.getAppId())).collect(Collectors.toMap(PreinstalledApplication::getAppId, Function.identity(), (a, b) -> a));
                // 查询预设应用的作业账号信息
                List<AppJobAccountVO> preAccounts = queryAppJobAccount(preMap.keySet());
                // 如果获取到预设应用的作业账号信息，则补充应用代码和名称，并添加到结果列表中
                if(CollectionUtils.isNotEmpty(preAccounts)){
                    preAccounts.forEach(item->{
                        PreinstalledApplication preInfo = preMap.get(item.getAppId());
                        if(Objects.nonNull(preInfo)){
                            item.setAppCode(preInfo.getApplicationCode());
                            item.setAppName(preInfo.getApplicationName());
                        }
                    });
                    allAccounts.addAll(preAccounts);
                }
            }
        }
    }

    /**
     * 根据应用ID列表查询相关的应用作业账户信息
     * 此方法首先检查输入的应用ID列表是否为空，如果为空则直接返回一个空列表
     * 接着，通过查询获得与这些应用ID相关的同步作业信息，并将这些信息按应用ID分组
     * 最后，遍历分组后的信息，为每个应用ID创建一个AppJobAccountDTO对象，包含该应用ID相关的所有账户ID，并将这些对象添加到结果列表中返回
     *
     * @param appIdList 应用ID列表，用于查询相关的应用作业账户信息
     * @return 返回一个AppJobAccountDTO对象列表，每个对象包含一个应用ID及其相关的所有账户ID
     */
    private List<AppJobAccountVO> queryAppJobAccount(Collection<String> appIdList){
        // 检查输入的应用ID列表是否为空，如果为空则直接返回一个空列表
        if(CollectionUtils.isEmpty(appIdList)){
            return Collections.emptyList();
        }
        // 通过查询获得与这些应用ID相关的同步作业信息
        List<SyncJobInfo> syncJobInfos = syncJobInfoMapper.queryAppAccountIds(appIdList);
        if(CollectionUtils.isEmpty(syncJobInfos)){
            return Collections.emptyList();
        }
        // 将查询到的同步作业信息按应用ID分组
        Map<String, List<SyncJobInfo>> collect = syncJobInfos.stream().collect(Collectors.groupingBy(SyncJobInfo::getAppId));
        // 初始化结果列表，大小为分组后的信息数量
        List<AppJobAccountVO> result = new ArrayList<>(collect.size());
        // 遍历分组后的信息，为每个应用ID创建一个AppJobAccountDTO对象，并将其添加到结果列表中
        collect.forEach((k, v) -> {
            AppJobAccountVO appJobAccountDTO = new AppJobAccountVO();
            // 设置AppJobAccountDTO对象的账户ID列表和应用ID
            List<AppAccountDTO> appAccountList = v.stream().map(item -> new AppAccountDTO(item.getAccountId(), item.getAccountName())).collect(Collectors.toList());
            appJobAccountDTO.setAccounts(appAccountList);
            appJobAccountDTO.setAppId(k);
            // 将创建的AppJobAccountDTO对象添加到结果列表中
            result.add(appJobAccountDTO);
        });
        // 返回结果列表
        return result;
    }


    @Override
    public List<AppJobVO> queryAppJobList(String applicationCode, Integer applicationType) {
        // 根据不同的应用类型，查询相应的应用信息
        String appCode;
        if(ApplicationTypeEnum.BS_APPLICATION.getType().equals(applicationType)){
            // 查询第三方SSO信息
            ThirdSsoInfo thirdSsoInfo = thirdSsoInfoMapper.queryAppConfigByAppCode(applicationCode);
            // 如果查询到第三方SSO信息，则设置查询请求的应用代码为该信息的ID
            if(Objects.isNull(thirdSsoInfo)){
                return Collections.emptyList();
            }
            appCode = String.valueOf(thirdSsoInfo.getId());
        }else if(ApplicationTypeEnum.CS_APPLICATION.getType().equals(applicationType)){
            // 查询ERP SSO信息
            ErpSsoInfo erpSsoInfo = erpSsoInfoMapper.queryAppConfigByAppCode(applicationCode);
            // 如果查询到ERP SSO信息，则设置查询请求的应用代码为该信息的ID
            if(Objects.isNull(erpSsoInfo)){
                return Collections.emptyList();
            }
            appCode = String.valueOf(erpSsoInfo.getId());
        }else if(ApplicationTypeEnum.PRESET_APPLICATION.getType().equals(applicationType)){
            // 查询预装应用信息
            PreinstalledApplication application = preinstalledApplicationService.getPreByCode(applicationCode, null);
            // 如果查询到预装应用信息，则设置查询请求的应用代码为该信息的ID
            if(Objects.isNull(application)){
                return Collections.emptyList();
            }
            appCode = String.valueOf(application.getId());
        }else if(ApplicationTypeEnum.IAM_APPLICATION.getType().equals(applicationType)){
            appCode = applicationCode;
        }else {
            return Collections.emptyList();
        }
        List<AppJobDTO.Job> appJobList = labelSystemDataService.getAppJobList(appCode, applicationType, true, null);
        if(CollectionUtils.isNotEmpty(appJobList)){
            return appJobList.stream().map(AppJobVO::new).collect(Collectors.toList());
        }
        return Collections.emptyList();
    }

    /**
     * 查询应用或作业链接
     * 根据请求的 dataSource 类型，查询相应的应用信息，并构建自定义应用链接
     * 如果需要，添加授权码
     *
     * @param jobUrlReq 包含应用代码和作业代码等信息的请求对象
     * @return 返回构建的自定义应用链接对象
     * @throws Exception 抛出异常，如果查询或构建过程中发生错误
     */
    @Override
    public AppLinkDTO queryAppOrJobLink(AppJobUrlReq jobUrlReq) throws Exception{
        // 创建标签系统数据对象，用于存储查询和构建链接所需的信息
        LabelSystemData labelSystemData = new LabelSystemData();
        // 设置数据类型为请求的 dataSource
        labelSystemData.setDataType(jobUrlReq.getDataSource());
        // 初始化数据分类为4，表示应用或作业
        labelSystemData.setDataCategory(4);

        // 根据不同的应用类型，查询相应的应用信息
        if(ApplicationTypeEnum.BS_APPLICATION.getType().equals(jobUrlReq.getDataSource())){
            // 查询第三方SSO信息
            ThirdSsoInfo thirdSsoInfo = thirdSsoInfoMapper.queryAppConfigByAppCode(jobUrlReq.getAppCode());
            // 如果查询结果为空，返回null
            if(Objects.isNull(thirdSsoInfo)){
                return null;
            }
            // 设置应用代码和应用ID
            labelSystemData.setAppCode(String.valueOf(thirdSsoInfo.getId()));
            labelSystemData.setAppId(thirdSsoInfo.getAppId());
        }else if(ApplicationTypeEnum.CS_APPLICATION.getType().equals(jobUrlReq.getDataSource())){
            // 查询ERP SSO信息
            ErpSsoInfo erpSsoInfo = erpSsoInfoMapper.queryAppConfigByAppCode(jobUrlReq.getAppCode());
            // 如果查询结果为空，返回null
            if(Objects.isNull(erpSsoInfo)){
                return null;
            }
            // 设置应用代码和应用ID
            labelSystemData.setAppCode(String.valueOf(erpSsoInfo.getId()));
            labelSystemData.setAppId(erpSsoInfo.getAppId());
        }else if(ApplicationTypeEnum.PRESET_APPLICATION.getType().equals(jobUrlReq.getDataSource())){
            // 查询预装应用信息
            PreinstalledApplication application = preinstalledApplicationService.getPreByCode(jobUrlReq.getAppCode(), null);
            // 如果查询结果为空，返回null
            if(Objects.isNull(application)){
                return null;
            }
            // 设置应用代码和应用ID
            labelSystemData.setAppCode(String.valueOf(application.getId()));
            labelSystemData.setAppId(application.getAppId());
        }else if(ApplicationTypeEnum.IAM_APPLICATION.getType().equals(jobUrlReq.getDataSource())){
            // 对于IAM应用，直接使用appCode作为应用代码和应用ID
            labelSystemData.setAppCode(jobUrlReq.getAppCode());
            labelSystemData.setAppId(jobUrlReq.getAppCode());
        }else {
            // 如果dataSource类型不匹配任何已知应用类型，返回null
            return null;
        }

        // 如果请求中包含作业代码，设置相应属性
        if(StringUtils.isNotBlank(jobUrlReq.getJobCode())){
            labelSystemData.setIsJob(0);
            labelSystemData.setDataCategory(3);
            labelSystemData.setWorkCode(jobUrlReq.getJobCode());
        }

        // 根据初始化的标签系统数据对象，查询系统数据
        LabelSystemData systemData = labelSystemDataMapper.getByInit(labelSystemData);
        // 如果查询结果不为空且有效状态为1，更新标签系统数据对象
        if(Objects.nonNull(systemData)&& Objects.equals(1,systemData.getValidStatus())){
            labelSystemData = systemData;
        }else if(StringUtils.isNotBlank(labelSystemData.getWorkCode()) &&
                !ApplicationTypeEnum.IAM_APPLICATION.getType().equals(jobUrlReq.getDataSource())){
            List<SyncJobInfo> syncJobInfos = syncJobInfoMapper.selByCode(labelSystemData.getAppId(), labelSystemData.getWorkCode());
            if(CollectionUtils.isEmpty(syncJobInfos)){
                throw BusinessException.create("job code not exist");
            }
        }
        // 调用服务构建自定义应用链接
        AppLinkDTO appLinkDTO = labelSystemClickService.buildCustomApp(labelSystemData);
        // 如果请求中要求添加授权码，添加授权码
        if(BooleanUtils.isTrue(jobUrlReq.getAddAuthCode())){
            addAuthorize(appLinkDTO);
        }
        // 返回构建的自定义应用链接对象
        return appLinkDTO;
    }

    private void addAuthorize(AppLinkDTO appLink) throws Exception {
        if (appLink == null) {
            return;
        }
        // 获取授权信息
        AuthorizeDTO authorizeDTO = iamService.queryAuthorizeInfo(appLink.getApplicationAppId(), appLink.getCallBackUrl());
        if (null == authorizeDTO) {
            return;
        }
        appLink.setCloudwebsite(appLink.getCloudwebsite() + "&code=" + authorizeDTO.getCode());
    }

}
