package com.digiwin.athena.semc.service.open.impl;

import com.digiwin.athena.semc.common.Constants;
import com.digiwin.athena.semc.common.ResponseBody;
import com.digiwin.athena.semc.common.ResultPageBean;
import com.digiwin.athena.semc.common.enums.ApplicationTypeEnum;
import com.digiwin.athena.semc.dto.PageInfo;
import com.digiwin.athena.semc.dto.open.SystemToDoReq;
import com.digiwin.athena.semc.dto.portal.PreSystemToDoReq;
import com.digiwin.athena.semc.dto.portal.TodoListResp;
import com.digiwin.athena.semc.entity.portal.LabelSystemData;
import com.digiwin.athena.semc.entity.portal.PreinstalledApplication;
import com.digiwin.athena.semc.entity.sso.ErpSsoInfo;
import com.digiwin.athena.semc.entity.sso.ThirdSsoInfo;
import com.digiwin.athena.semc.mapper.bench.SyncJobInfoMapper;
import com.digiwin.athena.semc.mapper.portal.LabelSystemDataMapper;
import com.digiwin.athena.semc.mapper.sso.ErpSsoInfoMapper;
import com.digiwin.athena.semc.mapper.sso.ThirdSsoInfoMapper;
import com.digiwin.athena.semc.service.open.OpenThirdToDoService;
import com.digiwin.athena.semc.service.portal.IPreinstalledApplicationService;
import com.digiwin.athena.semc.service.portal.LabelSystemDataService;
import com.digiwin.athena.semc.service.portal.LabelSystemSourceService;
import com.digiwin.athena.semc.service.portal.TodoListService;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.stereotype.Service;

import javax.annotation.Resource;
import java.util.*;

/**
 * @author 13816
 */
@Service
public class OpenThirdToDoServiceImpl implements OpenThirdToDoService {

    @Resource
    private LabelSystemSourceService labelSystemSourceService;

    @Resource
    private LabelSystemDataService labelSystemDataService;

    @Resource
    private LabelSystemDataMapper labelSystemDataMapper;

    @Resource
    private TodoListService todoListService;

    @Resource
    private ThirdSsoInfoMapper thirdSsoInfoMapper;

    @Resource
    private ErpSsoInfoMapper erpSsoInfoMapper;

    @Resource
    private IPreinstalledApplicationService preinstalledApplicationService;

    /**
     * 查询待办应用列表
     * 本方法从标签系统数据中查询并过滤出唯一的三方待办应用信息
     * 首先，它检查是否存在预定义的标签系统数据如果存在数据，则查询详细信息并去除重复的三方待办应用
     *
     * @return 返回过滤后的三方待办应用列表如果没有可用的数据或过滤后列表为空，则返回空列表
     */
    @Override
    public List<LabelSystemData> queryTodoAppList() {
        //校验组件是否存在
        List<Long> dataIds = labelSystemSourceService.selectPreSystem(Constants.DataTypeEnum.TYPE_THIRD_TODO.getVal(), Constants.LabelTypeEnum.SYSTEM_PRE.getVal());
        if(CollectionUtils.isEmpty(dataIds)){
            return Collections.emptyList();
        }
        List<LabelSystemData> labelSystemData = labelSystemDataService.queryDataInfoBy(dataIds, true);
        //过滤重复三方待办应用
        if(CollectionUtils.isNotEmpty(labelSystemData)){
            Set<String> appCodes = new HashSet<>();
            Iterator<LabelSystemData> iterator = labelSystemData.iterator();
            while (iterator.hasNext()) {
                LabelSystemData next = iterator.next();
                if(!appCodes.add(next.getAppCode())){
                    iterator.remove();
                }
            }
        }
        return labelSystemData;
    }

    /**
     * 查询待办事项列表
     * 根据不同的查询条件（ID或应用代码），构造查询请求对象并调用服务方法获取待办事项列表
     * 如果查询条件为空，则返回一个空的响应对象
     *
     * @param todoListReq 系统待办事项查询请求对象，包含分页信息和查询条件
     * @return TodoListResp 待办事项列表响应对象，包含查询结果
     */
    @Override
    public TodoListResp queryTodoList(SystemToDoReq todoListReq) {
        // 初始化待办事项列表响应对象
        TodoListResp todoListResp = new TodoListResp();
        // 初始化预处理系统待办事项请求对象
        PreSystemToDoReq preSystemToDoReq = new PreSystemToDoReq();
        // 设置分页信息
        preSystemToDoReq.setPageNum(todoListReq.getPageNum());
        preSystemToDoReq.setPageSize(todoListReq.getPageSize());

        // 如果查询条件中包含ID，直接使用该ID进行查询
        if(Objects.nonNull(todoListReq.getId())){
            preSystemToDoReq.setId(todoListReq.getId());
            todoListResp = todoListService.selectTodoList(preSystemToDoReq);
        // 如果查询条件中包含应用代码，先根据应用代码获取系统数据ID，再进行查询
        }else if(StringUtils.isNotBlank(todoListReq.getAppCode())){
            LabelSystemData labelSystemData = new LabelSystemData();
            labelSystemData.setAppCode(todoListReq.getAppCode());
            labelSystemData.setDataCategory(0);
            LabelSystemData systemData = labelSystemDataMapper.getByInit(labelSystemData);
            // 如果根据应用代码找到了对应的系统数据，则使用该数据的ID进行查询
            if(Objects.nonNull(systemData)){
                preSystemToDoReq.setId(systemData.getId());
                todoListResp = todoListService.selectTodoList(preSystemToDoReq);
            }
        }
        return todoListResp;
    }

    /**
     * 根据应用代码和类型查询待办事项列表
     * 此方法首先根据应用类型查询相应的应用信息，然后根据查询到的应用信息设置查询待办事项列表的请求参数
     * 最后调用查询方法获取并返回待办事项列表
     *
     * @param applicationCode 应用代码，用于查询特定应用的信息
     * @param applicationType 应用类型，决定查询哪种类型的应用信息
     * @param todoListReq 待办事项列表查询请求对象，包含查询条件
     * @return 返回查询到的待办事项列表响应对象
     */
    @Override
    public TodoListResp queryTodoListByCode(String applicationCode,Integer applicationType,SystemToDoReq todoListReq) {
        // 初始化查询请求的ID和应用代码为null，确保使用正确的查询条件
        todoListReq.setId(null);
        todoListReq.setAppCode(null);

        // 根据不同的应用类型，查询相应的应用信息
        if(ApplicationTypeEnum.BS_APPLICATION.getType().equals(applicationType)){
            // 查询第三方SSO信息
            ThirdSsoInfo thirdSsoInfo = thirdSsoInfoMapper.queryAppConfigByAppCode(applicationCode);
            // 如果查询到第三方SSO信息，则设置查询请求的应用代码为该信息的ID
            if(Objects.nonNull(thirdSsoInfo)){
                todoListReq.setAppCode(String.valueOf(thirdSsoInfo.getId()));
            }
        }else if(ApplicationTypeEnum.CS_APPLICATION.getType().equals(applicationType)){
            // 查询ERP SSO信息
            ErpSsoInfo erpSsoInfo = erpSsoInfoMapper.queryAppConfigByAppCode(applicationCode);
            // 如果查询到ERP SSO信息，则设置查询请求的应用代码为该信息的ID
            if(Objects.nonNull(erpSsoInfo)){
                todoListReq.setAppCode(String.valueOf(erpSsoInfo.getId()));
            }
        }else if(ApplicationTypeEnum.PRESET_APPLICATION.getType().equals(applicationType)){
            // 查询预装应用信息
            PreinstalledApplication application = preinstalledApplicationService.getPreByCode(applicationCode, null);
            // 如果查询到预装应用信息，则设置查询请求的应用代码为该信息的ID
            if(Objects.nonNull(application)){
                todoListReq.setAppCode(String.valueOf(application.getId()));
            }
        }
        // 根据设置好的查询请求，调用查询方法获取待办事项列表并返回
        return queryTodoList(todoListReq);
    }
}
