package com.digiwin.athena.semc.service.portal.impl;

import com.baomidou.mybatisplus.core.conditions.Wrapper;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;

import com.alibaba.fastjson.JSONObject;
import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.digiwin.athena.appcore.auth.AppAuthContextHolder;
import com.digiwin.athena.semc.common.Constants;
import com.digiwin.athena.semc.common.ResponseBody;
import com.digiwin.athena.semc.common.ResultPageBean;
import com.digiwin.athena.semc.common.enums.BizAuthTypeEnum;
import com.digiwin.athena.semc.common.enums.BizObjectTypeEnum;
import com.digiwin.athena.semc.common.enums.BusinessSceneEnum;
import com.digiwin.athena.semc.common.enums.FolderLevelEnum;
import com.digiwin.athena.semc.common.enums.FolderTypeEnum;
import com.digiwin.athena.semc.dto.PageInfo;
import com.digiwin.athena.semc.dto.portal.FineReportQueryReq;
import com.digiwin.athena.semc.dto.portal.SaveReportUrlParamReq;
import com.digiwin.athena.semc.entity.common.BizObjAuthRel;
import com.digiwin.athena.semc.entity.common.Folder;
import com.digiwin.athena.semc.entity.portal.FineReport;
import com.digiwin.athena.semc.mapper.common.FolderMapper;
import com.digiwin.athena.semc.mapper.portal.FineReportMapper;
import com.digiwin.athena.semc.proxy.eoc.service.EocService;
import com.digiwin.athena.semc.proxy.eoc.service.model.UserDeptInfoDTO;
import com.digiwin.athena.semc.service.common.IBizObjAuthRelService;
import com.digiwin.athena.semc.service.portal.IFineReportService;
import com.digiwin.athena.semc.vo.common.BizObjAuthRelVo;
import com.digiwin.athena.semc.vo.common.FolderViewVo;
import com.digiwin.athena.semc.vo.common.ReportCountVo;
import com.digiwin.athena.semc.vo.portal.FineReportVo;

import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.collections.MapUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.util.ArrayList;
import java.util.Comparator;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.function.Function;
import java.util.stream.Collectors;

import javax.annotation.Resource;

import io.github.linpeilie.Converter;
import lombok.extern.slf4j.Slf4j;

/**
 * @description: 帆软报表实现类
 * @createDate: 2023/5/26 13:56
 * @author: sunyfa
 */
@Slf4j
@Service
public class FineReportServiceImpl extends ServiceImpl<FineReportMapper, FineReport> implements IFineReportService {
    @Resource
    private IBizObjAuthRelService bizObjAuthRelService;

    @Resource
    private EocService eocService;

    @Resource
    private FolderMapper folderMapper;

    @Resource
    private FineReportMapper fineReportMapper;

    @Resource
    private Converter converter;

    /**
     * 一级部门
     */
    private static final int FIRST_LEVEL = 1;

    /**
     * 二级部门
     */
    private static final int SECOND_LEVEL = 2;

    /**
     * 三级部门
     */
    private static final int THIRD_LEVEL = 3;


    /**
     * @param
     * @return
     * @description: 新增/修改报表
     * @author: sunyfa
     */
    @Override
    @Transactional
    public void addOrModFineReport(FineReport fineReport, List<BizObjAuthRelVo> org, List<BizObjAuthRelVo> role, List<BizObjAuthRelVo> user) {
        Long fineReportId = fineReport.getId();
        // 新增
        if (Objects.isNull(fineReport.getId())) {
            fineReport.setCreateUserId(AppAuthContextHolder.getContext().getAuthoredUser().getUserId());
            fineReportMapper.insertFineReport(fineReport);
            fineReportId = fineReport.getId();
        } else { // 更新
            updateById(fineReport);
            // 删除报表的权限信息
            bizObjAuthRelService.getBaseMapper().delete(new QueryWrapper<BizObjAuthRel>()
                    .eq("biz_obj_type", BizObjectTypeEnum.FINE_REPORT.getValue()).eq("biz_obj_id", fineReport.getId()));
        }

        // 保存帆软报表的权限对象
        List<BizObjAuthRelVo> allBizObjAuthRelVoList = new ArrayList<>();
        allBizObjAuthRelVoList.addAll(org == null ? Lists.newArrayList() : org);
        allBizObjAuthRelVoList.addAll(role == null ? Lists.newArrayList() : role);
        allBizObjAuthRelVoList.addAll(user == null ? Lists.newArrayList() : user);
        if (CollectionUtils.isNotEmpty(allBizObjAuthRelVoList)) {
            List<BizObjAuthRel> bizObjAuthRelList = new ArrayList<>();
            Long finalFineReportId = fineReportId;
            allBizObjAuthRelVoList.forEach(x -> {
                BizObjAuthRel bizObjAuthRel = converter.convert(x, BizObjAuthRel.class);
                bizObjAuthRel.setBizObjId(String.valueOf(finalFineReportId));
                bizObjAuthRelList.add(bizObjAuthRel);
            });
            bizObjAuthRelService.saveBatch(bizObjAuthRelList);
        }
    }

    /**
     * @param
     * @return
     * @description: 删除报表
     * @author: sunyfa
     */
    @Override
    @Transactional
    public void delFineReport(Long id) {
        // 删除帆软报表的权限对象
        bizObjAuthRelService.getBaseMapper().delete(new QueryWrapper<BizObjAuthRel>().eq("biz_obj_type", BizObjectTypeEnum.FINE_REPORT.getValue()).eq("biz_obj_id", id));
        // 删除帆软报表
        removeById(id);
    }

    /**
     * @param
     * @return
     * @description: 查询当前目录下的所有报表
     * @author: sunyfa
     */
    @Override
    public List<FineReportVo> qryFineReport(Long folderId) {
        // 查询当前目录及子目录
        List<Folder> folderList = folderMapper.qryCurrentAndChildFolderList(FolderTypeEnum.FINE_REPORT.getValue(), folderId);
        List<Long> folderIdList = folderList.stream().map(Folder::getId).collect(Collectors.toList());
        // 查询当前目录及子目录的所有帆软报表,按报表顺序和创建时间升序排序
        List<FineReport> fineReportList = fineReportMapper.selectList(new QueryWrapper<FineReport>()
                .in("folder_id", folderIdList).orderByAsc("report_order", "create_time"));
        log.debug("FineReportService query fine report list, result:{}", JSONObject.toJSONString(fineReportList));
        if (CollectionUtils.isEmpty(fineReportList)) {
            return Lists.newArrayList();
        }
        // 构建报表返回列表
        return buildFineReportVoList(folderList, fineReportList);
    }

    /**
     * 构建报表返回列表
     *
     * @param folderList     目录列表
     * @param fineReportList 报表列表
     * @return
     */
    @Override
    public List<FineReportVo> buildFineReportVoList(List<Folder> folderList, List<FineReport> fineReportList) {
        List<FineReportVo> resultFineReportList = Lists.newArrayList();
        Map<Long, String> folderIdMap = folderList.stream().collect(Collectors.toMap(Folder::getId, Folder::getName));
        // 构建报表的返回对象
        fineReportList.forEach(x -> {
            FineReportVo fineReportVo = converter.convert(x, FineReportVo.class);
            fineReportVo.setFolderName(folderIdMap.get(x.getFolderId()));
            resultFineReportList.add(fineReportVo);
        });
        List<FineReportVo> sortedReportList = resultFineReportList.stream().sorted(Comparator.comparing(FineReportVo::getFolderName, Comparator.nullsLast(String::compareTo))
                .thenComparing(FineReportVo::getReportOrder, Comparator.nullsLast(Integer::compareTo))
                .thenComparing(FineReportVo::getCreateTime, Comparator.nullsLast(String::compareTo))).collect(Collectors.toList());

        // 构建报表的权限对象
        List<Long> reportIdList = fineReportList.stream().map(FineReport::getId).collect(Collectors.toList());
        QueryWrapper<BizObjAuthRel> cond = new QueryWrapper<BizObjAuthRel>().eq("biz_obj_type", BizObjectTypeEnum.FINE_REPORT.getValue()).in("biz_obj_id", reportIdList);
        List<BizObjAuthRel> bizObjAuthRelList = bizObjAuthRelService.getBaseMapper().selectList(cond);
        log.debug("FineReportService query business auth relation, result:{}", JSONObject.toJSONString(bizObjAuthRelList));
        if (CollectionUtils.isEmpty(bizObjAuthRelList)) {
            return sortedReportList;
        }

        // 根据报表id分类
        Map<String, List<BizObjAuthRel>> bizObjAuthRelMap = bizObjAuthRelList.stream().collect(Collectors.groupingBy(BizObjAuthRel::getBizObjId));
        sortedReportList.forEach(x -> {
            // 当前报表包含的角色列表
            if (CollectionUtils.isNotEmpty(bizObjAuthRelMap.get(String.valueOf(x.getId())))) {
                // 根据角色类型分类
                Map<Integer, List<BizObjAuthRel>> currentBizObjAuthRelMap = bizObjAuthRelMap.get(String.valueOf(x.getId())).stream().collect(Collectors.groupingBy(BizObjAuthRel::getAuthType));
                for (Map.Entry<Integer, List<BizObjAuthRel>> entry : currentBizObjAuthRelMap.entrySet()) {
                    List<BizObjAuthRelVo> bizObjAuthRelVoList = entry.getValue().stream().map(bizObjAuthRel -> {
                        BizObjAuthRelVo bizObjAuthRelVo =converter.convert(bizObjAuthRel, BizObjAuthRelVo.class);
                        return bizObjAuthRelVo;
                    }).collect(Collectors.toList());
                    // 设置组织权限
                    if (BizAuthTypeEnum.ORG.getValue().equals(entry.getKey())) {
                        x.setOrg(bizObjAuthRelVoList);
                    } else if (BizAuthTypeEnum.ROLE.getValue().equals(entry.getKey())) { // 设置角色权限
                        x.setRole(bizObjAuthRelVoList);
                    } else if (BizAuthTypeEnum.USER.getValue().equals(entry.getKey())) { // 设置用户权限
                        x.setUser(bizObjAuthRelVoList);
                    }
                }
            }
        });
        return sortedReportList;
    }

    @Override
    public ResultPageBean fineReportSearch(String tenantId, FineReportQueryReq reportPageQueryReq) {
        Integer startNum = (reportPageQueryReq.getPageNum() - 1) * reportPageQueryReq.getPageSize();
        List<FineReportVo> voList = folderMapper.fineReportSearch(reportPageQueryReq.getQryCondition(), tenantId, startNum, reportPageQueryReq.getPageSize());
        Integer totalCount = folderMapper.countReportSearch(reportPageQueryReq.getQryCondition(), tenantId);
        return ResultPageBean.success(ResponseBody.getInstance(voList, PageInfo.getPageInfo(reportPageQueryReq.getPageNum(), reportPageQueryReq.getPageSize(), totalCount)));
    }

    /**
     * @param
     * @return
     * @description: 查询当前目录下当前用户有权限的帆软报表，需要在链接中拼接用户信息
     * @author: sunyfa
     */
    @Override
    public List<FineReportVo> qryCurrentUserFineReport(Long folderId, String empId) {
        List<FineReportVo> resultFineReportList = Lists.newArrayList();
        // 查询当前用户 + 当前角色 + 当前组织的所有报表
        List<BizObjAuthRel> bizObjAuthRelList = bizObjAuthRelService.qryBizObjAuthRel(BizObjectTypeEnum.FINE_REPORT.getValue());
        if (CollectionUtils.isEmpty(bizObjAuthRelList)) {
            return resultFineReportList;
        }
        // 查询所有报表信息
        List<String> reportIdList = bizObjAuthRelList.stream().map(BizObjAuthRel::getBizObjId).collect(Collectors.toList());
        // 查询当前目录及子目录
        List<Folder> folderList = folderMapper.qryCurrentAndChildFolderList(FolderTypeEnum.FINE_REPORT.getValue(), folderId);
        List<Long> folderIdList = folderList.stream().map(Folder::getId).collect(Collectors.toList());
        List<FineReport> fineReportList = fineReportMapper.selectList(new QueryWrapper<FineReport>()
                .in("folder_id", folderIdList).in("id", reportIdList).orderByAsc("report_order", "create_time"));
        log.debug("FineReportService query fine report list result:{}", JSONObject.toJSONString(fineReportList));

        // 如果存在配置部门的报表，则查询当前用户的组织结构树，获取部门id
        boolean existDepartment = fineReportList.stream().anyMatch(x -> x.getDepartmentConfigFlag() != null);
        Map<Integer, List<UserDeptInfoDTO>> levelDeptMap = Maps.newHashMap();
        if (existDepartment) {
            List<UserDeptInfoDTO> deptInfoDTOList = queryDeptList(empId);
            levelDeptMap = deptInfoDTOList.stream().collect(Collectors.groupingBy(UserDeptInfoDTO::getLevel));
        }
        // 构建报表的返回对象
        Map<Long, String> folderIdMap = folderList.stream().collect(Collectors.toMap(Folder::getId, Folder::getName));
        for (FineReport x : fineReportList) {
            FineReportVo fineReportVo = converter.convert(x, FineReportVo.class);
            fineReportVo.setFolderName(folderIdMap.get(x.getFolderId()));

            // 报表链接中添加后缀
            StringBuilder reportUrl = new StringBuilder(fineReportVo.getUrl());
            // 配人员
            if (x.getUserConfigFlag() != null && Constants.DEFAULT_CONFIGURED_FLAG_YES.equals(x.getUserConfigFlag())) {
                reportUrl.append(reportUrl.toString().contains("?") ? "&" : "?").append("userId=").append(AppAuthContextHolder.getContext().getAuthoredUser().getUserId());
            }
            // 配部门，如果取不到当前部门就取一级部门id
            if (MapUtils.isNotEmpty(levelDeptMap)) {
                String deptId = CollectionUtils.isNotEmpty(levelDeptMap.get(x.getDepartmentConfigFlag()))
                        ? levelDeptMap.get(x.getDepartmentConfigFlag()).get(0).getDeptId() : levelDeptMap.get(FIRST_LEVEL).get(0).getDeptId();
                reportUrl.append(reportUrl.toString().contains("?") ? "&" : "?").append("departmentId=").append(deptId);
            }
            // 配工厂
            if (StringUtils.isNotBlank(x.getFactoryConfigName())) {
                reportUrl.append(reportUrl.toString().contains("?") ? "&" : "?").append("plant=").append(x.getFactoryConfigName());
            }
            fineReportVo.setUrl(reportUrl.toString());
            resultFineReportList.add(fineReportVo);
        }
        return resultFineReportList.stream().sorted(Comparator.comparing(FineReportVo::getFolderName, Comparator.nullsLast(String::compareTo))
                .thenComparing(FineReportVo::getReportOrder, Comparator.nullsLast(Integer::compareTo))
                .thenComparing(FineReportVo::getCreateTime, Comparator.nullsLast(String::compareTo))).collect(Collectors.toList());
    }

    /**
     * 部门设置层级
     *
     * @param empId
     * @return
     */
    private List<UserDeptInfoDTO> queryDeptList(String empId) {
        List<UserDeptInfoDTO> noRepeatDeptList = Lists.newArrayList();
        // 查询员工所在部门和所有上级部门
        List<UserDeptInfoDTO> deptInfoDTOList = eocService.queryUserDeptList(empId);
        if (CollectionUtils.isNotEmpty(deptInfoDTOList)) {
            // 部门去重
            Map<Long, UserDeptInfoDTO> deptInfoDTOMap = deptInfoDTOList.stream().collect(Collectors.toMap(UserDeptInfoDTO::getDeptSid, Function.identity(), (a, b) -> a));
            noRepeatDeptList = Lists.newArrayList(deptInfoDTOMap.values());
            // 一级部门
            List<UserDeptInfoDTO> firstLevelDeptList = noRepeatDeptList.stream().filter(x -> x.getParentDeptSid() == 0)
                    .peek(x -> x.setLevel(FIRST_LEVEL)).collect(Collectors.toList());
            // 二级部门
            List<Long> firstLevelDeptSidList = firstLevelDeptList.stream().map(UserDeptInfoDTO::getDeptSid).collect(Collectors.toList());
            List<UserDeptInfoDTO> secondLevelDeptList = noRepeatDeptList.stream().filter(x -> firstLevelDeptSidList.contains(x.getParentDeptSid()))
                    .peek(x -> x.setLevel(SECOND_LEVEL)).collect(Collectors.toList());
            // 三级部门
            List<Long> secondLevelDeptSidList = secondLevelDeptList.stream().map(UserDeptInfoDTO::getDeptSid).collect(Collectors.toList());
            List<UserDeptInfoDTO> thirdLevelDeptList = noRepeatDeptList.stream().filter(x -> !firstLevelDeptSidList.contains(x.getDeptSid()) && !secondLevelDeptSidList.contains(x.getDeptSid()))
                    .peek(x -> x.setLevel(THIRD_LEVEL)).collect(Collectors.toList());
        }
        return noRepeatDeptList;
    }

    @Override
    public Boolean delFolderData() {
        // TODO 删除当前目录下的所有报表数据

        return null;
    }

    /**
     * 校验报表是否存在
     *
     * @param type         校验报表名称或链接
     * @param fineReportVo 报表
     * @return
     */
    @Override
    public boolean isReportExist(int type, FineReportVo fineReportVo) {
        QueryWrapper<FineReport> condition = new QueryWrapper<>();
        if (Objects.nonNull(fineReportVo.getId())) {
            condition.ne("id", fineReportVo.getId());
        }
        // 校验报表名
        if (type == 1) {
            condition.eq("name", fineReportVo.getName());
            return exists(condition);
        } else if (type == 2) { // 校验报表链接
            condition.eq("url", fineReportVo.getUrl());
            return exists(condition);
        }
        return false;
    }

    @Override
    public Integer queryMaxOrder(Long folderId, String tenantId) {
        return fineReportMapper.queryMaxOrder(folderId, tenantId);
    }

    /**
     * 构建目录树结构，同时计算报表数量和子目录数量
     *
     * @return
     */
    @Override
    public List<FolderViewVo> qryFineReportFolder(Integer scene) {
        List<FolderViewVo> folderRespList = Lists.newArrayList();
        // 查询所有目录列表,如果没有目录，则新建默认目录
        List<Folder> folderList = folderMapper.selectList(new QueryWrapper<Folder>().eq("obj_type", FolderTypeEnum.FINE_REPORT.getValue()).orderByAsc("folder_order"));
        if (CollectionUtils.isEmpty(folderList)) {
            Folder folder = Folder.builder().name("报表目录").objType(FolderTypeEnum.FINE_REPORT.getValue()).level(1).folderOrder(1).build();
            folderMapper.insertFolder(folder);
            FolderViewVo folderViewVo = converter.convert(folder, FolderViewVo.class);
            folderRespList.add(folderViewVo);
            return folderRespList;
        }

        // 如果存在目录，则计算目录下报表数量和子目录数量
        // 管理后台，展示目录下的所有报表数量
        List<ReportCountVo> reportCountList = folderMapper.queryFolderReportCount();
        Map<Long, Integer> folderIdMap = reportCountList.stream().collect(Collectors.toMap(ReportCountVo::getFolderId, ReportCountVo::getReportCount));

        // portal页面，展示权限内的报表数据
        Map<Long, List<FineReport>> authRelReportMap = Maps.newHashMap();
        if (Objects.equals(BusinessSceneEnum.PORTAL.getValue(), scene)) {
            List<BizObjAuthRel> bizObjAuthRelList = bizObjAuthRelService.qryBizObjAuthRel(BizObjectTypeEnum.FINE_REPORT.getValue());
            if (CollectionUtils.isNotEmpty(bizObjAuthRelList)) {
                Set<String> reportIdSet = bizObjAuthRelList.stream().map(BizObjAuthRel::getBizObjId).collect(Collectors.toSet());
                List<FineReport> reportList = fineReportMapper.selectBatchIds(reportIdSet);
                authRelReportMap = reportList.stream().collect(Collectors.groupingBy(FineReport::getFolderId));
            }
        }

        // 所有一级目录
        List<Folder> firstFolderList = folderList.stream().filter(x -> x.getParentFolderId() == null).collect(Collectors.toList());
        for (Folder folder : firstFolderList) {
            int totalReportCount = buildTotalReportCount(folderIdMap, authRelReportMap, scene, folder);
            // 二级目录下报表数量
            List<Long> secondFolderIdList = folderList.stream().filter(x -> Objects.equals(folder.getId(), x.getParentFolderId())).map(Folder::getId).collect(Collectors.toList());
            totalReportCount = AccumulatedTotalReportCount(folderIdMap, authRelReportMap, scene, secondFolderIdList, totalReportCount);
            // 三级目录下报表数量
            List<Long> thirdFolderIdList = folderList.stream().filter(x -> secondFolderIdList.contains(x.getParentFolderId())).map(Folder::getId).collect(Collectors.toList());
            totalReportCount = AccumulatedTotalReportCount(folderIdMap, authRelReportMap, scene, thirdFolderIdList, totalReportCount);

            FolderViewVo folderViewVo = converter.convert(folder, FolderViewVo.class);
            folderViewVo.setChildCount(secondFolderIdList.size());
            folderViewVo.setDataCount(totalReportCount);
            folderRespList.add(folderViewVo);
        }
        // 所有二级目录
        List<Folder> secondLevelList = folderList.stream().filter(x -> FolderLevelEnum.SECOND.getValue().equals(x.getLevel())).collect(Collectors.toList());
        for (Folder folder : secondLevelList) {
            int totalReportCount = buildTotalReportCount(folderIdMap, authRelReportMap, scene, folder);

            // 三级目录下报表数量
            List<Long> thirdFolderIdList = folderList.stream().filter(x -> Objects.equals(folder.getId(), x.getParentFolderId())).map(Folder::getId).collect(Collectors.toList());
            totalReportCount = AccumulatedTotalReportCount(folderIdMap, authRelReportMap, scene, thirdFolderIdList, totalReportCount);

            FolderViewVo folderViewVo = converter.convert(folder, FolderViewVo.class);
            folderViewVo.setChildCount(thirdFolderIdList.size());
            folderViewVo.setDataCount(totalReportCount);
            folderRespList.add(folderViewVo);
        }
        // 所有三级目录
        List<Folder> thirdLevelList = folderList.stream().filter(x -> FolderLevelEnum.THIRD.getValue().equals(x.getLevel())).collect(Collectors.toList());
        for (Folder folder : thirdLevelList) {
            FolderViewVo folderViewVo = converter.convert(folder, FolderViewVo.class);
            if (Objects.equals(BusinessSceneEnum.BACK_MANAGEMENT.getValue(), scene)) {
                folderViewVo.setDataCount(folderIdMap.get(folder.getId()));
            } else {
                folderViewVo.setDataCount(!Objects.isNull(authRelReportMap.get(folder.getId())) ? authRelReportMap.get(folder.getId()).size() : 0);
            }
            folderRespList.add(folderViewVo);
        }
        // 构建目录树
        List<FolderViewVo> firstFolderViewList = folderRespList.stream().filter(x -> x.getParentFolderId() == null).collect(Collectors.toList());
        buildTree(firstFolderViewList, folderRespList);
        return firstFolderViewList;
    }

    /**
     * 构建目录树结构
     *
     * @param parentFolderList 父目录
     * @param allFolderList    所有目录
     */
    private void buildTree(List<FolderViewVo> parentFolderList, List<FolderViewVo> allFolderList) {
        for (FolderViewVo parent : parentFolderList) {
            // 子目录
            List<FolderViewVo> childFolderList = allFolderList.stream().filter(x -> parent.getId().equals(x.getParentFolderId())).collect(Collectors.toList());
            if (!CollectionUtils.isEmpty(childFolderList)) {
                parent.setChildren(childFolderList);
                buildTree(childFolderList, allFolderList);
            }
        }
    }

    /**
     * 当前目录下报表数量
     *
     * @param folderIdMap 目录报表数量
     * @param folder      目录
     * @return
     */
    private int buildTotalReportCount(Map<Long, Integer> folderIdMap, Map<Long, List<FineReport>> authRelReportMap, Integer scene, Folder folder) {
        // 目录及子目录下报表总数量
        if (Objects.equals(BusinessSceneEnum.BACK_MANAGEMENT.getValue(), scene)) {
            return MapUtils.isNotEmpty(folderIdMap) && !Objects.isNull(folderIdMap.get(folder.getId()))
                    ? folderIdMap.get(folder.getId()) : 0;
        } else {
            return MapUtils.isNotEmpty(authRelReportMap) && !Objects.isNull(authRelReportMap.get(folder.getId()))
                    ? authRelReportMap.get(folder.getId()).size() : 0;
        }
    }

    /**
     * 累加报表数量
     *
     * @param folderIdMap      目录下报表数量
     * @param folderIdList     目录id列表
     * @param totalReportCount 总数量
     * @return
     */
    private int AccumulatedTotalReportCount(Map<Long, Integer> folderIdMap, Map<Long, List<FineReport>> authRelReportMap, Integer scene, List<Long> folderIdList, int totalReportCount) {
        if (CollectionUtils.isEmpty(folderIdList)) {
            return totalReportCount;
        }
        for (Long id : folderIdList) {
            if (BusinessSceneEnum.BACK_MANAGEMENT.getValue().equals(scene)
                    && !MapUtils.isEmpty(folderIdMap) && !Objects.isNull(folderIdMap.get(id))) {
                totalReportCount += folderIdMap.get(id);
            }
            if (BusinessSceneEnum.PORTAL.getValue().equals(scene)
                    && !MapUtils.isEmpty(authRelReportMap) && !Objects.isNull(authRelReportMap.get(id))) {
                totalReportCount += authRelReportMap.get(id).size();
            }
        }
        return totalReportCount;
    }

    /**
     * 更新帆软报表链接后缀
     *
     * @param saveReportUrlParamReq 链接后置配置入参
     */
    @Override
    public void saveReportUrlParam(SaveReportUrlParamReq saveReportUrlParamReq) {
        fineReportMapper.batchUpdateParam(saveReportUrlParamReq, saveReportUrlParamReq.getIds(), AppAuthContextHolder.getContext().getAuthoredUser().getUserId());
    }

    /**
     * 根据报表id查询具体的报表
     *
     * @param reportId 报表id
     * @param scene    场景
     * @return
     */
    @Override
    public FineReport queryFineReportById(Long reportId, Integer scene) {
        FineReport fineReport;
        // 管理后台直接查询报表信息
        if (BusinessSceneEnum.BACK_MANAGEMENT.getValue().equals(scene)) {
            fineReport = fineReportMapper.selectOne(new QueryWrapper<FineReport>().eq("id", reportId));
        } else {    // 首页查询的范围是当前用户 + 当前角色 + 当前组织的报表
            List<BizObjAuthRel> bizObjAuthRelList = bizObjAuthRelService.qryBizObjAuthRel(BizObjectTypeEnum.FINE_REPORT.getValue());
            if (CollectionUtils.isEmpty(bizObjAuthRelList)) {
                return null;
            }
            boolean existFlag = bizObjAuthRelList.stream().anyMatch(x -> String.valueOf(reportId).equals(x.getBizObjId()));
            if (!existFlag) {
                return null;
            }
            fineReport = fineReportMapper.selectOne(new QueryWrapper<FineReport>().eq("id", reportId));
        }
        return fineReport;
    }

    public boolean exists(Wrapper<FineReport> queryWrapper) {
        Long count = fineReportMapper.selectCount(queryWrapper);
        return null != count && count > 0;
    }
}