package com.digiwin.athena.semc.service.portal.impl;

import cn.hutool.core.util.StrUtil;
import com.alibaba.excel.EasyExcelFactory;
import com.alibaba.excel.ExcelReader;
import com.alibaba.excel.event.SyncReadListener;
import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.digiwin.athena.appcore.auth.AppAuthContextHolder;
import com.digiwin.athena.semc.common.Constants;
import com.digiwin.athena.semc.common.PageInfoResp;
import com.digiwin.athena.semc.common.enums.FolderTypeEnum;
import com.digiwin.athena.semc.dto.portal.QueryImportRecordReq;
import com.digiwin.athena.semc.entity.common.ImportRecord;
import com.digiwin.athena.semc.env.EnvProperties;
import com.digiwin.athena.semc.mapper.common.ImportRecordMapper;
import com.digiwin.athena.semc.service.portal.IImportRecordService;
import com.digiwin.athena.semc.util.DateUtils;
import com.digiwin.athena.semc.vo.common.ImportRecordResp;
import com.digiwin.athena.semc.vo.portal.ImportExcelVo;
import com.digiwin.athena.semc.vo.portal.ImportLabelExcelVo;
import com.digiwin.dap.middleware.dmc.DMC;
import com.digiwin.dap.middleware.dmc.DMCBuilder;
import com.digiwin.dap.middleware.dmc.model.FileInfo;
import com.digiwin.dap.middleware.dmc.model.ShareInfo;
import io.github.linpeilie.Converter;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import javax.annotation.Resource;
import java.io.ByteArrayInputStream;
import java.io.File;
import java.io.InputStream;
import java.nio.file.Files;
import java.util.List;
import java.util.stream.Collectors;

/**
 * @description: 导入记录
 * @createDate: 2023/8/15
 * @author: sungqz
 */
@Slf4j
@Service
public class ImportReportServiceImpl extends ServiceImpl<ImportRecordMapper, ImportRecord> implements IImportRecordService {

    @Resource
    private EnvProperties envProperties;

    @Autowired
    ImportRecordMapper importRecordMapper;

    @Resource
    private Converter converter;
    /**
     * 文件预览地址
     */
    public static final String DMC_FILE_PREVIEW_PATH = "/api/dmc/v2/file/intelligententry/preview/";

    /**
     * 记录导入历史
     *
     * @param importRecord 导入记录对象
     */
    @Override
    public void insertImportRecord(ImportRecord importRecord) {
        this.save(importRecord);
    }

    /**
     * 读取导入报表的内容
     *
     * @param fileId 文件id
     * @return
     */
    @Override
    public List<ImportExcelVo> readExcel(String fileId) {
        DMC dmcInstance = buildDmc();
        List<ImportExcelVo> excelVoList;
        try {
            byte[] byteArray = dmcInstance.download(fileId);
            ByteArrayInputStream input = new ByteArrayInputStream(byteArray);
            SyncReadListener excelListener = new SyncReadListener();
            // 读取sheet
            ExcelReader excelReader = EasyExcelFactory.read(input).build();
            excelReader.read(EasyExcelFactory.readSheet(0).head(ImportExcelVo.class).headRowNumber(1).registerReadListener(excelListener).build());
            // 这里千万别忘记关闭，读的
            excelReader.finish();
            excelVoList = doReadSync(excelListener);
        } catch (Exception e) {
            log.error("FineReportServiceImpl read excel occur error", e);
            throw new RuntimeException(e);
        }
        return excelVoList;
    }

    /**
     * 下载文件
     *
     * @param fileId 文件id
     * @return
     */
    @Override
    public List<ImportLabelExcelVo> readLabelExcel(String fileId){
        DMC dmcInstance = buildDmc();
        List<ImportLabelExcelVo> excelVoList;
        try {
            byte[] byteArray = dmcInstance.download(fileId);
            ByteArrayInputStream input = new ByteArrayInputStream(byteArray);
            SyncReadListener excelListener = new SyncReadListener();
            // 读取sheet
            ExcelReader excelReader = EasyExcelFactory.read(input).build();
            excelReader.read(EasyExcelFactory.readSheet(0).head(ImportLabelExcelVo.class).headRowNumber(2).registerReadListener(excelListener).build());
            // 这里千万别忘记关闭，读的
            excelReader.finish();
            excelVoList = doReadSync(excelListener);
        } catch (Exception e) {
            log.error("read label excel occur error", e);
            throw new RuntimeException(e);
        }
        return excelVoList;
    }

    /**
     * 获取dmc实例
     *
     * @return
     */
    private DMC buildDmc() {
        return DMCBuilder.create().build(envProperties.getDmcUri(), Constants.DMC_USER_NAME, Constants.DMC_USER, envProperties.getSemcAppId());
    }

    /**
     * 获取数据
     *
     * @param listener 监听
     * @param <T>
     * @return
     */
    @SuppressWarnings("unchecked")
    private static <T> List<T> doReadSync(SyncReadListener listener) {
        return (List<T>) listener.getList();
    }

    /**
     * 上传文件
     *
     * @param file 文件
     * @return
     */
    @Override
    public ShareInfo uploadExcel(File file) {
        try (InputStream input = Files.newInputStream(file.toPath())) {
            byte[] bytes = new byte[input.available()];
            int readLine = input.read(bytes);
            if (readLine <= 0) {
                log.info("upload excel file is empty");
            }
            FileInfo fileInfo = new FileInfo();
            fileInfo.setFileName(file.getName());
            fileInfo.setExtension("xlsx");
            return uploadAndShareFile(bytes, fileInfo);
        } catch (Exception e) {
            log.error("upload excel occur error", e);
            throw new RuntimeException(e);
        }
    }

    /**
     * 调用dmc上传并分享文件
     *
     * @param bytes    文件流
     * @param fileInfo 文件信息
     * @return
     * @throws Exception
     */
    private ShareInfo uploadAndShareFile(byte[] bytes, FileInfo fileInfo) throws Exception {
        DMC instance = buildDmc();
        FileInfo uploadFile = instance.upload(bytes, fileInfo);
        return instance.share(uploadFile.getId());
    }

    /**
     * 查询导入操作人列表
     *
     * @return
     */
    @Override
    public List<ImportRecord> queryOperatorList() {
        try {
            QueryWrapper<ImportRecord> condition = new QueryWrapper<ImportRecord>()
                    .eq("obj_type", FolderTypeEnum.FINE_REPORT.getValue()).eq("tenant_id", AppAuthContextHolder.getContext().getAuthoredUser().getTenantId());
            return importRecordMapper.selectList(condition);
        } catch (Exception e) {
            log.error("ImportReportServiceImpl query operator list occur error", e);
            throw new RuntimeException(e);
        }
    }

    /**
     * 查询导入记录列表
     *
     * @param queryImportRecordReq 查询参数
     * @param sceneFlag 业务场景标识
     * @return
     */
    @Override
    public PageInfoResp<ImportRecordResp> queryImportRecord(QueryImportRecordReq queryImportRecordReq, Integer sceneFlag) {
        try {
            QueryWrapper<ImportRecord> condition = new QueryWrapper<>();
            if (CollectionUtils.isNotEmpty(queryImportRecordReq.getUserIdList())) {
                condition.in("create_user_id", queryImportRecordReq.getUserIdList());
            }
            if (StringUtils.isBlank(queryImportRecordReq.getStartTime()) || StringUtils.isBlank(queryImportRecordReq.getEndTime())) {
                String currentDate = DateUtils.getNowTime(DateUtils.DATE_TIME_NORMAL_FORMATTER);
                condition.ge("create_time", DateUtils.addDate(currentDate, DateUtils.DATE_TIME_NORMAL_FORMATTER, -1, DateUtils.YEAR));
                condition.le("create_time", currentDate);
            } else {
                condition.ge("create_time", queryImportRecordReq.getStartTime());
                condition.le("create_time", queryImportRecordReq.getEndTime());
            }
            if(StrUtil.isNotBlank(queryImportRecordReq.getUserIdName())){
                condition.and(i -> i.like("create_user_name", queryImportRecordReq.getUserIdName()));
            }
            condition.eq("obj_type", sceneFlag).eq("tenant_id", AppAuthContextHolder.getContext().getAuthoredUser().getTenantId()).orderByDesc("create_time");
            PageInfoResp<ImportRecordResp> pageInfoResp = new PageInfoResp<>();
            Page<ImportRecord> pageResult = importRecordMapper.selectPage(new Page<>(queryImportRecordReq.getPageNum(), queryImportRecordReq.getPageSize()), condition);
            if (pageResult == null || CollectionUtils.isEmpty(pageResult.getRecords())) {
                return pageInfoResp;
            }
            // 拼接文件预览链接
            List<ImportRecordResp> respList = pageResult.getRecords().stream().map(x -> {
                ImportRecordResp resp = converter.convert(x, ImportRecordResp.class);
                resp.setFileUrl(envProperties.getDmcUri() + DMC_FILE_PREVIEW_PATH + x.getFileId().trim());
                return resp;
            }).collect(Collectors.toList());

            pageInfoResp.setTotalPages(Integer.parseInt(String.valueOf(pageResult.getPages())));
            pageInfoResp.setPageNo(queryImportRecordReq.getPageNum());
            pageInfoResp.setPageSize(queryImportRecordReq.getPageSize());
            pageInfoResp.setTotalRecords(Integer.parseInt(String.valueOf(pageResult.getTotal())));
            pageInfoResp.setList(respList);
            return pageInfoResp;
        } catch (Exception e) {
            log.error("ImportReportServiceImpl query import record occur error", e);
            throw new RuntimeException(e);
        }
    }
}