package com.digiwin.athena.semc.service.portal.impl;

import com.google.common.collect.Lists;

import com.alibaba.fastjson.JSONObject;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.digiwin.athena.appcore.auth.AppAuthContextHolder;
import com.digiwin.athena.appcore.auth.domain.AuthoredUser;
import com.digiwin.athena.semc.common.Constants;
import com.digiwin.athena.semc.common.ResponseBody;
import com.digiwin.athena.semc.common.enums.ApplicationTypeEnum;
import com.digiwin.athena.semc.common.enums.BizAuthTypeEnum;
import com.digiwin.athena.semc.dto.PageInfo;
import com.digiwin.athena.semc.dto.portal.KnowledgeApplicationListReq;
import com.digiwin.athena.semc.dto.portal.KnowledgeApplicationListResp;
import com.digiwin.athena.semc.dto.portal.KnowledgeApplicationNewDataReq;
import com.digiwin.athena.semc.entity.applink.AppLinkDTO;
import com.digiwin.athena.semc.entity.common.BizObjAuthRel;
import com.digiwin.athena.semc.entity.portal.KnowledgeApplicationRef;
import com.digiwin.athena.semc.mapper.common.BizObjAuthRelMapper;
import com.digiwin.athena.semc.mapper.portal.KnowledgeApplicationMapper;
import com.digiwin.athena.semc.proxy.ania.service.AniaService;
import com.digiwin.athena.semc.proxy.ania.service.model.KnowledgeAssistantDTO;
import com.digiwin.athena.semc.proxy.eoc.service.EocService;
import com.digiwin.athena.semc.proxy.eoc.service.model.UserDeptInfoDTO;
import com.digiwin.athena.semc.proxy.iam.service.IamService;
import com.digiwin.athena.semc.proxy.iam.service.model.OrgRoleUserDTO;
import com.digiwin.athena.semc.proxy.iam.service.model.UserDTO;
import com.digiwin.athena.semc.service.applink.AppLinkService;
import com.digiwin.athena.semc.service.portal.IknowledgeApplicationService;

import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.ObjectUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.function.Function;
import java.util.stream.Collectors;

import javax.annotation.Resource;

import lombok.extern.slf4j.Slf4j;

@Slf4j
@Service
public class KnowledgeApplicationServiceImpl extends ServiceImpl<KnowledgeApplicationMapper, KnowledgeApplicationRef> implements IknowledgeApplicationService {

    @Autowired
    private KnowledgeApplicationMapper knowledgeApplicationMapper;

    @Resource
    private BizObjAuthRelMapper bizObjAuthRelMapper;

    @Autowired
    private AniaService aniaService;

    @Autowired
    private IamService iamService;

    @Autowired
    private EocService eocService;

    @Autowired
    private AppLinkService appLinkService;

    @Override
    public ResponseBody getKnowledgeApplicationList(KnowledgeApplicationListReq req) {
        List<KnowledgeApplicationListResp> result = new ArrayList<>();
        List<KnowledgeApplicationRef> list = knowledgeApplicationMapper.selectByPage(req.getIsEnable()
        );
        Integer total = 0;

        if (CollectionUtils.isEmpty(list)) {
            return ResponseBody.getInstance(result, PageInfo.getPageInfo(req.getPageNum(), req.getPageSize(), total));
        }

        List<KnowledgeAssistantDTO> kList = aniaService.getKnowledgeAssistants(req.getTenantId());
        Map<String, KnowledgeAssistantDTO> xx = new HashMap<>();
        if (CollectionUtils.isNotEmpty(kList)) {
            for (KnowledgeAssistantDTO knowledgeAssistantDTO : kList) {
                xx.put(knowledgeAssistantDTO.getAssistantCode(), knowledgeAssistantDTO);
            }
        }
        Map<String, AppLinkDTO> appMap = getAppMap();

        list = list.stream().filter(d -> {
            if (StringUtils.isBlank(req.getApplication())) {
                return true;
            }else {
                boolean nameType ;
                AppLinkDTO appLinkDTO = appMap.get(""+d.getApplicationType()+"@@@" + d.getApplicationId());
                if(Objects.isNull(appLinkDTO) || StringUtils.isBlank(appLinkDTO.getName())){
                    nameType =  false;
                }else {
                    String nameUp = appLinkDTO.getName().toUpperCase();
                    String requestUp = req.getApplication().toUpperCase();
                    nameType = nameUp.contains(requestUp);
                }
                String idUp = d.getApplicationId().toUpperCase();
                String requestUp = req.getApplication().toUpperCase();

                boolean idType = idUp.contains(requestUp);
                return nameType || idType;

            }
        }).filter(d -> {
            if (StringUtils.isBlank(req.getKnowledge())) {
                return true;
            } else {
                boolean nameType;
                KnowledgeAssistantDTO assistantDTO = xx.get(d.getKnowledgeId());
                if (Objects.isNull(assistantDTO) || StringUtils.isBlank(assistantDTO.getName())) {
                    nameType = false;
                }else {
                    String nameUp = assistantDTO.getName().toUpperCase();
                    String requestUp = req.getKnowledge().toUpperCase();
                    nameType = nameUp.contains(requestUp);
                }
                String idUp = d.getKnowledgeId().toUpperCase();
                String requestUp = req.getKnowledge().toUpperCase();
                boolean idType = idUp.contains(requestUp);
                return nameType || idType;
            }

        }).collect(Collectors.toList());

        total = list.size();

        int fromIndex = (req.getPageNum() - 1) * req.getPageSize();
        if (fromIndex >= list.size() || fromIndex < 0) {
            return ResponseBody.getInstance(result, PageInfo.getPageInfo(req.getPageNum(), req.getPageSize(), total));
        }
        int toIndex = fromIndex + req.getPageSize();
        if (toIndex > list.size()) {
            toIndex = list.size();
        }
        list = list.subList(fromIndex, toIndex);


        for (KnowledgeApplicationRef knowledgeApplicationRef : list) {
            KnowledgeApplicationListResp resp = new KnowledgeApplicationListResp();
            resp.setApplicationId(knowledgeApplicationRef.getApplicationId());
            resp.setAppPrimaryId(knowledgeApplicationRef.getAppPrimaryId());
            resp.setKnowledgeId(knowledgeApplicationRef.getKnowledgeId());
            resp.setApplicationSource(knowledgeApplicationRef.getApplicationSource());
            resp.setApplicationType(knowledgeApplicationRef.getApplicationType());
            String applicationName = "";
            AppLinkDTO appLinkDTO = appMap.get("" + knowledgeApplicationRef.getApplicationType() + "@@@" + knowledgeApplicationRef.getApplicationId());
            if (Objects.nonNull(appLinkDTO)) {
                applicationName = appLinkDTO.getName();
            }
            resp.setApplicationName(applicationName);
            String knowledgeName = "";
            KnowledgeAssistantDTO assistantDTO = xx.get(knowledgeApplicationRef.getKnowledgeId());
            if (Objects.nonNull(assistantDTO)) {
                knowledgeName = assistantDTO.getName();
            }
            resp.setKnowledgeName(knowledgeName);
            resp.setId(knowledgeApplicationRef.getId());
            resp.setIsEnable(knowledgeApplicationRef.getIsEnable());
            resp.setKnowledgeDesc(knowledgeName + "（" + knowledgeApplicationRef.getKnowledgeId() + "）");
            resp.setApplicationDes(applicationName + "（" + knowledgeApplicationRef.getApplicationId() + "）");
            resp.setModifyUserDesc(knowledgeApplicationRef.getModifyUserName() + "（" + knowledgeApplicationRef.getModifyUserId() + "）");
            resp.setCreateUserId(knowledgeApplicationRef.getCreateUserId());
            resp.setCreateTime(knowledgeApplicationRef.getCreateTime());
            resp.setModifyTime(knowledgeApplicationRef.getModifyTime());
            resp.setModifyUserId(knowledgeApplicationRef.getModifyUserId());
            resp.setModifyUserName(knowledgeApplicationRef.getModifyUserName());
            resp.setCreateUserName(knowledgeApplicationRef.getCreateUserName());
            result.add(resp);
        }

        return ResponseBody.getInstance(result, PageInfo.getPageInfo(req.getPageNum(), req.getPageSize(), total));
    }

    @Override
    public void batchInsertRecord(List<KnowledgeApplicationNewDataReq> reqList) {
        List<KnowledgeApplicationRef> result = new ArrayList<>();

        for (KnowledgeApplicationNewDataReq knowledgeApplicationNewDataReq : reqList) {
            KnowledgeApplicationRef ref = new KnowledgeApplicationRef();
            ref.setApplicationSource(knowledgeApplicationNewDataReq.getApplicationSource());
            ref.setApplicationType(knowledgeApplicationNewDataReq.getApplicationType());
            ref.setApplicationId(knowledgeApplicationNewDataReq.getApplicationId());
            ref.setAppPrimaryId(knowledgeApplicationNewDataReq.getAppPrimaryId());
            ref.setKnowledgeId(knowledgeApplicationNewDataReq.getKnowledgeId());
            ref.setIsEnable(knowledgeApplicationNewDataReq.getIsEnable());
            ref.setTenantId(knowledgeApplicationNewDataReq.getTenantId());
            ref.setModifyUserName(knowledgeApplicationNewDataReq.getModifyUserName());
            ref.setCreateUserName(knowledgeApplicationNewDataReq.getCreateUserName());
            result.add(ref);
        }
        this.saveBatch(result);

        // 启用的知识库
        List<KnowledgeApplicationRef> enableList = result.stream().filter(x -> Constants.VALID_STATUS_ENABLE.equals(x.getIsEnable())).collect(Collectors.toList());
        if (CollectionUtils.isNotEmpty(enableList)) {
            log.info("batchInsertRecord | batch insert record authorize user app, param:{}, actionType:{}", JSONObject.toJSONString(enableList), Constants.IamAuthorizeTypeEnum.INSERT.getVal());
            // 给用户授权应用
            authorizeUserApp(enableList, Constants.IamAuthorizeTypeEnum.INSERT.getVal());
        }
    }

    /**
     * 批量给用户授权（取消授权）应用权限
     *
     * @param result     应用列表
     * @param actionType 操作行为，1-删除授权，0-新增授权
     */
    @Override
    public void authorizeUserApp(List<KnowledgeApplicationRef> result, Integer actionType) {
        // 查询应用关联的用户权限
        List<BizObjAuthRel> authRelList = bizObjAuthRelMapper.queryAuthListByAppId(result);
        log.info("KnowledgeApplicationServiceImpl | query auth list by appId, param:{}, result:{}", JSONObject.toJSONString(result), JSONObject.toJSONString(authRelList));
        if (CollectionUtils.isEmpty(authRelList)) {
            return;
        }

        // 查询所有组织或角色下用户列表
        List<OrgRoleUserDTO> orgUserList = Lists.newArrayList();
        List<OrgRoleUserDTO> roleUserList = Lists.newArrayList();
        List<UserDeptInfoDTO> deptUserList = Lists.newArrayList();
        List<Long> orgSidList = authRelList.stream().filter(x -> BizAuthTypeEnum.ORG.getValue().equals(x.getAuthType())).map(BizObjAuthRel::getAuthId).distinct().collect(Collectors.toList());
        if (CollectionUtils.isNotEmpty(orgSidList)) {
            orgUserList = iamService.queryUserByAllOrgSidList(orgSidList);
            log.info("KnowledgeApplicationServiceImpl | query org user by orgSidList, param:{}, result:{}", JSONObject.toJSONString(orgSidList), JSONObject.toJSONString(orgUserList));
        }
        List<Long> roleSidList = authRelList.stream().filter(x -> BizAuthTypeEnum.ROLE.getValue().equals(x.getAuthType())).map(BizObjAuthRel::getAuthId).distinct().collect(Collectors.toList());
        if (CollectionUtils.isNotEmpty(roleSidList)) {
            roleUserList = iamService.queryUserByAllRoleSidList(roleSidList);
        }
        List<Long> deptSidList = authRelList.stream().filter(x -> BizAuthTypeEnum.DEPARTMENT.getValue().equals(x.getAuthType())).map(BizObjAuthRel::getAuthId).distinct().collect(Collectors.toList());
        if (CollectionUtils.isNotEmpty(deptSidList)) {
            deptUserList = eocService.queryUserByDeptSidList(deptSidList);
            log.info("KnowledgeApplicationServiceImpl | query user by deptSidList, param:{}, result:{}", JSONObject.toJSONString(deptSidList), JSONObject.toJSONString(deptUserList));
        }

        // 给用户授权（或取消授权）应用
        Map<String, List<BizObjAuthRel>> appIdMap = authRelList.stream().collect(Collectors.groupingBy(x -> x.getBizObjSubType() + "-" + x.getBizObjId()));
        for (KnowledgeApplicationRef applicationRef : result) {
            List<BizObjAuthRel> appAuthRelList = appIdMap.get(applicationRef.getApplicationType() + "-" + applicationRef.getAppPrimaryId());
            if (CollectionUtils.isEmpty(appAuthRelList)) {
                continue;
            }

            List<Long> allUserSidList = Lists.newArrayList();
            // 应用关联的组织下用户sid
            List<Long> currentOrgSidList = appAuthRelList.stream().filter(x -> BizAuthTypeEnum.ORG.getValue().equals(x.getAuthType())).map(BizObjAuthRel::getAuthId).distinct().collect(Collectors.toList());
            orgUserList.forEach(x -> {
                if (currentOrgSidList.contains(x.getOrgSid())) {
                    List<Long> userSidList = x.getUsers().stream().map(UserDTO::getSid).collect(Collectors.toList());
                    allUserSidList.addAll(userSidList);
                }
            });

            // 应用关联的角色下用户sid
            List<Long> currentRoleSidList = appAuthRelList.stream().filter(x -> BizAuthTypeEnum.ROLE.getValue().equals(x.getAuthType())).map(BizObjAuthRel::getAuthId).distinct().collect(Collectors.toList());
            roleUserList.forEach(x -> {
                if (currentRoleSidList.contains(x.getRoleSid())) {
                    List<Long> userSidList = x.getUsers().stream().map(UserDTO::getSid).collect(Collectors.toList());
                    allUserSidList.addAll(userSidList);
                }
            });

            // 应用关联的部门下用户sid
            List<Long> currentDeptSidList = appAuthRelList.stream().filter(x -> BizAuthTypeEnum.DEPARTMENT.getValue().equals(x.getAuthType())).map(BizObjAuthRel::getAuthId).distinct().collect(Collectors.toList());
            deptUserList.forEach(x -> {
                if (currentDeptSidList.contains(x.getDeptSid())) {
                    List<Long> userSidList = Lists.newArrayList();
                    x.getEmps().forEach(empInfo -> {
                        if (ObjectUtils.isNotEmpty(empInfo.getEmpUserSid())) {
                            userSidList.add(empInfo.getEmpUserSid());
                        }
                    });
                    allUserSidList.addAll(userSidList);
                }
            });

            // 应用关联的用户sid
            List<Long> userSidList = appAuthRelList.stream().filter(x -> BizAuthTypeEnum.USER.getValue().equals(x.getAuthType())).map(BizObjAuthRel::getAuthId).collect(Collectors.toList());
            allUserSidList.addAll(userSidList);

            // cac接口限制用户列表最大为100个
            List<List<Long>> partList = Lists.partition(allUserSidList, 100);
            for (List<Long> tempList : partList) {
                log.info("KnowledgeApplicationServiceImpl | start batch user authorize app, appId:{}, userSid:{}, actionType:{}", applicationRef.getApplicationId(), JSONObject.toJSONString(tempList), actionType);
                iamService.batchUserAuthorizeApp(applicationRef.getKnowledgeId(), tempList, actionType);
            }
        }
    }

    private Map<String, AppLinkDTO> getAppMap() {
        AuthoredUser user = AppAuthContextHolder.getContext().getAuthoredUser();
        List<AppLinkDTO> appLinkListDTO = appLinkService.queryManageListSync(user);
        appLinkListDTO = filterRepeatApp(appLinkListDTO);
        Map<String, AppLinkDTO> result = new HashMap<>();
        if (CollectionUtils.isNotEmpty(appLinkListDTO)) {
            for (AppLinkDTO appLinkDTO : appLinkListDTO) {
                result.put("" + appLinkDTO.getType() + "@@@" + appLinkDTO.getCode(), appLinkDTO);
            }
        }
        return result;
    }

    private List<AppLinkDTO> filterRepeatApp(List<AppLinkDTO> appLinkListDT) {
        List<AppLinkDTO> appLinkDTOList = new ArrayList<>();
        //预设表里的数据
        List<AppLinkDTO> matchedLinkList = appLinkListDT.stream().filter(x -> x.getDataSource().equals(ApplicationTypeEnum.PRESET_APPLICATION.getType())).collect(Collectors.toList());
        Map<String, AppLinkDTO> preListMap = matchedLinkList.stream().collect(Collectors.toMap(AppLinkDTO::getAppId, Function.identity(), (a, b) -> a));
        //IAM 数据
        List<AppLinkDTO> iamLinkList = appLinkListDT.stream().filter(x -> x.getDataSource().equals(ApplicationTypeEnum.IAM_APPLICATION.getType())).collect(Collectors.toList());
        Map<String, AppLinkDTO> iamListMap = iamLinkList.stream().collect(Collectors.toMap(AppLinkDTO::getPrimaryId, Function.identity(), (a, b) -> a));
        for (AppLinkDTO appLink : appLinkListDT) {
            //预设表数据，且预设表的appid 和 cs、bs、iam返回的应用code 一致，则需要过滤改应用
            AppLinkDTO iamData = iamListMap.get(appLink.getAppId());
            if (appLink.getDataSource().equals(ApplicationTypeEnum.PRESET_APPLICATION.getType()) && null != iamData) {
                continue;
            }
            AppLinkDTO preData = preListMap.get(appLink.getPrimaryId());
            //非预设表数据，且预设表的appid 和 cs、bs、iam返回的应用code 一致，则应用名称取预设表的名称
            if (!appLink.getDataSource().equals(ApplicationTypeEnum.PRESET_APPLICATION.getType()) && null != preData) {
                appLink.setName(preData.getName());
            }

            if (StringUtils.isNoneBlank(appLink.getCode())) {
                appLinkDTOList.add(appLink);
            }
        }
        return appLinkDTOList;
    }
}