package com.digiwin.athena.semc.service.portal.impl;

import com.google.common.collect.Lists;

import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.digiwin.athena.appcore.auth.AppAuthContextHolder;
import com.digiwin.athena.semc.common.Constants;
import com.digiwin.athena.semc.dto.portal.LabelResp;
import com.digiwin.athena.semc.dto.portal.QueryLabelReq;
import com.digiwin.athena.semc.entity.portal.Label;
import com.digiwin.athena.semc.mapper.portal.LabelMapper;
import com.digiwin.athena.semc.service.portal.ILabelService;

import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.i18n.LocaleContextHolder;
import org.springframework.stereotype.Service;

import java.util.List;

import javax.annotation.Resource;

import io.github.linpeilie.Converter;

@Service
public class LabelServiceImpl extends ServiceImpl<LabelMapper, Label> implements ILabelService {

    @Autowired
    LabelMapper labelMapper;

    @Resource
    private Converter converter;

    /**
     * 查询组件列表
     *
     * @param queryLabelReq 请求实体
     * @return 标签列表
     */
    @Override
    public List<LabelResp> queryLabelList(QueryLabelReq queryLabelReq) {
        List<LabelResp> labelRespList = Lists.newArrayList();
        QueryWrapper<Label> condition = new QueryWrapper<>();
        if (CollectionUtils.isNotEmpty(queryLabelReq.getLabelIdList())) {
            condition.in("id", queryLabelReq.getLabelIdList());
        }
        if (CollectionUtils.isNotEmpty(queryLabelReq.getSystemFlag())) {
            condition.in("system_flag", queryLabelReq.getSystemFlag());
        }
        if (StringUtils.isNotBlank(queryLabelReq.getLabelKey())) {
            condition.eq("label_key", queryLabelReq.getLabelKey());
        }
        if (queryLabelReq.getValidStatus() != null) {
            condition.eq("valid_status", queryLabelReq.getValidStatus());
        }
        condition.orderByAsc("system_flag");
        List<Label> labelList = labelMapper.selectList(condition);
        if (CollectionUtils.isEmpty(labelList)) {
            return labelRespList;
        }
        // 根据语言环境返回对应的title,默认返回繁体
        final String locale = LocaleContextHolder.getLocale().toString();
        labelList.forEach(x -> {
            LabelResp labelResp = converter.convert(x, LabelResp.class);
            if (StringUtils.isBlank(locale) || locale.equals(Constants.ZH_TW_LOCALE)) {
                labelResp.setTitle(x.getTitleTw());
            } else if (locale.equals(Constants.ZH_CN_LOCALE)) {
                labelResp.setTitle(x.getTitleCn());
            } else {
                labelResp.setTitle(x.getTitleUs());
            }
            labelResp.setTitle(StringUtils.isBlank(labelResp.getTitle()) ? x.getTitleTw() : labelResp.getTitle());
            labelRespList.add(labelResp);
        });
        return labelRespList;
    }

    /**
     * 查询单个组件信息
     *
     * @param labelId 组件id
     * @return
     */
    @Override
    public Label queryLabelInfo(long labelId) {
        QueryWrapper<Label> queryWrapper = new QueryWrapper<>();
        queryWrapper.eq("id", labelId);
        return labelMapper.selectOne(queryWrapper);
    }

    /**
     * 修改生效状态
     *
     * @param labelId 组件id
     * @param status  状态
     * @return
     */
    @Override
    public int updateLabelStatus(Long labelId, Integer status) {
        return labelMapper.updateLabelStatus(labelId, status, AppAuthContextHolder.getContext().getAuthoredUser().getUserId());
    }
}