package com.digiwin.athena.semc.service.portal.impl;

import cn.hutool.core.util.ObjUtil;
import com.digiwin.athena.appcore.auth.AppAuthContextHolder;
import com.digiwin.athena.semc.common.Constants;
import com.digiwin.athena.semc.common.enums.ApplicationTypeEnum;
import com.digiwin.athena.semc.common.enums.EAIServiceNameEnum;
import com.digiwin.athena.semc.dto.portal.QueryGroupReq;
import com.digiwin.athena.semc.entity.applink.AppLinkDTO;
import com.digiwin.athena.semc.entity.portal.LabelSystemData;
import com.digiwin.athena.semc.env.EnvProperties;
import com.digiwin.athena.semc.proxy.athena.service.atmc.AtmcService;
import com.digiwin.athena.semc.proxy.athena.service.atmc.model.GroupNameLangDTO;
import com.digiwin.athena.semc.proxy.athena.service.atmc.model.GroupRuleDTO;
import com.digiwin.athena.semc.proxy.athena.service.atmc.model.GroupValueDTO;
import com.digiwin.athena.semc.proxy.esp.service.ESPService;
import com.digiwin.athena.semc.proxy.tripartite.service.TripartiteService;
import com.digiwin.athena.semc.service.applink.AppLinkService;
import com.digiwin.athena.semc.service.portal.LabelSystemBusinessTodoService;
import com.digiwin.athena.semc.util.Utils;
import com.digiwin.athena.semc.vo.portal.GroupRuleResp;
import com.digiwin.athena.semc.vo.portal.GroupRuleValueResp;
import com.google.common.collect.Lists;
import lombok.extern.slf4j.Slf4j;
import net.sf.json.JSONArray;
import net.sf.json.JSONObject;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.ObjectUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.i18n.LocaleContextHolder;
import org.springframework.stereotype.Service;

import javax.annotation.Resource;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.stream.Collectors;

/**
 * 业务待办处理类
 *
 * @author sungqz
 * @since 2024-07-31
 */
@Slf4j
@Service
public class LabelSystemBusinessTodoServiceImpl implements LabelSystemBusinessTodoService {

    @Resource
    private AtmcService atmcService;

    @Resource
    private ESPService espService;

    @Resource
    private TripartiteService tripartiteService;

    @Autowired
    private AppLinkService appLinkService;

    @Resource
    private EnvProperties envProperties;

    /**
     * 查询雅典娜智驱平台分组条件或分组条件值
     *
     * @param queryGroupReq 入参
     * @return 返回
     */
    @Override
    public List<GroupRuleResp> queryAgileGroupList(QueryGroupReq queryGroupReq) {
        List<GroupRuleResp> groupRuleRespList = Lists.newArrayList();
        // 调用小AI接口查询分组条件
        List<GroupRuleDTO> groupRuleList;
        if (StringUtils.isBlank(queryGroupReq.getGroupCode())) {
            groupRuleList = atmcService.queryGroupRule();
        } else { // 调用小AI接口根据分组条件查询分组条件值
            groupRuleList = atmcService.queryGroupConditionRule(queryGroupReq.getGroupCode());
        }
        if (CollectionUtils.isEmpty(groupRuleList)) {
            return groupRuleRespList;
        }

        // 构建返回列表
        String lang = LocaleContextHolder.getLocale().toString();
        groupRuleRespList = groupRuleList.stream().map(x -> {
            GroupRuleResp groupRuleResp = new GroupRuleResp();
            groupRuleResp.setName(x.getName());
            groupRuleResp.setCode(x.getCode());
            if (Constants.ZH_CN_LOCALE.equals(lang)) {
                return groupRuleResp;
            }
            // 返回对应语系的名称
            if (ObjUtil.isNotNull(x.getLang()) && ObjUtil.isNotNull(x.getLang().getName())) {
                GroupNameLangDTO.LangDTO langDTO = x.getLang().getName();
                switch (lang) {
                    case Constants.ZH_TW_LOCALE:
                        groupRuleResp.setName(langDTO.getZh_TW());
                        break;
                    case Constants.EN_US_LOCALE:
                        groupRuleResp.setName(langDTO.getEn_US());
                        break;
                    default:
                        groupRuleResp.setName(langDTO.getZh_CN());
                        break;
                }
            }
            return groupRuleResp;
        }).collect(Collectors.toList());
        return groupRuleRespList;
    }

    /**
     * 查询三方应用的分组条件
     *
     * @param queryGroupReq 入参
     * @return 返回
     */
    @Override
    public List<GroupRuleResp> queryThirdGroupList(QueryGroupReq queryGroupReq) {
        List<GroupRuleResp> groupRuleRespList = Lists.newArrayList();
        JSONArray jsonArray;
        // 混合云模式
        if (Constants.DataModelEnum.MODEL_HYBRID_CLOUD.getVal().equals(queryGroupReq.getDataModel())) {
            // body
            Map<String, Object> param = new HashMap<>();
            param.put("locale", LocaleContextHolder.getLocale().toString());

            // 通过ESP查询三方系统的分组条件
            Map<String, Object> result = espService.queryByEsp(queryGroupReq.getMiddleSystemName(),
                    queryGroupReq.getMiddleSystemUid(), EAIServiceNameEnum.BUSINESS_TODO_GROUP.getServiceName(), null, param, "", null);
            log.info("query third group list for data model 0. param:{}, result:{}", param, result);
            if (ObjUtil.isNull(result) || Objects.isNull(result.get("group_list"))) {
                return groupRuleRespList;
            }
            jsonArray = JSONArray.fromObject(result.get("group_list"));
        } else {    // 非混合云模式
            Map<String, Object> result = tripartiteService.queryThirdBusinessTodoGroupList(queryGroupReq.getRestUrl());
            if (ObjUtil.isNull(result) || Objects.isNull(result.get("groupList"))) {
                return groupRuleRespList;
            }
            jsonArray = JSONArray.fromObject(result.get("groupList"));
        }

        // 构建返回对象
        for (Object item : jsonArray) {
            GroupRuleResp groupRuleResp = new GroupRuleResp();
            JSONObject jsonObject = (JSONObject) item;
            groupRuleResp.setCode(jsonObject.containsKey("code") ? jsonObject.getString("code") : "");
            groupRuleResp.setName(jsonObject.containsKey("name") ? jsonObject.getString("name") : "");
            if (jsonObject.containsKey("group_condition_list")) {
                JSONArray conditionList = jsonObject.getJSONArray("group_condition_list");
                List<GroupRuleResp> groupConditionList = Lists.newArrayList();
                for (Object o : conditionList) {
                    JSONObject jsonObjectItem = (JSONObject) o;
                    GroupRuleResp groupRuleCondition = new GroupRuleResp();
                    groupRuleCondition.setCode(jsonObjectItem.containsKey("code") ? jsonObjectItem.getString("code") : "");
                    groupRuleCondition.setName(jsonObjectItem.containsKey("name") ? jsonObjectItem.getString("name") : "");
                    groupConditionList.add(groupRuleCondition);
                }
                groupRuleResp.setGroupConditionList(groupConditionList);
            }
            groupRuleRespList.add(groupRuleResp);
        }
        return groupRuleRespList;
    }

    /**
     * 查询雅典娜智驱平台业务待办数
     *
     * @param clientId        客户端id
     * @param labelSystemData 入参
     * @return 返回
     */
    @Override
    public List<GroupRuleValueResp> queryAgileGroupValueList(String clientId, LabelSystemData labelSystemData) {
        List<GroupRuleValueResp> valueRespList = Lists.newArrayList();
        if (StringUtils.isBlank(labelSystemData.getCountCode()) || StringUtils.isBlank(labelSystemData.getCountContentCode())) {
            return valueRespList;
        }

        // 调用小AI接口获取业务待办数
        List<JSONObject> groupList = Lists.newArrayList();
        JSONObject jsonObject = new JSONObject();
        jsonObject.put("groupCode", labelSystemData.getCountCode());
        jsonObject.put("groupValue", labelSystemData.getCountContentCode());
        groupList.add(jsonObject);
        List<GroupValueDTO> groupValueList = atmcService.queryGroupTodoValue(clientId, groupList);
        if (CollectionUtils.isEmpty(groupValueList)) {
            return valueRespList;
        }

        // 构建返回列表
        valueRespList = groupValueList.stream().filter(x -> labelSystemData.getCountCode().equals(x.getGroupCode())
                && labelSystemData.getCountContentCode().equals(x.getGroupValue())).map(x -> {
            GroupRuleValueResp groupRuleResp = new GroupRuleValueResp();
            groupRuleResp.setGroupCode(x.getGroupCode());
            groupRuleResp.setGroupConditionCode(x.getGroupValue());
            groupRuleResp.setValue(x.getValue());
            groupRuleResp.setDataType(ApplicationTypeEnum.VIRTUAL_APPLICATION.getType());
            return groupRuleResp;
        }).collect(Collectors.toList());
        return valueRespList;
    }

    /**
     * 查询三方应用的分组条件待办数
     *
     * @param labelSystemData 入参
     * @return 返回
     */
    public GroupRuleValueResp queryThirdGroupValueList(LabelSystemData labelSystemData) {
        GroupRuleValueResp groupRuleValueResp = new GroupRuleValueResp();
        JSONObject jsonObject;
        // 混合云模式
        if (Constants.DataModelEnum.MODEL_HYBRID_CLOUD.getVal().equals(labelSystemData.getDataModel())) {
            // header
            Map<String, String> extHeader = new HashMap<>();
            extHeader.put("digi-userToken", Utils.getUserToken());
            extHeader.put("digi-appToken", envProperties.getAppToken());

            // body
            Map<String, Object> param = new HashMap<>();
            param.put("locale", LocaleContextHolder.getLocale().toString());
            // 如果应用只有业务待办类型，没有下级，则按条件值传参查询
            if (StringUtils.isBlank(labelSystemData.getCountContentCode())) {
                param.put("group_condition_code", labelSystemData.getCountCode());
            } else {
                param.put("group_code", labelSystemData.getCountCode());
                param.put("group_condition_code", labelSystemData.getCountContentCode());
            }

            // 通过ESP查询三方系统的分组条件
            Map<String, Object> result = espService.queryByEsp(labelSystemData.getMiddleSystemName(),
                    labelSystemData.getMiddleSystemUid(), EAIServiceNameEnum.BUSINESS_TODO_VALUE.getServiceName(), extHeader, param, "", null);
            log.info("query third group value list of data model 0 result. param:{}, result:{}", param, result);
            if (ObjUtil.isNull(result) || Objects.isNull(result.get("group_value"))) {
                return groupRuleValueResp;
            }
            jsonObject = JSONObject.fromObject(result.get("group_value"));
        } else { // 非混合云
            Map<String, Object> result = tripartiteService.queryThirdBusinessTodoValueList(labelSystemData, envProperties.getAppToken());
            if (ObjUtil.isNull(result) || Objects.isNull(result.get("groupValue"))) {
                return groupRuleValueResp;
            }
            jsonObject = JSONObject.fromObject(result.get("groupValue"));
        }

        // 构建返回对象
        groupRuleValueResp.setGroupCode(jsonObject.containsKey("group_code") ? jsonObject.getString("group_code") : "");
        groupRuleValueResp.setGroupName(jsonObject.containsKey("group_name") ? jsonObject.getString("group_name") : "");
        groupRuleValueResp.setGroupConditionCode(jsonObject.containsKey("group_condition_code") ? jsonObject.getString("group_condition_code") : "");
        groupRuleValueResp.setGroupConditionName(jsonObject.containsKey("group_condition_name") ? jsonObject.getString("group_condition_name") : "");
        groupRuleValueResp.setValue(jsonObject.containsKey("value") ? jsonObject.getInt("value") : null);
        groupRuleValueResp.setUrl(jsonObject.containsKey("url") ? jsonObject.getString("url") : null);
        // 返回应用信息，用于sso跳转
        List<AppLinkDTO> appLinkList = appLinkService.queryManageListSync(AppAuthContextHolder.getContext().getAuthoredUser());
        for (AppLinkDTO appLinkDTO : appLinkList) {
            if (!(labelSystemData.getDataType() + "-" + labelSystemData.getAppCode()).equals(appLinkDTO.getDataSource() + "-" + appLinkDTO.getPrimaryId())) {
                continue;
            }
            groupRuleValueResp.setAppName(appLinkDTO.getName());
            groupRuleValueResp.setAppToken(appLinkDTO.getAppToken());
            groupRuleValueResp.setCallBackUrl(appLinkDTO.getCallBackUrl());
            groupRuleValueResp.setApplicationAppId(appLinkDTO.getApplicationAppId());
            groupRuleValueResp.setProtocolType(appLinkDTO.getProtocolType());
            groupRuleValueResp.setSystemType(appLinkDTO.getSystemType());
            groupRuleValueResp.setDataType(labelSystemData.getDataType());
            // 构建跳转链接
            String url = appLinkService.parseJumpUrl(jsonObject.containsKey("url") ? jsonObject.getString("url") : "", appLinkDTO);
            groupRuleValueResp.setUrl(url);
            break;
        }
        return groupRuleValueResp;
    }
}