package com.digiwin.athena.semc.service.portal.impl;

import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.digiwin.athena.appcore.auth.AppAuthContextHolder;
import com.digiwin.athena.semc.common.Constants;
import com.digiwin.athena.semc.dto.portal.AddLogoReq;
import com.digiwin.athena.semc.dto.portal.LogoReq;
import com.digiwin.athena.semc.entity.common.PlatformConfig;
import com.digiwin.athena.semc.entity.portal.Logo;
import com.digiwin.athena.semc.env.EnvProperties;
import com.digiwin.athena.semc.mapper.common.PlatformConfigMapper;
import com.digiwin.athena.semc.mapper.portal.LogoMapper;
import com.digiwin.athena.semc.service.portal.ILogoService;

import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.i18n.LocaleContextHolder;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.util.ArrayList;
import java.util.List;
import java.util.stream.Collectors;

import javax.annotation.Resource;

import io.github.linpeilie.Converter;

@Service
public class LogoServiceImpl extends ServiceImpl<LogoMapper, Logo> implements ILogoService {

    @Autowired
    LogoMapper logoMapper;

    @Autowired
    PlatformConfigMapper platformConfigMapper;

    @Resource
    private EnvProperties envProperties;

    @Resource
    private Converter converter;

    /**
     * 添加首页logo图片
     *
     * @param req 请求实体
     * @return
     */
    @Override
    public int addLogoPic(AddLogoReq req) {
        Logo logo = new Logo();
        logo.setPicUrl(req.getPicUrl());
        return logoMapper.insert(logo);
    }

    /**
     * 根据语言类型返回logo图片地址
     *
     * @return logo图片列表
     */
    @Override
    public String queryLogoByCondition(Integer sceneFlag) {
        Logo logo = new Logo();
        QueryWrapper<Logo> condition = new QueryWrapper<>();
        condition.eq("scene_flag", sceneFlag);
        List<Logo> logoList = logoMapper.selectList(condition);
        if (CollectionUtils.isEmpty(logoList)) {
            return "";
        }
        // 如果是登录页，则不区分多租户，只有一条记录，直接返回
        if (sceneFlag.equals(Constants.SceneFlagEnum.LOGIN_BEFORE_PAGE.getFlag())) {
            logo = logoList.get(0);
        } else if (sceneFlag.equals(Constants.SceneFlagEnum.LOGIN_AFTER_PAGE.getFlag())) { // 如果是登录后首页，则返回对应租户的配置logo
            // 如果租户的值为空，则说明不区分多租户展示，只有一条记录，直接返回
            boolean noTenantFlag = logoList.stream().allMatch(x -> StringUtils.isBlank(x.getTenantId()));
            if (noTenantFlag) {
                logo = logoList.get(0);
            } else {
                String tenantId = AppAuthContextHolder.getContext().getAuthoredUser().getTenantId();
                List<Logo> tempLogoList = logoList.stream().filter(x -> tenantId.equals(x.getTenantId())).collect(Collectors.toList());
                logo = tempLogoList.get(0);
            }
        }
        // 根据语言环境返回对应的logo链接
        String locale = LocaleContextHolder.getLocale().toString();
        if (StringUtils.isBlank(locale) || locale.equals(Constants.ZH_CN_LOCALE)) {
            return logo.getPicUrl();
        } else if (locale.equals(Constants.ZH_TW_LOCALE)) {
            return logo.getPicUrlTw();
        } else {
            return logo.getPicUrlUs();
        }
    }

    @Override
    @Transactional
    public int saveSett(LogoReq req) {
        //先删除登录后首页logo信息和首页标题信息
        List<Logo> logoList =selectByParam(new ArrayList<>(),AppAuthContextHolder.getContext().getAuthoredUser().getTenantId(),getIsShowSet());
        if(CollectionUtils.isNotEmpty(logoList)){
            List<Long> logoIdList = logoList.stream().map(Logo::getId).collect(Collectors.toList());
            logoMapper.deleteBatchIds(logoIdList);
        }
        String  tenantId=AppAuthContextHolder.getContext().getAuthoredUser().getTenantId();
        //组装企业标识信息保存
        Logo homeLogo=converter.convert(req, Logo.class);
        homeLogo.setPicUrl(req.getLoginPicId());
        homeLogo.setPicUrlTw(req.getLoginPicIdTw());
        homeLogo.setPicUrlUs(req.getLoginPicIdUs());
        homeLogo.setTenantId(tenantId);
        homeLogo.setSceneFlag(Constants.SceneFlagEnum.LOGIN_AFTER_LOGO.getFlag());
        logoMapper.insert(homeLogo);
        //保存标题信息
        Logo titleLogo= converter.convert(req, Logo.class);
        titleLogo.setPicUrl(req.getTitlePicId());
        titleLogo.setPicUrlTw(req.getTitlePicIdTw());
        titleLogo.setPicUrlUs(req.getTitlePicIdUs());
        titleLogo.setTenantId(tenantId);
        titleLogo.setSceneFlag(Constants.SceneFlagEnum.LOGIN_AFTER_TITLE.getFlag());
        logoMapper.insert(titleLogo);
        return 0;
    }

    @Override
    public LogoReq getSett() {
        //查询企业标识配置信息
        String tenantId = AppAuthContextHolder.getContext().getAuthoredUser().getTenantId();
        //优先取租户的配置
        List<Logo> logoList = selectByParam(new ArrayList<>(), tenantId, getIsShowSet());
        LogoReq logoReq = new LogoReq();
        if (CollectionUtils.isEmpty(logoList)) {
            //查查询默认配置
            logoList= selectDefSett();
        }
        //默认配置和租户都没有数据
        if (CollectionUtils.isEmpty(logoList)) {
            return logoReq;
        }
        for(Logo logo:logoList){
            //组装返回
            String logoFileUrl = StringUtils.isEmpty(logo.getPicUrl()) ? "" : envProperties.getDmcUri() + Constants.DMC_FILE_PREVIEW_PATH + logo.getPicUrl();
            String loginPicUrlTw = StringUtils.isEmpty(logo.getPicUrlTw()) ? "" : envProperties.getDmcUri() + Constants.DMC_FILE_PREVIEW_PATH + logo.getPicUrlTw();
            String loginPicUrlUs = StringUtils.isEmpty(logo.getPicUrlUs()) ? "" : envProperties.getDmcUri() + Constants.DMC_FILE_PREVIEW_PATH + logo.getPicUrlUs();
            String logoPicId = logo.getPicUrl();
            String logoPicIdTw = logo.getPicUrlTw();
            String logoPicIdUs = logo.getPicUrlUs();

            //登录后logo信息、logo、跳转状态、官网等
            if (logo.getSceneFlag().equals(Constants.SceneFlagEnum.LOGIN_AFTER_LOGO.getFlag())) {
                //拼接logo地址
                logoReq.setLoginPicId(logoPicId);
                logoReq.setLoginPicIdTw(logoPicIdTw);
                logoReq.setLoginPicIdUs(logoPicIdUs);
                logoReq.setLoginPicUrl(logoFileUrl);
                logoReq.setLoginPicUrlTw(loginPicUrlTw);
                logoReq.setLoginPicUrlUs(loginPicUrlUs);
                logoReq.setJumpStatus(logo.getJumpStatus());
                logoReq.setJumpStatusTw(logo.getJumpStatusTw());
                logoReq.setJumpStatusUs(logo.getJumpStatusUs());
                logoReq.setWebSiteAddr(logo.getWebSiteAddr());
                logoReq.setWebSiteAddrTw(logo.getWebSiteAddrTw());
                logoReq.setWebSiteAddrUs(logo.getWebSiteAddrUs());
            }
            //首页标题logo信息、文字标题
            if (logo.getSceneFlag().equals(Constants.SceneFlagEnum.LOGIN_AFTER_TITLE.getFlag())) {
                //拼接标题logo地址
                logoReq.setTitlePicId(logoPicId);
                logoReq.setTitlePicIdTw(logoPicIdTw);
                logoReq.setTitlePicIdUs(logoPicIdUs);
                logoReq.setTitlePicUrl(logoFileUrl);
                logoReq.setTitlePicUrlTw(loginPicUrlTw);
                logoReq.setTitlePicUrlUs(loginPicUrlUs);
                logoReq.setTitle(logo.getTitle());
                logoReq.setTitleTw(logo.getTitleTw());
                logoReq.setTitleUs(logo.getTitleUs());
            }
        }

        return logoReq;
    }

    @Override
    public LogoReq querySettInfo() {
        // 根据语言环境返回对应的信息
        String locale = LocaleContextHolder.getLocale().toString();
        //查询企业标识配置信息
        LogoReq  logoReq=getSett();
        //简体
        if (StringUtils.isBlank(locale) || locale.equals(Constants.ZH_CN_LOCALE)) {
            return logoReq;
        }//繁体
        else if (locale.equals(Constants.ZH_TW_LOCALE)) {
            logoReq.setJumpStatus(logoReq.getJumpStatusTw());
            logoReq.setWebSiteAddr(logoReq.getWebSiteAddrTw());
            logoReq.setTitle(logoReq.getTitleTw());
            logoReq.setLoginPicUrl(logoReq.getLoginPicUrlTw());
            logoReq.setTitlePicUrl(logoReq.getTitlePicUrlTw());
        }//英文
        else {
            logoReq.setJumpStatus(logoReq.getJumpStatusUs());
            logoReq.setWebSiteAddr(logoReq.getWebSiteAddrUs());
            logoReq.setTitle(logoReq.getTitleUs());
            logoReq.setLoginPicUrl(logoReq.getLoginPicUrlUs());
            logoReq.setTitlePicUrl(logoReq.getTitlePicUrlUs());
        }
        return logoReq;
    }

    /**
     * 获取拥有企业标识设置权限的渠道
     * @return
     */
    public String getIsShowSet(){
        String channel=Constants.ChannelEnum.GROUND.getVal();
        QueryWrapper queryWrapper = new QueryWrapper<Logo>();
        queryWrapper.eq("is_show_set_log", Constants.SystemFlagEnum.DEFAULT);
        List<PlatformConfig> platformConfigList= platformConfigMapper.selectList(queryWrapper);
        if(CollectionUtils.isNotEmpty(platformConfigList)){
            channel=platformConfigList.get(0).getChannel();
        }
        return channel;
    }
    @Override
    public PlatformConfig getPlatformConfig(String channel) {
        if(StringUtils.isEmpty(channel)){
            return null;
        }
        QueryWrapper queryWrapper = new QueryWrapper<Logo>();
        queryWrapper.eq("channel", channel);
        PlatformConfig platformConfig= platformConfigMapper.selectOne(queryWrapper);
        return platformConfig;
    }

    @Override
    public LogoReq queryLoginLogo(String locale,String channel) {
        //查询登录前logo和标题
        List<Integer> sceneFlagList=new ArrayList<>();
        sceneFlagList.add(Constants.SceneFlagEnum.HOME_LOGIN_LOGO.getFlag());
        sceneFlagList.add(Constants.SceneFlagEnum.HOME_LOGIN_TITLE_LOGO.getFlag());
        if(StringUtils.isEmpty(channel)){
            channel=Constants.ChannelEnum.PLATFORM.getVal();
        }
        List<Logo> logoList = selectByParam(sceneFlagList, "",channel);
        LogoReq logoReq = new LogoReq();
        if (CollectionUtils.isEmpty(logoList)) {
            return logoReq;
        }
        //组装返回
        for (Logo logo : logoList) {
            String logoFileUrl = StringUtils.isEmpty(logo.getPicUrl())?"":envProperties.getDmcUri() + Constants.DMC_FILE_PREVIEW_PATH + logo.getPicUrl();
            String loginPicUrlTw = StringUtils.isEmpty(logo.getPicUrlTw())?"":envProperties.getDmcUri() + Constants.DMC_FILE_PREVIEW_PATH + logo.getPicUrlTw();
            String loginPicUrlUs = StringUtils.isEmpty(logo.getPicUrlUs())?"":envProperties.getDmcUri() + Constants.DMC_FILE_PREVIEW_PATH + logo.getPicUrlUs();
            //登录前logo信息、logo、跳转状态、官网等
            if (logo.getSceneFlag().equals(Constants.SceneFlagEnum.HOME_LOGIN_LOGO.getFlag())) {
                //拼接logo地址
                logoReq.setLoginPicUrl(logoFileUrl);
                logoReq.setLoginPicUrlTw(loginPicUrlTw);
                logoReq.setLoginPicUrlUs(loginPicUrlUs);
            }
            //登录前标题logo信息、文字标题
            if (logo.getSceneFlag().equals(Constants.SceneFlagEnum.HOME_LOGIN_TITLE_LOGO.getFlag())) {
                //拼接标题logo地址
                logoReq.setTitlePicUrl(logoFileUrl);
                logoReq.setTitlePicUrlTw(loginPicUrlTw);
                logoReq.setTitlePicUrlUs(loginPicUrlUs);
                logoReq.setTitle(logo.getTitle());
                logoReq.setTitleTw(logo.getTitleTw());
                logoReq.setTitleUs(logo.getTitleUs());
            }
        }
        // 根据语言环境返回对应的logo链接
        if (StringUtils.isNotEmpty(locale) && locale.equals(Constants.ZH_TW_LOCALE)) {
            logoReq.setTitlePicUrl(logoReq.getTitlePicUrlTw());
            logoReq.setLoginPicUrl(logoReq.getLoginPicUrlTw());
            logoReq.setTitle(logoReq.getTitleTw());
        } else if (StringUtils.isNotEmpty(locale) && locale.equals(Constants.EN_US_LOCALE)) {
            logoReq.setTitlePicUrl(logoReq.getTitlePicUrlUs());
            logoReq.setLoginPicUrl(logoReq.getLoginPicUrlUs());
            logoReq.setTitle(logoReq.getTitleUs());
        }
        return logoReq;
    }


    /**
     * 查查询默认配置
     * @return
     */
    public List<Logo> selectDefSett() {
        QueryWrapper queryWrapper = new QueryWrapper<Logo>();

        queryWrapper.eq("scene_flag", Constants.SceneFlagEnum.LOGIN_AFTER_TITLE.getFlag());
        queryWrapper.isNull("tenant_id");
        List<Logo> logoList = logoMapper.selectList(queryWrapper);
        return logoList;
    }

    /**
     * 查询1:登录后首页logo；2：首页网页标题logo 配置信息
     * @return
     */
    public List<Logo> selectByParam(List<Integer> sceneFlagList,String tenantId,String channel) {
        QueryWrapper queryWrapper = new QueryWrapper<Logo>();
        //为空则查询登录后首页logo和首页标题logo
        if(CollectionUtils.isEmpty(sceneFlagList)){
            sceneFlagList.add(Constants.SceneFlagEnum.LOGIN_AFTER_TITLE.getFlag());
            sceneFlagList.add(Constants.SceneFlagEnum.LOGIN_AFTER_LOGO.getFlag());
        }
        queryWrapper.in("scene_flag", sceneFlagList);
        if(StringUtils.isNotEmpty(tenantId)){
            queryWrapper.eq("tenant_id", tenantId);
        }
        if(StringUtils.isNotEmpty(channel)){
            queryWrapper.eq("channel", channel);
        }
        List<Logo> logoList = logoMapper.selectList(queryWrapper);
        return logoList;
    }
}