package com.digiwin.athena.semc.service.portal.impl;

import com.google.common.collect.Lists;

import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.digiwin.athena.appcore.auth.AppAuthContextHolder;
import com.digiwin.athena.appcore.auth.domain.AuthoredUser;
import com.digiwin.athena.semc.common.BizException;
import com.digiwin.athena.semc.common.Constants;
import com.digiwin.athena.semc.common.ErrorCodeConstant;
import com.digiwin.athena.semc.common.ResponseBody;
import com.digiwin.athena.semc.common.ResultPageBean;
import com.digiwin.athena.semc.common.enums.ApplicationTypeEnum;
import com.digiwin.athena.semc.common.enums.EAIServiceNameEnum;
import com.digiwin.athena.semc.common.enums.IfNotIntegerEnum;
import com.digiwin.athena.semc.common.enums.SourceEnum;
import com.digiwin.athena.semc.dto.PageInfo;
import com.digiwin.athena.semc.dto.portal.LabelSystemDataDto;
import com.digiwin.athena.semc.dto.portal.LabelSystemPreReq;
import com.digiwin.athena.semc.dto.portal.NewsPageQueryResp;
import com.digiwin.athena.semc.dto.portal.NoticePageQueryReq;
import com.digiwin.athena.semc.entity.portal.News;
import com.digiwin.athena.semc.mapper.portal.NewsMapper;
import com.digiwin.athena.semc.proxy.esp.service.ESPService;
import com.digiwin.athena.semc.proxy.tripartite.service.TripartiteService;
import com.digiwin.athena.semc.service.portal.LabelSystemDataService;
import com.digiwin.athena.semc.service.portal.NewsService;

import net.sf.json.JSONArray;

import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.stream.Collectors;

import javax.annotation.Resource;

/**
 * 公司新聞(News)表服务实现类
 *
 * @author sunyfa
 * @since 2022-12-06 13:38:38
 */
@Service
public class NewsServiceImpl extends ServiceImpl<NewsMapper, News> implements NewsService {

    @Autowired
    LabelSystemDataService labelSystemDataService;

    @Resource
    private ESPService espService;

    @Autowired
    TripartiteService tripartiteService;

    @Override
    public List<News> handleEspResult(Map<String, Object> result, NoticePageQueryReq.EaiSysInfo eaiSysInfo,
                                      NewsPageQueryResp newsPageQueryResp, String qryCondition) {
        Object espNewsList = result.get("news_list");

        QueryWrapper<News> queryWrapper = new QueryWrapper<>();
        queryWrapper.eq("app_name", eaiSysInfo.getAppName());
        queryWrapper.eq("news_user", AppAuthContextHolder.getContext().getAuthoredUser().getUserId());
        queryWrapper.eq("tenant_id", AppAuthContextHolder.getContext().getAuthoredUser().getTenantId());

        List<News> filteredNewsList = Lists.newArrayList();

        if (Objects.isNull(espNewsList)) {
            newsPageQueryResp.setRed(false);
            newsPageQueryResp.setTotalUnreadCount(0);
            newsPageQueryResp.setTotalResults(0);
        } else {
            Integer totalResults = result.get("total_results") == null ? 0 : Integer.parseInt(result.get("total_results").toString());
            newsPageQueryResp.setTotalResults(totalResults);

            // 查询当前系统的当前登录用户的所有新闻数据
            List<News> existNewsList = list(queryWrapper);

            Map<String, News> existsNewsMap = new HashMap<>();

            if (CollectionUtils.isNotEmpty(existNewsList)) {
                existsNewsMap = existNewsList.stream().collect(Collectors.toMap(News::getExternalNewsId, a -> a, (k1, k2) -> k1));
            }

            List<News> insertNewsList = Lists.newArrayList();

            JSONArray array = JSONArray.fromObject(espNewsList);
            for (Object json : array) {
                News news = com.alibaba.fastjson.JSONObject.parseObject(json.toString(), News.class);

                if (existsNewsMap.containsKey(news.getExternalNewsId())) {
                    BeanUtils.copyProperties(existsNewsMap.get(news.getExternalNewsId()), news);
                    news.setIsNew(IfNotIntegerEnum.NO.getValue());
                } else {
                    news.setNewsSource(SourceEnum.EXTERNAL.getSource());
                    news.setAppName(eaiSysInfo.getAppName());
                    news.setNewsUser(AppAuthContextHolder.getContext().getAuthoredUser().getUserId());
                    news.setIsRead(IfNotIntegerEnum.NO.getValue());
                    news.setIsNew(IfNotIntegerEnum.YES.getValue());

                    insertNewsList.add(news);
                }

                if (IfNotIntegerEnum.NO.getValue().equals(news.getIsRead())) {
                    newsPageQueryResp.setRed(true);
                    newsPageQueryResp.setTotalUnreadCount(newsPageQueryResp.getTotalUnreadCount() + 1);
                }

                if ((StringUtils.isEmpty(qryCondition))
                        || (StringUtils.isNotEmpty(qryCondition) && news.getNewsTitle().contains(qryCondition))) {
                    filteredNewsList.add(news);
                }

                news.setEaiSysName(eaiSysInfo.getEaiSysName());
                news.setEaiSysUid(eaiSysInfo.getEaiSysUid());
                news.setAppToken(eaiSysInfo.getAppToken());
                news.setAppName(eaiSysInfo.getAppName());
            }

            // 保存新增的数据
            if (CollectionUtils.isNotEmpty(insertNewsList)) {
                saveBatch(insertNewsList);
            }
        }

        return filteredNewsList;
    }

    /**
     * 查询通知公告数据
     *
     * @param noticePageQueryReq 请求参数
     * @return 返回对象
     */
    @Override
    public ResultPageBean pageQueryNews(LabelSystemPreReq noticePageQueryReq) {
        // 查询数据源信息和sso信息
        LabelSystemDataDto labelSystemDataDto = labelSystemDataService.getSystemDataSso(noticePageQueryReq.getId());
        if (null == labelSystemDataDto) {
            return ResultPageBean.bizException(BizException.getDefaultBizException(ErrorCodeConstant.PARAM_ILLEGAL_ERROR, "数据源不存在"));
        }

        Map<String, Object> result;
        // 混合云模式，走esp-互联网中台-三方地端服务
        if (Constants.DataModelEnum.MODEL_HYBRID_CLOUD.getVal().equals(labelSystemDataDto.getDataModel())) {
            // 通过ESP查询三方系统的待办数据,需要支持分页和发布时间倒序
            result = espService.queryByEsp(labelSystemDataDto.getMiddleSystemName(),
                    labelSystemDataDto.getMiddleSystemUid(), EAIServiceNameEnum.NEWS.getServiceName(), null, null, noticePageQueryReq.getQryCondition(),
                    PageInfo.getPageInfo(noticePageQueryReq.getPageNum(), noticePageQueryReq.getPageSize()));
        }
        // 非混合云模式,需要支持分页和发布时间倒序
        else {
            result = tripartiteService.selectDataPagePost(labelSystemDataDto.getRestUrl(), noticePageQueryReq, labelSystemDataDto.getAppToken());
        }

        // 处理返回结果
        NoticePageQueryReq.EaiSysInfo eaiSys = new NoticePageQueryReq.EaiSysInfo();
        eaiSys.setAppName(labelSystemDataDto.getAppCode());
        eaiSys.setAppToken(labelSystemDataDto.getAppToken());
        eaiSys.setUserBindFlag(labelSystemDataDto.getUserBindFlag());

        NewsPageQueryResp newsPageQueryResp = new NewsPageQueryResp();
        List<News> newsList = handleEspResult(result, eaiSys, newsPageQueryResp, noticePageQueryReq.getQryCondition());
        //String tenantId = AppAuthContextHolder.getContext().getAuthoredUser().getTenantId();
        newsList.forEach(x -> x.setNewsUrl(getNewsUrl(x,eaiSys,AppAuthContextHolder.getContext().getAuthoredUser(),labelSystemDataDto)));
        newsPageQueryResp.setNewsList(newsList);

        // 构建返回对象
        ResultPageBean resultPageBean = ResultPageBean.success(ResponseBody.getInstance(newsPageQueryResp, PageInfo.getPageInfo(
                noticePageQueryReq.getPageNum(), noticePageQueryReq.getPageSize(), newsPageQueryResp.getTotalResults())));
        return resultPageBean;
    }

    private String getNewsUrl(News news, NoticePageQueryReq.EaiSysInfo eaiSys, AuthoredUser user,LabelSystemDataDto labelSystemDataDto) {
        String tenantId = user.getTenantId();
        String newsUrl = null;
        if (labelSystemDataDto == null || 1 == labelSystemDataDto.getProtocolType()) {
            newsUrl = news.getNewsUrl() +
                    (news.getNewsUrl().contains("?") ? "&" : "?") + "appToken=" + eaiSys.getAppToken() + "&tenantId=" + tenantId + "&bindMode=" + eaiSys.getUserBindFlag();
        }
        if (null !=labelSystemDataDto && 2 == labelSystemDataDto.getProtocolType() && ApplicationTypeEnum.CS_APPLICATION.getType().equals(labelSystemDataDto.getDataType())){
            newsUrl = news.getNewsUrl();
        }
        if (null !=labelSystemDataDto &&  2 == labelSystemDataDto.getProtocolType() && ApplicationTypeEnum.BS_APPLICATION.getType().equals(labelSystemDataDto.getDataType())){
            newsUrl = labelSystemDataDto.getRestUrl();
        }
        return newsUrl;
    }
}
