package com.digiwin.athena.semc.service.portal.impl;

import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.digiwin.athena.semc.common.Constants;
import com.digiwin.athena.semc.dto.portal.VisitPortalReq;
import com.digiwin.athena.semc.entity.portal.PortalVisit;
import com.digiwin.athena.semc.mapper.portal.PortalVisitMapper;
import com.digiwin.athena.semc.service.portal.PortalVisitService;
import com.digiwin.athena.semc.util.Utils;
import org.apache.commons.lang3.ObjectUtils;
import org.springframework.stereotype.Repository;

import javax.annotation.Resource;
import java.time.LocalDateTime;
import java.util.List;

/**
 * 门户访问记录接口 实现
 *
 * @author sungqz
 * @since 2025-09-18
 */
@Repository
public class PortalVisitServiceImpl extends ServiceImpl<PortalVisitMapper, PortalVisit> implements PortalVisitService {

    @Resource
    PortalVisitMapper portalVisitMapper;

    /**
     * 查询门户访问记录
     *
     * @param clientType 客户端类型 0：PC端 1：移动端
     * @param visitType  访问类型，0：访问首页门户 1：访问管理后台门户
     * @return 返回
     */
    @Override
    public List<PortalVisit> queryPortalVisitList(Integer clientType, Integer visitType) {
        LambdaQueryWrapper<PortalVisit> queryWrapper = new LambdaQueryWrapper<>();
        queryWrapper.eq(PortalVisit::getClientType, clientType);
        queryWrapper.eq(PortalVisit::getVisitType, visitType);
        // 如果访问首页，则查询用户的访问记录
        if (Constants.VisitTypeEnum.ATHENA_PORTAL.getValue().equals(visitType)) {
            queryWrapper.eq(PortalVisit::getUserId, Utils.getUserId());
        } else { // 如果访问管理后台，则用户应该为空
            queryWrapper.isNull(PortalVisit::getUserId);
        }
        queryWrapper.orderByDesc(PortalVisit::getVisitTime);
        return portalVisitMapper.selectList(queryWrapper);
    }

    /**
     * 记录ISV门户的访问记录
     *
     * @param visitPortalReq 请求对象
     */
    @Override
    public void recordVisit(VisitPortalReq visitPortalReq) {
        // 查询当前门户的访问记录
        List<PortalVisit> portalVisitList = queryPortalVisitList(visitPortalReq.getClientType(), visitPortalReq.getVisitType());
        PortalVisit portalVisit = portalVisitList.stream().filter(x -> x.getVisitEleId().equals(visitPortalReq.getDesignerPortalId())).findFirst().orElse(null);
        // 添加访问记录
        if (ObjectUtils.isEmpty(portalVisit)) {
            portalVisit = new PortalVisit();
            portalVisit.setVisitEleId(visitPortalReq.getDesignerPortalId());
            portalVisit.setClientType(visitPortalReq.getClientType());
            portalVisit.setVisitType(visitPortalReq.getVisitType());
            portalVisit.setVisitTime(LocalDateTime.now());
            portalVisit.setTenantId(Utils.getTenantId());
            if (Constants.VisitTypeEnum.ATHENA_PORTAL.getValue().equals(visitPortalReq.getVisitType())) {
                portalVisit.setUserId(Utils.getUserId());
            }
            portalVisitMapper.insert(portalVisit);
        } else { // 更新访问时间
            portalVisit.setVisitTime(LocalDateTime.now());
            portalVisitMapper.updateById(portalVisit);
        }
    }
}
