package com.digiwin.athena.semc.service.portal.impl;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.baomidou.mybatisplus.core.toolkit.StringUtils;
import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.digiwin.athena.appcore.auth.AppAuthContextHolder;
import com.digiwin.athena.appcore.auth.domain.AuthoredUser;
import com.digiwin.athena.appcore.util.SnowflakeIdWorker;
import com.digiwin.athena.semc.common.Constants;
import com.digiwin.athena.semc.common.enums.PreinstalledApplicationTypeEnum;
import com.digiwin.athena.semc.dto.erpsso.RegisterDTO;
import com.digiwin.athena.semc.dto.erpsso.ThirdSsoInfoDto;
import com.digiwin.athena.semc.dto.portal.PreinstalledApplicationConfigUpdateDto;
import com.digiwin.athena.semc.entity.portal.PreinstalledApplication;
import com.digiwin.athena.semc.entity.portal.PreinstalledApplicationInstance;
import com.digiwin.athena.semc.mapper.portal.PreinstalledApplicationInstanceMapper;
import com.digiwin.athena.semc.mapper.portal.PreinstalledApplicationMapper;
import com.digiwin.athena.semc.service.portal.IPreinstalledApplicationService;
import com.digiwin.athena.semc.service.sso.IThirdSsoInfoService;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.compress.utils.Lists;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import javax.annotation.Resource;
import java.util.*;
import java.util.function.Function;
import java.util.stream.Collectors;

/**
 * @description: 预装应用服务类
 * @createDate: 2023/1/29 11:12
 * @author: sunyfa
 */
@Slf4j
@Service
public class PreinstalledApplicationServiceImpl implements IPreinstalledApplicationService {
    @Resource
    private PreinstalledApplicationMapper preinstalledApplicationMapper;

    @Resource
    private PreinstalledApplicationInstanceMapper preinstalledApplicationInstanceMapper;

    @Autowired
    IThirdSsoInfoService thirdSsoInfoService;

    /**
     * @param user 当前登陆人
     * @return
     * @description: 查询预装应用列表
     * @author: sunyfa
     */
    @Override
    public List<PreinstalledApplication> queryPreinstalledAppList(AuthoredUser user) {
        // 查询基础预设应用列表
        List<PreinstalledApplication> preinstalledApplicationList = preinstalledApplicationMapper.selectList(Wrappers.emptyWrapper());
        // 查询租户下的预设应用列表
        QueryWrapper<PreinstalledApplicationInstance> queryWrapper = new QueryWrapper<>();
        queryWrapper.eq("tenant_id", user.getTenantId());

        // 如果租户下没有应用列表，则新增应用
        List<PreinstalledApplicationInstance> instanceList = preinstalledApplicationInstanceMapper.selectList(queryWrapper);
        if (CollectionUtils.isEmpty(instanceList)) {
            insertInstance(user, preinstalledApplicationList);
        } else { // 如果租户下存在应用列表，则以租户下的应用返回
            // 判断是否有新的内置应用，如果有，则给当前租户初始化
            List<Long> instancePreIdList = instanceList.stream().map(PreinstalledApplicationInstance::getPreinstalledApplicationId).collect(Collectors.toList());
            List<PreinstalledApplication> newAppList = preinstalledApplicationList.stream().filter(x -> !instancePreIdList.contains(x.getId())).collect(Collectors.toList());
            if (CollectionUtils.isNotEmpty(newAppList)) {
                insertInstance(user, newAppList);
                instanceList = preinstalledApplicationInstanceMapper.selectList(queryWrapper);
            }

            // 过滤掉失效的应用
            List<Long> noValidIdList = instanceList.stream().filter(x -> Constants.VALID_STATUS_UNUSABLE.equals(x.getStatus())).map(PreinstalledApplicationInstance::getPreinstalledApplicationId).collect(Collectors.toList());
            preinstalledApplicationList = preinstalledApplicationList.stream().filter(x -> !noValidIdList.contains(x.getId())).collect(Collectors.toList());
            Map<Long, PreinstalledApplicationInstance> applicationIdMap = instanceList.stream().collect(Collectors.toMap(PreinstalledApplicationInstance::getPreinstalledApplicationId, Function.identity(), (a, b) -> a));
            preinstalledApplicationList.forEach(x -> {
                PreinstalledApplicationInstance instance = applicationIdMap.get(x.getId());
                if (instance != null) {
                    x.setUserBindFlag(instance.getUserBindFlag()); // 以租户下设置为准
                    x.setAppSid(x.getAppSid() == null ? instance.getAppSid() : x.getAppSid());
                    x.setAppId(StringUtils.isEmpty(x.getAppId()) ? instance.getAppId() : x.getAppId());
                    x.setAppSecret(StringUtils.isEmpty(x.getAppSecret()) ? instance.getAppSecret() : x.getAppSecret());
                }
            });
        }
        preinstalledApplicationList.sort(Comparator.comparing(PreinstalledApplication::getApplicationName));
        return preinstalledApplicationList;
    }

    /**
     * 新增当前租户的内置应用
     *
     * @param user                        用户信息
     * @param preinstalledApplicationList 内置应用
     */
    private void insertInstance(AuthoredUser user, List<PreinstalledApplication> preinstalledApplicationList) {
        List<PreinstalledApplicationInstance> instanceList = Lists.newArrayList();
        preinstalledApplicationList.forEach(x -> {
            PreinstalledApplicationInstance instance = new PreinstalledApplicationInstance();
            instance.setId(SnowflakeIdWorker.getInstance().newId());
            instance.setPreinstalledApplicationId(x.getId());
            instance.setUserBindFlag(x.getUserBindFlag());
            instance.setLinkDisplayFlag(Constants.LinkDisplayFlagEnum.YES.getFlag());
            instance.setStatus(Constants.VALID_STATUS_ENABLE);
            instance.setCreateUserId(user.getUserId());
            instance.setModifyUserId(user.getUserId());
            instance.setTenantId(user.getTenantId());

            // 生成appId
            if (Arrays.asList(Constants.BindFlagEnum.AUTO.getFlag(), Constants.BindFlagEnum.HAND.getFlag()).contains(x.getUserBindFlag())) {
                RegisterDTO registerReq = new RegisterDTO();
                registerReq.setName(x.getApplicationCode());
                registerReq.setDescription(x.getApplicationDescription());
                registerReq.setAppToken(x.getAppToken());
                if (PreinstalledApplicationTypeEnum.BS_APPLICATION.getValue().equals(x.getApplicationType())) {
                    ThirdSsoInfoDto thirdSsoInfoDto = Optional.ofNullable(JSON.toJavaObject(JSONObject.parseObject(x.getApplicationConfig()), ThirdSsoInfoDto.class)).orElse(new ThirdSsoInfoDto());
                    registerReq.setCallbackUrl(thirdSsoInfoDto.getCallBackUrl());
                } else {
                    registerReq.setCallbackUrl(String.format(Constants.CS_APP_CALLBACK_URL, x.getApplicationCode()));
                }
                try {
                    RegisterDTO registerResp = thirdSsoInfoService.registerApp(registerReq);
                    instance.setAppSid(registerResp.getSid());
                    x.setAppSid(registerResp.getSid());
                    instance.setAppId(registerResp.getId());
                    x.setAppId(registerResp.getId());
                    instance.setAppSecret(registerResp.getSecret());
                    x.setAppSecret(registerResp.getSecret());
                } catch (Exception e) {
                    log.error("PreinstalledApplicationServiceImpl invoke iam error error, registerReq:{}", registerReq, e);
                }
            }
            instanceList.add(instance);
        });
        if (CollectionUtils.isNotEmpty(instanceList)) {
            preinstalledApplicationInstanceMapper.batchInsertInstance(instanceList);
        }
    }

    /**
     * @param user                      当前登陆人
     * @param applicationType           应用类型
     * @param preinstalledApplicationId 预装应用ID
     * @return JSONObject 配置详情
     * @description: 查询预装应用的配置详情
     * @author: sunyfa
     */
    @Override
    public JSONObject queryPreinstalledAppDetail(AuthoredUser user, String applicationType, long preinstalledApplicationId) {
        // 根据主键查询预设应用
        PreinstalledApplication preinstalledApplication = preinstalledApplicationMapper.selectById(preinstalledApplicationId);
        JSONObject jsonObject = JSONObject.parseObject(preinstalledApplication.getApplicationConfig());

        QueryWrapper<PreinstalledApplicationInstance> queryWrapper = new QueryWrapper<>();
        queryWrapper.eq("preinstalled_application_id", preinstalledApplicationId);
        queryWrapper.eq("tenant_id", user.getTenantId());
        PreinstalledApplicationInstance preinstalledApplicationInstance = preinstalledApplicationInstanceMapper.selectOne(queryWrapper);

        if (Objects.nonNull(preinstalledApplicationInstance)) {
            jsonObject.put("appSid", preinstalledApplicationInstance.getAppSid());
            jsonObject.put("appId", preinstalledApplicationInstance.getAppId());
            jsonObject.put("appSecret", preinstalledApplicationInstance.getAppSecret());
        }

        // 特殊应用(T100，TT等)额外查询单独的特殊配置
        if (PreinstalledApplicationTypeEnum.SPECIAL_APPLICATION.getType().equals(applicationType)) {
            if (Objects.nonNull(preinstalledApplicationInstance) && Objects.nonNull(preinstalledApplicationInstance.getApplicationExtConfig())) {
                Iterator<String> keys = preinstalledApplicationInstance.getApplicationExtConfig().keys();
                while (keys.hasNext()) {
                    String key = keys.next();
                    jsonObject.put(key, preinstalledApplicationInstance.getApplicationExtConfig().get(key));
                }
            }
        }
        return jsonObject;
    }

    /**
     * @param user                                   当前登录用户
     * @param preinstalledApplicationConfigUpdateDto 需要更新的配置对象
     * @return void
     * @description: 更新预设应用的配置
     * @author: sunyfa
     */
    @Override
    public void updatePreinstalledAppConfig(AuthoredUser user, PreinstalledApplicationConfigUpdateDto preinstalledApplicationConfigUpdateDto) {
        // 查询租户下预设应用列表
        QueryWrapper<PreinstalledApplicationInstance> queryWrapper = new QueryWrapper<>();
        queryWrapper.eq("preinstalled_application_id", preinstalledApplicationConfigUpdateDto.getPreinstalledApplicationId().get(0));
        queryWrapper.eq("tenant_id", user.getTenantId());
        PreinstalledApplicationInstance instance = buildInstance(preinstalledApplicationConfigUpdateDto, user);
        // 更新应用
        preinstalledApplicationInstanceMapper.updatePreinstalledApplicationInstance(instance);
    }

    /**
     * 构建租户下应用
     *
     * @param preinstalledApplicationConfigUpdateDto 请求参数
     * @param user                                   用户信息
     * @return
     */
    private PreinstalledApplicationInstance buildInstance(PreinstalledApplicationConfigUpdateDto preinstalledApplicationConfigUpdateDto, AuthoredUser user) {
        PreinstalledApplicationInstance instance = new PreinstalledApplicationInstance();
        instance.setPreinstalledApplicationId(preinstalledApplicationConfigUpdateDto.getPreinstalledApplicationId().get(0));
        instance.setApplicationExtConfig(preinstalledApplicationConfigUpdateDto.getConfig());
        instance.setUserBindFlag(preinstalledApplicationConfigUpdateDto.getUserBindFlag());
        instance.setStatus(preinstalledApplicationConfigUpdateDto.getStatus());
        instance.setTenantId(user.getTenantId());
        instance.setModifyUserId(user.getUserId());
        return instance;
    }

    /**
     * 查询所有预设应用列表
     *
     * @param idList 应用id列表
     * @return
     */
    @Override
    public List<PreinstalledApplication> queryPreinstalledApplicationList(List<Long> idList) {
        QueryWrapper<PreinstalledApplication> queryWrapper = new QueryWrapper<>();
        queryWrapper.in("id", idList);
        return preinstalledApplicationMapper.selectList(queryWrapper);
    }

    /**
     * 更新应用信息
     *
     * @param ssoIdList       应用id列表
     * @param userBindFlag    归户模式
     * @param linkDisplayFlag 是否显示在友情链接 0:否；1:是
     * @return
     */
    @Override
    public void updateAppInfo(List<Long> ssoIdList, Integer userBindFlag, Integer linkDisplayFlag) {
        String tenantId = AppAuthContextHolder.getContext().getAuthoredUser().getTenantId();
        preinstalledApplicationMapper.updateAppInfo(ssoIdList, userBindFlag, linkDisplayFlag, tenantId);
    }

    @Override
    public PreinstalledApplication getPreByCode(String applicationCode, String appId) {
        QueryWrapper<PreinstalledApplication> queryWrapper = new QueryWrapper<>();
        if (StringUtils.isNotEmpty(applicationCode)) {
            queryWrapper.eq("application_code", applicationCode);
        }
        if (StringUtils.isNotEmpty(appId)) {
            queryWrapper.eq("app_id", appId);
        }
        return preinstalledApplicationMapper.selectOne(queryWrapper);
    }

    @Override
    public PreinstalledApplicationInstance getPreINstance(Long preinstalledId, String extConfig, String appId) {
        // 查询应用的动态参数
        QueryWrapper<PreinstalledApplicationInstance> queryWrapper = new QueryWrapper<>();
        if (null != preinstalledId) {
            List<Long> idList = new ArrayList<>();
            idList.add(preinstalledId);
            queryWrapper.in("preinstalled_application_id", idList);
        }
        if (StringUtils.isNotEmpty(extConfig)) {
            queryWrapper.like("application_ext_config", extConfig);
        }
        if (StringUtils.isNotEmpty(appId)) {
            queryWrapper.eq("app_id", appId);
        }
        AuthoredUser user = AppAuthContextHolder.getContext().getAuthoredUser();
        queryWrapper.eq("tenant_id", user.getTenantId());
        List<PreinstalledApplicationInstance> applicationInstanceList = preinstalledApplicationInstanceMapper.selectList(queryWrapper);
        if (CollectionUtils.isEmpty(applicationInstanceList)) {
            return null;
        }
        return applicationInstanceList.get(0);
    }

    @Override
    public Map<Long, PreinstalledApplication> getPreInfoMap() {
        List<PreinstalledApplication> preinstalledAppList = preinstalledApplicationMapper.selectList(new QueryWrapper<PreinstalledApplication>());
        Map<Long, PreinstalledApplication> preInfoMap = preinstalledAppList.stream().collect(Collectors.toMap(PreinstalledApplication::getId, Function.identity(), (a, b) -> a));
        return preInfoMap;
    }

    @Override
    public PreinstalledApplication getPreApp(Long id) {
        return preinstalledApplicationMapper.selectById(id);
    }

    @Override
    public Integer delApplicationInstance(List<Long> idList) {
        return preinstalledApplicationInstanceMapper.deleteBatchIds(idList);
    }
}
