package com.digiwin.athena.semc.service.portal.impl;

import com.digiwin.athena.semc.dto.portal.*;
import com.digiwin.athena.semc.dto.portal.todo.TodoFounderDto;
import com.digiwin.athena.semc.service.portal.*;
import com.baomidou.mybatisplus.core.conditions.Wrapper;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.baomidou.mybatisplus.core.conditions.update.LambdaUpdateWrapper;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.digiwin.athena.appcore.auth.AppAuthContextHolder;
import com.digiwin.athena.appcore.auth.domain.AuthoredUser;
import com.digiwin.athena.appcore.util.MessageUtils;
import com.digiwin.athena.appcore.util.ResponseEntityWrapper;
import com.digiwin.athena.semc.common.Constants;
import com.digiwin.athena.semc.common.ErrorCodeConstant;
import com.digiwin.athena.semc.common.I18NKey;
import com.digiwin.athena.semc.common.enums.ApplicationTypeEnum;
import com.digiwin.athena.semc.common.enums.EAIServiceNameEnum;
import com.digiwin.athena.semc.common.enums.IfNotIntegerEnum;
import com.digiwin.athena.semc.common.enums.MaycurCodeEnum;
import com.digiwin.athena.semc.common.enums.MaycurCulomnEnum;
import com.digiwin.athena.semc.common.enums.MaycurTaskActionTypeEnum;
import com.digiwin.athena.semc.common.enums.ProtocolTypeEnum;
import com.digiwin.athena.semc.common.enums.SourceEnum;
import com.digiwin.athena.semc.dto.PageInfo;
import com.digiwin.athena.semc.dto.erpsso.ErpSsoSpecialDTO;
import com.digiwin.athena.semc.dto.mq.MessageDO;
import com.digiwin.athena.semc.entity.portal.LabelSystemData;
import com.digiwin.athena.semc.entity.portal.PreinstalledApplication;
import com.digiwin.athena.semc.entity.portal.PreinstalledApplicationInstance;
import com.digiwin.athena.semc.entity.portal.ThirdTodoConfig;
import com.digiwin.athena.semc.entity.portal.TodoField;
import com.digiwin.athena.semc.entity.portal.TodoList;
import com.digiwin.athena.semc.entity.portal.TodoRead;
import com.digiwin.athena.semc.env.EnvProperties;
import com.digiwin.athena.semc.mapper.portal.LabelSystemDataMapper;
import com.digiwin.athena.semc.mapper.portal.TodoFieldMapper;
import com.digiwin.athena.semc.mapper.portal.TodoListMapper;
import com.digiwin.athena.semc.mapper.portal.TodoReadMapper;
import com.digiwin.athena.semc.proxy.eoc.service.EocService;
import com.digiwin.athena.semc.proxy.esp.service.ESPService;
import com.digiwin.athena.semc.proxy.iam.service.IamService;
import com.digiwin.athena.semc.proxy.iam.service.model.AccountDTO;
import com.digiwin.athena.semc.proxy.iam.service.model.AppUserDTO;
import com.digiwin.athena.semc.proxy.maycur.MayCurService;
import com.digiwin.athena.semc.proxy.tripartite.service.TripartiteService;
import com.digiwin.athena.semc.service.cache.RedisLock;
import com.digiwin.athena.semc.service.mq.MessageSendService;
import com.digiwin.athena.semc.util.DateUtils;
import com.digiwin.athena.semc.util.Utils;
import com.digiwin.athena.semc.vo.maycur.MaycurCalloutParamsReq;
import com.digiwin.athena.semc.vo.maycur.MaycurTodoNoticeReq;
import com.digiwin.athena.semc.vo.portal.ReimburseUrlVO;

import net.sf.json.JSONArray;

import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.collections.MapUtils;
import org.apache.commons.lang3.ObjectUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.i18n.LocaleContextHolder;
import org.springframework.http.ResponseEntity;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.nio.charset.StandardCharsets;
import java.text.DateFormat;
import java.text.SimpleDateFormat;
import java.util.*;
import java.util.function.Function;
import java.util.stream.Collectors;

import javax.annotation.Resource;

import io.github.linpeilie.Converter;
import lombok.extern.slf4j.Slf4j;

/**
 * 待办列表(TodoList)表服务实现类
 *
 * @author sunyfa
 * @since 2022-12-06 13:48:51
 */
@Service
@Slf4j
public class TodoListServiceImpl extends ServiceImpl<TodoListMapper, TodoList> implements TodoListService {

    @Resource
    private TodoFieldMapper todoFieldMapper;
    @Resource
    private TodoListMapper todoListMapper;
    @Resource
    private TodoReadMapper todoReadMapper;
    @Autowired
    private TodoFieldService todoFieldService;
    @Autowired
    LabelSystemDataService labelSystemDataService;
    @Autowired
    TripartiteService tripartiteService;
    @Autowired
    ThirdTodoConfigService thirdTodoConfigService;
    @Resource
    private LabelSystemDataMapper labelSystemDataMapper;
    @Resource
    private ESPService espService;
    @Resource
    private MessageUtils messageUtils;
    @Resource
    private IamService iamService;
    @Resource
    private MessageSendService messageSendService;
    @Resource
    MayCurService mayCurService;
    @Resource
    EocService eocService;

    @Resource
    private EnvProperties envProperties;

    @Resource
    private IPreinstalledApplicationService preinstalledApplicationService;

    @Resource
    private Converter converter;

    @Autowired
    private LabelSystemSourceService labelSystemSourceService;

    @Autowired
    private TodoAppSortService todoAppSortService;

    /**
     * 1.根据待办人查询上次待办的数据
     * 2.根据待办ID标识本次查询的待办哪些是新增
     * 3.删除上次待办数据，插入本地待办数据
     * 4.当前在待办页签，更新最新的浏览时间
     *
     * @param eaiSys 注册的应用
     * @param result 调用ESP返回的数据结果
     * @return 待办返回对象
     */
    @Override
    @Transactional
    public TodoListResp handleEspResult(AuthoredUser user, TodoListReq.EaiSys eaiSys, Map<String, Object> result, String qryCondition, LabelSystemDataDto labelSystemDataDto) {
        TodoListResp todoListResp = new TodoListResp();
        //获取数据
        if (Objects.isNull(result.get("to_do_list"))) {
            todoListResp.setRed(false);
            todoListResp.setTotalUnreadCount(0);
            todoListResp.setTotalResults(0);
        } else {
            //获取总条数
            Integer totalResults = result.get("total_results") == null ? 0 : Integer.parseInt(result.get("total_results").toString());
            todoListResp.setTotalResults(totalResults);
            // 查询当前系统当前用户的待办列表
            List<TodoList> existsTodoList = list(new QueryWrapper<TodoList>()
                    .eq("to_do_user", user.getUserId())
                    .eq("tenant_id", user.getTenantId())
                    .eq("app_name", eaiSys.getAppName()));

            Map<String, TodoList> existsTodoMap = CollectionUtils.isNotEmpty(existsTodoList) ?
                    existsTodoList.stream().collect(Collectors.toMap(TodoList::getToDoId, a -> a, (k1, k2) -> k1)) : Maps.newHashMap();

            // 需要新增的待办
            List<TodoList> insertTodoList = Lists.newArrayList();
            // 返回的待办数据
            List<TodoList> resultTodoList = Lists.newArrayList();

            JSONArray.fromObject(result.get("to_do_list")).stream().forEach(todoJson -> {
                TodoList todo = JSONObject.parseObject(todoJson.toString(), TodoList.class);
                // 拼接待办URL，用于SSO
                todo.setToDoUrl(getToDoUrl(todo.getToDoUrl(), user, labelSystemDataDto));
                if (existsTodoMap.containsKey(todo.getToDoId())) {
                    TodoList dbTodoList = existsTodoMap.get(todo.getToDoId());
                    String todoUrl = todo.getToDoUrl();
                    String todoData = todo.getToDoData();
                    BeanUtils.copyProperties(dbTodoList, todo);
                    // 拼接待办URL，用于SSO
                    todo.setToDoUrl(todoUrl);
                    todo.setToDoData(todoData);
                } else {
                    todo.setIsCompleted(IfNotIntegerEnum.NO.getValue());
                    todo.setToDoSource(SourceEnum.EXTERNAL.getSource());
                    todo.setAppName(eaiSys.getAppName());
                    todo.setToDoUser(user.getUserId());
                    todo.setIsRead(IfNotIntegerEnum.NO.getValue());
                    todo.setToDoSource(SourceEnum.EXTERNAL.getSource());
                    todo.setToDoData(JSONObject.parseObject(todoJson.toString()).getString("to_do_data"));
                    insertTodoList.add(todo);
                }

                if (IfNotIntegerEnum.NO.getValue().equals(todo.getIsRead())) {
                    todoListResp.setRed(true);
                    todoListResp.setTotalUnreadCount(todoListResp.getTotalUnreadCount() + 1);
                }
                //搜索匹配
                if (StringUtils.isNotEmpty(qryCondition) && todo.getToDoData().toLowerCase().contains(qryCondition.toLowerCase())) {
                    resultTodoList.add(todo);
                }
                if (StringUtils.isEmpty(qryCondition)) {
                    resultTodoList.add(todo);
                }
            });

            todoListResp.setTodoList(resultTodoList);

            if (CollectionUtils.isNotEmpty(insertTodoList)) {
                saveBatch(insertTodoList);
            }
        }

        // 设置返回的待办显示字段
        JSONArray fieldList = JSONArray.fromObject(result.get("display_field_list"));
        List<TodoField> todoFieldList = Lists.newArrayList();
        fieldList.stream().forEach(field -> {
            TodoField todoField = JSONObject.parseObject(field.toString(), TodoField.class);
            todoField.setToDoUser(user.getUserId());
            todoField.setAppName(eaiSys.getAppName());

            todoFieldList.add(todoField);
        });
        todoListResp.setTodoFieldList(todoFieldList);

        // 保存待办显示字段
        boolean exists = exists(new QueryWrapper<TodoField>()
                .eq("to_do_user", user.getUserId())
                .eq("tenant_id", user.getTenantId())
                .eq("app_name", eaiSys.getAppName()));

        if (!exists) {
            todoFieldService.saveBatch(todoFieldList);
        }

        return todoListResp;
    }

    /**
     * 拼接待办数据跳转url
     *
     * @param todoUrl
     * @param user
     * @param labelSystemDataDto
     * @return
     */
    private String getToDoUrl(String todoUrl, AuthoredUser user, LabelSystemDataDto labelSystemDataDto) {
        String toDoUrl = null;
        if (labelSystemDataDto != null && ProtocolTypeEnum.OAUTH.getType().equals(labelSystemDataDto.getProtocolType())) {
            toDoUrl = todoUrl +
                    (todoUrl.contains("?") ? "&" : "?") +
                    "appToken=" + labelSystemDataDto.getAppToken() +
                    "&appCode=" + labelSystemDataDto.getAppCode() +
                    "&tenantId=" + user.getTenantId() +
                    "&bindMode=" + labelSystemDataDto.getUserBindFlag() +
                    "&appId=" + labelSystemDataDto.getAppId();

        }
        if (labelSystemDataDto != null && ProtocolTypeEnum.CAS.getType() == labelSystemDataDto.getProtocolType() && ApplicationTypeEnum.CS_APPLICATION.getType().equals(labelSystemDataDto.getDataType())) {
            toDoUrl = todoUrl;
        }
        if (labelSystemDataDto != null && ProtocolTypeEnum.CAS.getType() == labelSystemDataDto.getProtocolType() && ApplicationTypeEnum.BS_APPLICATION.getType().equals(labelSystemDataDto.getDataType())) {
            toDoUrl = labelSystemDataDto.getCasServerUrl();
        }
        return toDoUrl;
    }

    /**
     * @param eaiSys    查询的系统
     * @param condition 条件
     * @return 筛选过的待办返回对象
     * @description: 帅选带条件时，根据条件查询本地数据做过滤
     * @author: sunyfa
     */
    @Override
    public TodoListResp localSearch(AuthoredUser user, TodoListReq.EaiSys eaiSys, String condition) {
        TodoListResp todoListResp = new TodoListResp();

        // 查询待办显示字段
        List<TodoField> todoFieldList = todoFieldMapper.selectList(new QueryWrapper<TodoField>()
                .eq("to_do_user", user.getUserId())
                .eq("tenant_id", user.getTenantId())
                .eq("app_name", eaiSys.getAppName()));

        List<TodoList> existsTodoList = list(new QueryWrapper<TodoList>()
                .eq("to_do_user", user.getUserId())
                .eq("tenant_id", user.getTenantId())
                .eq("app_name", eaiSys.getAppName())
                .like("to_do_data", condition));

        if (CollectionUtils.isNotEmpty(existsTodoList)) {
            todoListResp.setTodoList(existsTodoList);
        }

        todoListResp.setTodoFieldList(todoFieldList);

        return todoListResp;
    }

    @Override
    public TodoListResp pageQueryTodo(LabelSystemPreReq todoListReq) {
        TodoListResp todoListResp = new TodoListResp();
        String token = AppAuthContextHolder.getContext().getAuthoredUser().getToken();
        //查询数据源信息和sso信息
        LabelSystemDataDto labelSystemDataDto = labelSystemDataService.getSystemDataSso(todoListReq.getId());
        if (null == labelSystemDataDto) {
            todoListResp.setResultCode("3001");
            return todoListResp;
        }
        String appToken = StringUtils.isBlank(labelSystemDataDto.getAppToken()) ? envProperties.getAppToken() : labelSystemDataDto.getAppToken();
        Map<String, Object> result = new HashMap<>();
        Map<String, String> extHeader = new HashMap<>();
        extHeader.put("digi-userToken", token);
        extHeader.put("digi-appToken", appToken);
        //混合云模式，走esp-互联网中台-三方地端服务
        if (Constants.DataModelEnum.MODEL_HYBRID_CLOUD.getVal().equals(labelSystemDataDto.getDataModel())) {
            // 通过ESP查询三方系统的待办数据
            result = espService.queryByEsp(labelSystemDataDto.getMiddleSystemName(),
                    labelSystemDataDto.getMiddleSystemUid(), EAIServiceNameEnum.TODOLIST.getServiceName(), extHeader, null, todoListReq.getQryCondition(),
                    PageInfo.getPageInfo(todoListReq.getPageNum(), todoListReq.getPageSize()));
        }//非混合云模式
        else {
            result = tripartiteService.selectDataPagePost(labelSystemDataDto.getRestUrl(), todoListReq, appToken);
        }
        // 没有显示字段时，返回异常
        if (Objects.isNull(result.get("display_field_list"))) {
            log.error(messageUtils.getMessage(I18NKey.TODO_FIELD_MISSING_ERROR));
            todoListResp.setResultCode("3001");
            return todoListResp;
        }
        TodoListReq.EaiSys eaiSys = new TodoListReq.EaiSys();
        eaiSys.setAppName(labelSystemDataDto.getAppName());
        eaiSys.setAppToken(appToken);
        eaiSys.setUserBindFlag(labelSystemDataDto.getUserBindFlag());
        todoListResp = handleEspResult(AppAuthContextHolder.getContext().getAuthoredUser(), eaiSys, result, todoListReq.getQryCondition(), labelSystemDataDto);
        return todoListResp;
    }


    @Override
    public TodoListResp selectTodoList(PreSystemToDoReq req) {
        TodoListResp todoListResp = new TodoListResp();
        todoListResp.setTodoList(new ArrayList<>());
        todoListResp.setTodoFieldList(new ArrayList<>());
        //查询数据源信息和sso信息
        LabelSystemDataDto labelSystemDataDto = labelSystemDataService.getSystemDataSso(req.getId());
        if (null == labelSystemDataDto) {
            return todoListResp;
        }
        if (!labelSystemDataDto.getAppId().equals(req.getAppId())) {
            log.info("三方待办列表请求和查询的应用id不一致：{}，{}",req.getAppId(),labelSystemDataDto.getAppId());
        }
        req.setAppId(labelSystemDataDto.getAppId());
        String tokenId = null;
        String verifyUserId;
        //每刻国际版
        if (MaycurCodeEnum.MAYCUR_APPROVALS.getType().equals(labelSystemDataDto.getAppCode())) {
            // 获取员工工号
            String userId = this.eocService.getEmpJobNum();
            if (StringUtils.isEmpty(userId)) {
                return todoListResp;
            }
            verifyUserId = userId;
//            String userId = "0638";
            ResponseEntity<Map> response = this.mayCurService.getTokenId(labelSystemDataDto, userId);
            if (null != response && null != response.getBody()) {
                Map<String, Object> map = (Map<String, Object>) response.getBody().get("data");
                if (null != map) {
                    tokenId = map.get("tokenId").toString();
                }
            }
            if (StringUtils.isNotEmpty(tokenId)) {
                // 获取到每刻报销全量数据
                ResponseEntity<Map> responseEntity;
                List<Map<String, Object>> map = new LinkedList<>();
                List<Map<String, Object>> totalMapData = new LinkedList<>();
                int pageNum = -1;
                do {
                    pageNum++;
                    map = this.mayCurService.listMayCurData(tokenId, pageNum, labelSystemDataDto);
                    if (CollectionUtils.isNotEmpty(map)) {
                        totalMapData.addAll(map);
                    }
                } while (map.size() == 500);

                List<TodoField> todoFieldList = new LinkedList<>();
                this.assembleData(totalMapData, labelSystemDataDto, todoFieldList, userId);
                todoListResp.setTodoFieldList(todoFieldList);
            }
        }
        //非每刻的三方待办
        else {
            //查询当前鼎捷云用户对应地端用户
            verifyUserId = iamService.queryMappingEmpId(labelSystemDataDto.getAppCode());
            log.info("三方待办地端账号：{}，{}", labelSystemDataDto.getAppCode(),verifyUserId);
            if (StringUtils.isEmpty(verifyUserId)) {
                return todoListResp;
            }
            todoListResp.setTodoFieldList(selectFieldBy(req.getAppId()));
        }

        // 查询当前系统当前用户的待办列表
        Page<TodoList> todoDataPage = selectByPage(req, verifyUserId);
        List<TodoList> todoLists = todoDataPage.getRecords();
        if (CollectionUtils.isEmpty(todoLists)) {
            return todoListResp;
        }
        //封装已读未读、跳转url
        packageRead(todoLists, req, verifyUserId, labelSystemDataDto);
        todoListResp.setTodoList(todoLists);
        todoListResp.setTotalResults(Integer.parseInt(todoDataPage.getTotal() + ""));
        return todoListResp;
    }

    /**
     * 查询租户下的账套信息
     *
     * @param userDTOList 参数
     */
    private void queryAccountList(List<AppUserDTO> userDTOList) {
        // 查询当前租户下的账套信息
        List<AccountDTO> accountDTOList = iamService.queryTenantAccountMappingList(null, null, null);
        if (CollectionUtils.isEmpty(accountDTOList)) {
            return;
        }
        Map<String, List<AccountDTO>> accountMap = accountDTOList.stream().collect(Collectors.groupingBy(AccountDTO::getAppId));
        for (AppUserDTO appUserDTO : userDTOList) {
            if (CollectionUtils.isNotEmpty(accountMap.get(appUserDTO.getAppCode()))) {
                List<AccountDTO> accountList = accountMap.get(appUserDTO.getAppCode());
                List<String> accountIdList = Lists.newArrayList();
                accountList.forEach(x -> {
                    if (StringUtils.isNotBlank(x.getAccount())) {
                        accountIdList.add(x.getAccount());
                    } else if (StringUtils.isNotBlank(x.getCompanyId())) {
                        accountIdList.add(x.getCompanyId());
                    }
                });
                appUserDTO.setAccountIdList(accountIdList);
            }
        }
    }

    /**
     * 数据落表处理
     *
     * @param map
     * @param labelSystemDataDto
     * @param userId
     */
    private void assembleData(List<Map<String, Object>> map, LabelSystemDataDto labelSystemDataDto,
                              List<TodoField> todoFieldList, String userId) {
        List<TodoList> todoLists = new LinkedList<>();
        DateFormat df = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss");
        //需要更新的数据
        List<String> updateToDoIdList = new ArrayList<>();
        //需要新增的数据
        List<String> addToDoIdList = new ArrayList<>();

        int count = 0;
        for (Map<String, Object> item : map) {
            String status;
            if (Constants.APPROVING.equals(item.get("status").toString())) {
                status = Constants.APPROVINGZH;
            } else {
                status = Constants.CONFIRMINGZH;
            }
            JSONObject jsonObject = new JSONObject(true)
                    .fluentPut("type", item.get("type").toString())
                    .fluentPut("businessCode", item.get("businessCode").toString())
                    .fluentPut("name", item.get("name").toString())
                    .fluentPut("reimUserName", item.get("reimUserName").toString())
                    .fluentPut("coverName(departmentName)", item.get("coverName").toString() + "(" + item.get("departmentName").toString() + ")")
                    .fluentPut("submittedAt", df.format(item.get("submittedAt")))
                    .fluentPut("status", status);

            count++;
            TodoList todoList = new TodoList();
            todoList.setToDoSource(2);
            todoList.setAppName(labelSystemDataDto.getAppName());
            todoList.setAppId(labelSystemDataDto.getAppId());
            todoList.setToDoUser(userId);
            todoList.setToDoData(JSON.toJSONString(jsonObject));
            todoList.setToDoId(item.get("businessCode").toString());
            // 拼接PC端URL
            todoList.setToDoUrl(labelSystemDataDto.getCallBackUrl() + item.get("url").toString());
            // 拼接APP端URL
            todoList.setMobileUrl(labelSystemDataDto.getCallBackUrl() + item.get("url").toString());
            todoList.setMobileMainTitle(item.get("name").toString());
            todoList.setMobileSubTitle(df.format(item.get("submittedAt")));
            todoList.setSubmittedAt(DateUtils.strToDate(todoList.getMobileSubTitle()));
            String submitterEmployeeId = "";
            String submitterName = "";
            if (null != item.get("submitterEmployeeId")) {
                submitterEmployeeId = item.get("submitterEmployeeId").toString();
            }
            if (null != item.get("submitterName")) {
                submitterName = item.get("submitterName").toString();
            }
            todoList.setMobileLabel("工号:" + submitterEmployeeId + ";" + "名称:" + submitterName + ";" + "申请金额:" + item.get("amount"));
            todoLists.add(todoList);

            int i = 0;
            if (count == 1) {
                for (Map.Entry<String, Object> entry : jsonObject.entrySet()) {
                    TodoField todoField = new TodoField();
                    todoField.setAppName(labelSystemDataDto.getAppName());
                    todoField.setAppId(labelSystemDataDto.getAppId());
                    todoField.setToDoUser(userId);
                    todoField.setDisplayOrder(i++);
                    todoField.setDisplayName(MaycurCulomnEnum.getDesc(entry.getKey()));
                    todoField.setBindingDataKey(entry.getKey());
                    todoFieldList.add(todoField);
                }
            }
        }
        //提交时间倒序排序
        todoLists = todoLists.stream()
                .sorted(Comparator.comparing(TodoList::getSubmittedAt).reversed()).collect(Collectors.toList());

        //删除待办
        List<TodoList> deTodoList = delTodoList(userId, labelSystemDataDto.getAppId(), todoLists, updateToDoIdList, addToDoIdList);
        if (CollectionUtils.isNotEmpty(deTodoList)) {
            List<Long> toDoIds = deTodoList.stream().map(TodoList::getId).collect(Collectors.toList());
            todoListMapper.deleteBatchIds(toDoIds);
        }
        //遍历识别出需要更新和新增的数据
        List<TodoList> updateToDoLists = new LinkedList<>();
        List<TodoList> addToDoLists = new LinkedList<>();
        for (TodoList todo : todoLists) {
            if (updateToDoIdList.contains(todo.getToDoId())) {
                updateToDoLists.add(todo);
            }
            if (addToDoIdList.contains(todo.getToDoId())) {
                addToDoLists.add(todo);
            }
        }
        if (CollectionUtils.isNotEmpty(updateToDoLists)) {
            updateToDoLists.forEach(item -> {
                LambdaUpdateWrapper<TodoList> updateWrapper = new LambdaUpdateWrapper<>();
                updateWrapper.eq(TodoList::getAppId, item.getAppId());
                updateWrapper.eq(TodoList::getToDoId, item.getToDoId());
                updateWrapper.eq(TodoList::getToDoUser, item.getToDoUser());
                this.update(item, updateWrapper);
            });
        }
        if (CollectionUtils.isNotEmpty(addToDoLists)) {
            this.saveBatch(addToDoLists);
        }

        LambdaQueryWrapper<TodoField> queryTodoFieldWrapper = new LambdaQueryWrapper<>();
        queryTodoFieldWrapper.eq(TodoField::getAppId, labelSystemDataDto.getAppId());
        todoFieldService.remove(queryTodoFieldWrapper);
        todoFieldService.saveBatch(todoFieldList);
    }

    /**
     * 获取需要删除的数据库待办
     *
     * @param userId
     * @param appId
     * @param mayTodoLists
     * @return
     */
    public List<TodoList> delTodoList(String userId, String appId, List<TodoList> mayTodoLists, List<String> updateToDoIdList, List<String> addToDoIdList) {
        List<TodoList> deTodoList = new ArrayList<>();
        List<String> toDoUserList = new ArrayList<>();
        List<String> dbToDoIds = new ArrayList<>();
        toDoUserList.add(userId);
        //遍历需要删除的数据库中的待办
        List<TodoList> dbTodoList = selectToDoBy(appId, "", toDoUserList);
        if (CollectionUtils.isNotEmpty(dbTodoList)) {
            //每刻报销的待办单据编号
            dbToDoIds = dbTodoList.stream().map(TodoList::getToDoId).collect(Collectors.toList());
        }
        //每刻报销的待办单据编号
        List<String> mayToDoIds = mayTodoLists.stream().map(TodoList::getToDoId).collect(Collectors.toList());
        if (CollectionUtils.isEmpty(mayToDoIds)) {
            return dbTodoList;
        }
        //遍历数据库待办，识别出需要删除的待办
        for (TodoList dbTodo : dbTodoList) {
            //如果每刻报销返回的待办不包括数据中的，需要删除
            if (!mayToDoIds.contains(dbTodo.getToDoId())) {
                deTodoList.add(dbTodo);
            }
        }
        //遍历每刻待办，识别出需要新增的待办和更新的待办
        for (TodoList mayTodo : mayTodoLists) {
            if (dbToDoIds.contains(mayTodo.getToDoId())) {
                updateToDoIdList.add(mayTodo.getToDoId());
            } else {
                addToDoIdList.add(mayTodo.getToDoId());
            }

        }
        return deTodoList;
    }


    /**
     * 封装已读未读、跳转url
     *
     * @param todoLists
     */
    public void packageRead(List<TodoList> todoLists, PreSystemToDoReq req, String verifyUserId, LabelSystemDataDto labelSystemDataDto) {
        List<TodoRead> todoReadList = selectToDoReadBy(req, verifyUserId);
        List<String> toDoReadIdList = todoReadList.stream().map(TodoRead::getToDoId).collect(Collectors.toList());
        for (TodoList todo : todoLists) {
            todo.setIsRead(IfNotIntegerEnum.NO.getValue());
            if (toDoReadIdList.contains(todo.getToDoId())) {
                todo.setIsRead(IfNotIntegerEnum.YES.getValue());
            }
            if (!MaycurCodeEnum.MAYCUR_APPROVALS.getType().equals(labelSystemDataDto.getAppCode())) {
                AuthoredUser user = AppAuthContextHolder.getContext().getAuthoredUser();
                if (StringUtils.isNotEmpty(todo.getToDoUrl())) {
                    //拼接PC跳转url
                    String toDoUrl = getToDoUrl(todo.getToDoUrl(), user, labelSystemDataDto);
                    todo.setToDoUrl(toDoUrl);
                }
                if (StringUtils.isNotEmpty(todo.getMobileUrl())) {
                    //拼接App跳转url
                    String mobileUrl = getToDoUrl(todo.getMobileUrl(), user, labelSystemDataDto);
                    todo.setMobileUrl(mobileUrl);
                }
            }
        }
    }


    /**
     * 查询存在的待办数据
     *
     * @param appId
     * @return
     */
    public List<TodoField> selectFieldBy(String appId) {
        QueryWrapper queryWrapper = new QueryWrapper<TodoField>();
        //应用查询
        if (StringUtils.isNotEmpty(appId)) {
            queryWrapper.eq("app_id", appId);
        }
        return todoFieldMapper.selectList(queryWrapper);
    }

    /**
     * 查询待办的已读数据
     *
     * @param req
     * @param verifyUserId 地端用户id
     * @return
     */
    public List<TodoRead> selectToDoReadBy(PreSystemToDoReq req, String verifyUserId) {
        QueryWrapper queryWrapper = new QueryWrapper<TodoRead>();
        //应用查询
        if (StringUtils.isNotEmpty(req.getAppId())) {
            queryWrapper.eq("app_id", req.getAppId());
        }
        if (StringUtils.isNotEmpty(verifyUserId)) {
            queryWrapper.eq("to_do_user", verifyUserId);
        }
        return todoReadMapper.selectList(queryWrapper);
    }

    /**
     * 查询存在的待办数据
     *
     * @return
     */
    public List<TodoList> selectToDoBy(String appId, String toDoId, List<String> toDoUserList) {
        QueryWrapper queryWrapper = new QueryWrapper<TodoList>();
        //应用查询
        if (StringUtils.isNotEmpty(appId)) {
            queryWrapper.eq("app_id", appId);
        }
        //外部待办唯一id
        if (StringUtils.isNotEmpty(toDoId)) {
            queryWrapper.eq("to_do_id", toDoId);
        }
        //待办处理人
        if (CollectionUtils.isNotEmpty(toDoUserList)) {
            queryWrapper.in("to_do_user", toDoUserList);
        }
        return todoListMapper.selectList(queryWrapper);
    }

    /**
     * 查询存在的待办数据
     *
     * @return
     */
    public List<TodoList> selectToDoBy(String appId, String toDoId, List<String> toDoUserList
            ,Integer dataCategory,String verifyUserId) {
        QueryWrapper<TodoList> queryWrapper = new QueryWrapper<TodoList>();
        //应用查询
        if (StringUtils.isNotEmpty(appId)) {
            queryWrapper.eq("app_id", appId);
        }
        //外部待办唯一id
        if (StringUtils.isNotEmpty(toDoId)) {
            queryWrapper.eq("to_do_id", toDoId);
        }
        //待办处理人
        if (CollectionUtils.isNotEmpty(toDoUserList)) {
            queryWrapper.in("to_do_user", toDoUserList);
        }
        // 0：待处理、根据处理人查询创建人
        if (null != dataCategory && dataCategory == 0 && StringUtils.isNotEmpty(verifyUserId)) {
            queryWrapper.eq("to_do_user", verifyUserId);
        }
        // 1：已完成 查询处理人和创建人
        if (null != dataCategory && dataCategory == 1 && StringUtils.isNotEmpty(verifyUserId)) {
            queryWrapper.and(x -> x.eq("to_do_user", verifyUserId)
                    .or().eq("founder", verifyUserId));
        }
        return todoListMapper.selectList(queryWrapper);
    }

    /**
     * 获取查询待办数据QueryWrapper
     *
     * @return
     */
    public QueryWrapper getToDoWrapepr(String appId, List<String> toDoIds) {
        QueryWrapper queryWrapper = new QueryWrapper<TodoList>();
        //应用查询
        if (StringUtils.isNotEmpty(appId)) {
            queryWrapper.eq("app_id", appId);
        }
        //外部待办唯一id
        if (CollectionUtils.isNotEmpty(toDoIds)) {
            queryWrapper.in("to_do_id", toDoIds);
        }
        return queryWrapper;
    }

    public QueryWrapper getToDoWrapeprByUserId(String appId, List<String> toDoIds, String verifyUserId) {
        QueryWrapper queryWrapper = new QueryWrapper<TodoList>();
        //应用查询
        if (StringUtils.isNotEmpty(appId)) {
            queryWrapper.eq("app_id", appId);
        }
        //外部待办唯一id
        if (CollectionUtils.isNotEmpty(toDoIds)) {
            queryWrapper.in("to_do_id", toDoIds);
        }

        //用户id
        if (StringUtils.isNotEmpty(verifyUserId)) {
            queryWrapper.eq("to_do_user", verifyUserId);
        }
        return queryWrapper;
    }

    /**
     * 条件查询三方待办数据
     *
     * @param req
     * @param verifyUserId 地端用户id
     * @return
     */
    public Page<TodoList> selectByPage(PreSystemToDoReq req, String verifyUserId) {
        QueryWrapper<TodoList> queryWrapper = new QueryWrapper<>();
        queryWrapper.orderByAsc("id");
        //应用查询
        if (StringUtils.isNotEmpty(req.getAppId())) {
            queryWrapper.eq("app_id", req.getAppId());
        }
        //关键词搜索
        if (StringUtils.isNotEmpty(req.getQryCondition())) {
            queryWrapper.like("to_do_data", req.getQryCondition());
        }
        //创建人搜索
        /*if (StringUtils.isNotEmpty(req.getFounder())) {
            queryWrapper.eq("founder", req.getFounder());
        }*/
        //创建时间-开始时间
        /*if (StringUtils.isNotEmpty(req.getCreateTimeStart())) {
            queryWrapper.ge("to_do_create_time", req.getCreateTimeStart()+DateUtils.START_TIME);
        }*/
        //创建时间-结束时间
        /*if (StringUtils.isNotEmpty(req.getCreateTimeEnd())) {
            queryWrapper.le("to_do_create_time", req.getCreateTimeEnd()+DateUtils.END_TIME);
        }*/
        //完成时间-开始时间
        /*if (StringUtils.isNotEmpty(req.getEndTimeStart())) {
            queryWrapper.ge("to_do_end_time", req.getEndTimeStart()+DateUtils.START_TIME);
        }*/
        //完成时间-结束时间
        /*if (StringUtils.isNotEmpty(req.getEndTimeEnd())) {
            queryWrapper.le("to_do_end_time", req.getEndTimeEnd()+DateUtils.END_TIME);
        }*/
        //根据查询分类查询,0：待处理、1：已完成、2：我创建。默认0
       if (0 ==req.getDataCategory()) {
            queryWrapper.eq("is_completed", 0);
            queryWrapper.eq("to_do_user", verifyUserId);
        }
        if (1 ==req.getDataCategory()) {
            queryWrapper.eq("is_completed", 1);
            queryWrapper.and(x -> x.eq("to_do_user", verifyUserId));
        }
//        if (2 ==req.getDataCategory()) {
//            queryWrapper.eq("founder", verifyUserId);
//        }

        Page<TodoList> todoDataPage = todoListMapper.selectPage(new Page<>(req.getPageNum(), req.getPageSize()), queryWrapper);
        return todoDataPage;
    }

    @Override
    public ResponseEntity<?> getTodoListUrl(ReimburseReq reimburseReq) {
        ReimburseUrlVO reimburseUrlVO = new ReimburseUrlVO();

        TodoList todoList = todoListMapper.selectById(reimburseReq.getId());
        if (ObjectUtils.isEmpty(todoList)) {
            return ResponseEntityWrapper.wrapperOk(reimburseUrlVO);
        }

        long timestamp = System.currentTimeMillis();
        //取回调地址
        PreinstalledApplicationInstance applicationInstance = preinstalledApplicationService.getPreINstance(null, todoList.getAppId(), "");
        if (ObjectUtils.isEmpty(applicationInstance)) {
            applicationInstance = preinstalledApplicationService.getPreINstance(null, "", todoList.getAppId());
        }
        if (ObjectUtils.isEmpty(applicationInstance)) {
            return ResponseEntityWrapper.wrapperOk(reimburseUrlVO);
        }
        //查询预设主表
        PreinstalledApplication preApplication = preinstalledApplicationService.getPreApp(applicationInstance.getPreinstalledApplicationId());
        if (ObjectUtils.isEmpty(preApplication)) {
            return ResponseEntityWrapper.wrapperOk(reimburseUrlVO);
        }
        //TIPTOP
        if (Constants.TIPTOP_PRODUCT_TYPE.equals(preApplication.getProductCode())) {
            String ssoUrl = getTIPTOPUrl(applicationInstance, preApplication, todoList.getToDoUrl());
            String toDoUrl = getOtherArg(todoList.getToDoUrl());
            todoList.setToDoUrl(toDoUrl);
            ssoUrl = packageSsoUrl(ssoUrl, todoList.getToDoUrl());
            reimburseUrlVO.setToDoUrl(ssoUrl);
            reimburseUrlVO.setToDoMobileUrl(ssoUrl);
        }
        //T100
        if (Constants.T100_PRODUCT_TYPE.equals(preApplication.getProductCode())) {
            String ssoUrl = getT100Url(applicationInstance, preApplication);
            ssoUrl = packageSsoUrl(ssoUrl, todoList.getToDoUrl());
            reimburseUrlVO.setToDoUrl(ssoUrl);
            reimburseUrlVO.setToDoMobileUrl(ssoUrl);
        }
        //每刻报销
        if (MaycurCodeEnum.MAYCUR_APPROVALS.getType().equals(preApplication.getApplicationCode())) {
            //获取工号
            String userId = eocService.getEmpJobNum();
            if (StringUtils.isEmpty(userId)) {
                return ResponseEntityWrapper.wrapperFail(ErrorCodeConstant.NO_USER_ID, "无对应工号");
            }
            net.sf.json.JSONObject jsonObject = applicationInstance.getApplicationExtConfig();
            String appId = jsonObject.getString("appId");
            String appSecret = jsonObject.getString("appSecret");
            //获取ssoToken
            String ssoToken = mayCurService.getSsoToken(userId, appSecret, timestamp);
            String ssoUrl = mayCurService.getReplaceExistUrl(todoList.getToDoUrl(), appId, userId, timestamp, ssoToken);
            reimburseUrlVO.setToDoUrl(ssoUrl);
            reimburseUrlVO.setToDoMobileUrl(ssoUrl);
        }
        return ResponseEntityWrapper.wrapperOk(reimburseUrlVO);
    }

    /**
     * 拼接待办地址
     *
     * @param ssoUrl
     * @return
     */
    public String packageSsoUrl(String ssoUrl, String toDoUrl) {
        if (StringUtils.isEmpty(toDoUrl)) {
            return ssoUrl;
        }
        //&开头的
        if (toDoUrl.startsWith("&")) {
            ssoUrl = ssoUrl + toDoUrl;
        }
        //非&开头的，加上 &
        if (!toDoUrl.startsWith("&")) {
            ssoUrl = ssoUrl + "&" + toDoUrl;
        }
        return ssoUrl;
    }

    /**
     * 获取TIPTOP的SSO链接地址
     *
     * @param applicationInstance
     * @return
     */
    public String getTIPTOPUrl(PreinstalledApplicationInstance applicationInstance, PreinstalledApplication preApplication, String toDoUrl) {
        String ssoUrl = "";
        net.sf.json.JSONObject jsonObject = applicationInstance.getApplicationExtConfig();
        // 拼接TT的链接
        ErpSsoSpecialDTO erpSsoSpecialDTO = Optional.ofNullable(JSONObject.parseObject(preApplication.getApplicationConfig(), ErpSsoSpecialDTO.class)).orElse(new ErpSsoSpecialDTO());
        String ip = Optional.ofNullable(jsonObject).map(object -> object.getString("ip")).orElse("");
        // 查询云端账号绑定的地端员工账号
        String loginUserId = iamService.queryMappingEmpId(preApplication.getApplicationCode());
        // 获取tokenKey
        String tokenKey = Utils.encryptionTokenKey(erpSsoSpecialDTO.getSystemCode(), loginUserId);
        ssoUrl = erpSsoSpecialDTO.getProtocolHeader() + ip + "?Arg=" + erpSsoSpecialDTO.getArg() + "&Arg="
                + loginUserId + "&Arg=" + tokenKey + "&" + getOperCenter(toDoUrl) + "&" + getJoCode(toDoUrl) + "&Arg=" + LocaleContextHolder.getLocale();

        return ssoUrl;
    }

    /**
     * 获取作业代号
     *
     * @return
     */
    public String getJoCode(String toDoUrl) {
        String jobCode = "";
        //以&分隔，取第一个作为作业代号
        if (toDoUrl.contains("&") && toDoUrl.startsWith("") && toDoUrl.split("&").length > 2) {
            jobCode = toDoUrl.split("&")[1];
        }
        return jobCode;
    }


    /**
     * 获取运营中心
     *
     * @return
     */
    public String getOperCenter(String toDoUrl) {
        String operCenter = "";
        //以&分隔，取第2个作为运营中心
        if (toDoUrl.contains("&") && toDoUrl.startsWith("") && toDoUrl.split("&").length > 3) {
            operCenter = toDoUrl.split("&")[2];
        }
        return operCenter;
    }


    /**
     * 获取待办中剩余arg参数
     *
     * @param toDoUrl
     * @return
     */
    public String getOtherArg(String toDoUrl) {
        String resp = "";
        String[] argList = toDoUrl.split("&");
        for (int i = 0; i < argList.length; i++) {
            if (StringUtils.isEmpty(argList[i])) {
                continue;
            }
            if (i <= 2) {
                continue;
            }
            resp = resp + "&" + argList[i];
        }
        return resp;
    }

    /**
     * 获取T100的SSO链接
     *
     * @param applicationInstance
     * @return
     */
    public String getT100Url(PreinstalledApplicationInstance applicationInstance, PreinstalledApplication preApplication) {
        String ssoUrl = "";
        net.sf.json.JSONObject jsonObject = applicationInstance.getApplicationExtConfig();
        ErpSsoSpecialDTO erpSsoSpecialDTO = Optional.ofNullable(JSONObject.parseObject(preApplication.getApplicationConfig(), ErpSsoSpecialDTO.class)).orElse(new ErpSsoSpecialDTO());
        String protocolHeader = Optional.ofNullable(jsonObject).map(object -> object.getString("protocolHeader")).orElse("");
        String ip = Optional.ofNullable(jsonObject).map(object -> object.getString("ip")).orElse("");
        String companyCode = Optional.ofNullable(jsonObject).map(object -> object.getString("companyCode")).orElse("");
        String environment = StringUtils.isNotBlank(erpSsoSpecialDTO.getEnvironment()) ? "/w" + erpSsoSpecialDTO.getEnvironment() : "";
        String linkStr = protocolHeader + ip + environment + "/wa/r/app/gdc_azzi000?Arg=" + companyCode + "&Arg=" + erpSsoSpecialDTO.getArg() + "&Arg=QVRIRU5B&Arg=";
        // 查询云端账号绑定的地端员工账号
        String loginUserId = iamService.queryMappingEmpId(preApplication.getApplicationCode());
        // 参数hashKey，时间戳 + trust key + 登录账号，base64加密
        String hashKey = DateUtils.getNowTime(DateUtils.DATE_TIME_FORMATTER) + Constants.T100_TRUST_KEY + loginUserId;
        String encoderHashKey = Base64.getEncoder().encodeToString(hashKey.getBytes(StandardCharsets.UTF_8));
        encoderHashKey = encoderHashKey.replaceAll("\\+", "_plus_");
        ssoUrl = linkStr + encoderHashKey;
        return ssoUrl;
    }

    @Override
    @Transactional
    public Integer toDoReceive(TodoReceiveDto receiveDto) {
        List<TodoList> todoList = Lists.newArrayList();
        List<String> verifyUserldList = new ArrayList<>();
        //待删除待办ID
        List<String> todoIds = new ArrayList<>();
        //保存待办数据
        for (TodoListDto todoDto : receiveDto.getToDoList()) {
            //校验待办处理人
            if (StringUtils.isEmpty(todoDto.getToDoOwner())) {
                continue;
            }
            //校验待办id
            if (StringUtils.isEmpty(todoDto.getToDoId())) {
                continue;
            }
            //将代办人加入集合
            if (!verifyUserldList.contains(todoDto.getToDoOwner())) {
                verifyUserldList.add(todoDto.getToDoOwner());
            }
            TodoList toDo = converter.convert(todoDto, TodoList.class);
            toDo.setAppIdCode(receiveDto.getAppIdCode());
            toDo.setAppId(receiveDto.getAppCode());
            toDo.setMobileUrl(todoDto.getToDoMobileUrl());
            toDo.setToDoUser(todoDto.getToDoOwner());
            todoList.add(toDo);
            todoIds.add(todoDto.getToDoId());
        }
        if (CollectionUtils.isEmpty(todoList)) {
            return null;
        }
        //删除待办
        if (CollectionUtils.isNotEmpty(todoIds)) {
            //todoListMapper.delete(getToDoWrapepr(receiveDto.getAppCode(), todoIds));
            List<TodoList> delToDoList = todoListMapper.selectList(getToDoWrapepr(receiveDto.getAppCode(), todoIds));
            if (CollectionUtils.isNotEmpty(delToDoList)) {
                List<Long> delIdList = delToDoList.stream().map(TodoList::getId).collect(Collectors.toList());
                todoListMapper.deleteBatchIds(delIdList);
            }

        }
        //保存待办数据
        saveBatch(todoList);
        //删除该应用展示的标题数据
        QueryWrapper delFieldWrapper = new QueryWrapper<TodoField>();
        //应用查询
        delFieldWrapper.eq("app_id", receiveDto.getAppCode());
        todoFieldMapper.delete(delFieldWrapper);
        //保存待办标题
        List<TodoField> todoFieldList = Lists.newArrayList();
        for (TodoFieldDto toDoField : receiveDto.getDisplayFieldList()) {
            TodoField field = converter.convert(toDoField, TodoField.class);
            field.setAppId(receiveDto.getAppCode());
            todoFieldList.add(field);
        }
        //保存待办标题
        if (CollectionUtils.isNotEmpty(todoFieldList)) {
            todoFieldService.saveBatch(todoFieldList);
        }
        //推送通知给PC用户
        pushToUser(receiveDto.getAppCode(), verifyUserldList);
        return null;
    }

    @Override
    @Transactional
    public Integer approvalsNotice(MaycurTodoNoticeReq req) {
        MaycurCalloutParamsReq calloutParamsReq = req.getCalloutParams();
        //任务类型实际上只有两种，待处理和已完成，待处理即PROCESSING，
        //已完成包括COMPLETED、REJECTED、REJECTED_TO_PREVIOUS、RECALLED、FORWARD、REVERTED、CARBON_COPY这7种
        //如果待办是完成状态，删除该待办
        if (!MaycurTaskActionTypeEnum.PROCESSING.getType().equals(calloutParamsReq.getTaskActionType())) {
            List<TodoList> todoLists = selectToDoBy("", calloutParamsReq.getFormCode(), calloutParamsReq.getTaskAssignee());
            if (CollectionUtils.isEmpty(todoLists)) {
                return null;
            }
            List<Long> todoIdList = todoLists.stream().map(TodoList::getId).collect(Collectors.toList());
            todoListMapper.deleteBatchIds(todoIdList);
            //查询每刻报销的appid
            PreinstalledApplication preApp = preinstalledApplicationService.getPreByCode(MaycurCodeEnum.MAYCUR_APPROVALS.getType(), "");
            if (null == preApp) {
                return null;
            }
            //通知处理人
            pushToUser(preApp.getAppId(), calloutParamsReq.getTaskAssignee());
        }
        return null;
    }

    /**
     * 推送通知给PC用户
     *
     * @param appCode          应用id，sso配置中的appid
     * @param verifyUserldList 地端用户id
     */
    public void pushToUser(String appCode, List<String> verifyUserldList) {
        //给PC通知更新
        for (String verifyUserld : verifyUserldList) {
            try {
                MessageDO payload = new MessageDO();
                payload.setAppId(appCode);
                payload.setUserId(verifyUserld);
                payload.setType(Constants.VALID_STATUS_UNUSABLE);
                messageSendService.sendToClient(verifyUserld, payload, Constants.SEMC_TODO_USER);
            } catch (Exception e) {
                log.info("【pushToUser-sendToClient发送MQTT异常】verifyUserld：{},error：{}", verifyUserld, e);
                continue;
            }
        }
    }

    @Override
    public TodoCountResp selectTodoCount(PreSystemToDoCountReq req) {
        TodoCountResp resp = new TodoCountResp();
        resp.setToDoNum(0);
        resp.setToDoneNum(0);
        resp.setToSumNum(0);
        String verifyUserId;
        //每刻国际版
        if (MaycurCodeEnum.MAYCUR_APPROVALS.getType().equals(req.getAppCode())) {
            // 获取员工工号
            verifyUserId = this.eocService.getEmpJobNum();
        }
        else {
            //查询当前鼎捷云用户对应地端用户
            verifyUserId = iamService.queryMappingEmpId(req.getAppCode());
        }
        if (StringUtils.isBlank(verifyUserId)) {
            return resp;
        }
        LambdaQueryWrapper<TodoList> queryWrapper = new LambdaQueryWrapper<>();
        queryWrapper.eq(StringUtils.isNotBlank(req.getAppId()),TodoList::getAppId, req.getAppId());
        queryWrapper.like(StringUtils.isNotBlank(req.getQryCondition()),TodoList::getToDoData, req.getQryCondition());
        List<TodoList> todoLists = todoListMapper.selectList(queryWrapper);
        if (CollectionUtils.isEmpty(todoLists)) {
            return resp;
        }
        long toDoNum = todoLists.stream().filter(m -> Integer.valueOf(0).equals(m.getIsCompleted()) && verifyUserId.equals(m.getToDoUser())).count();
        resp.setToDoNum((int) toDoNum);
        long toDoneNum = todoLists.stream().filter(m -> Integer.valueOf(1).equals(m.getIsCompleted()) && verifyUserId.equals(m.getToDoUser())).count();
        resp.setToDoneNum((int) toDoneNum);
        resp.setToSumNum(resp.getToDoNum() + resp.getToDoneNum());
        return resp;
    }

    /**
     * 查询三方待办未处理数及未读数
     *
     * @param systemDataList 数据源信息
     * @return
     */
    @Override
    public JSONObject queryToDoCount(List<LabelSystemData> systemDataList) {
        JSONObject jsonObject = new JSONObject();
        List<AppUserDTO> userDTOList = Lists.newArrayList();

        // 三方应用列表
        List<LabelSystemData> commonSystemDataList = systemDataList.stream()
                .filter(x -> !Arrays.asList(MaycurCodeEnum.MAYCUR_CLOUD.getType(), MaycurCodeEnum.MAYCUR_APPROVALS.getType()).contains(x.getAppCode())).collect(Collectors.toList());
        if (CollectionUtils.isNotEmpty(commonSystemDataList)) {
            // 三方应用地端账号id
            userDTOList = queryVerifyUseridList(commonSystemDataList);
        }

        // 每刻应用
        List<LabelSystemData> mkSystemDataList = systemDataList.stream().filter(x -> MaycurCodeEnum.MAYCUR_APPROVALS.getType().equals(x.getAppCode())).collect(Collectors.toList());
        if (CollectionUtils.isNotEmpty(mkSystemDataList)) {
            String userId = eocService.getEmpJobNum();
            List<AppUserDTO> mkUserDTOList = mkSystemDataList.stream().map(x -> {
                AppUserDTO appUserDTO = new AppUserDTO();
                appUserDTO.setAppId(x.getAppId());
                appUserDTO.setVerifyUserId(userId);
                return appUserDTO;
            }).collect(Collectors.toList());
            userDTOList.addAll(mkUserDTOList);
        }
        if (CollectionUtils.isEmpty(userDTOList)) {
            jsonObject.put("toDoCount", 0);
            jsonObject.put("unReadCount", 0);
            return jsonObject;
        }

        // 待办列表
        List<TodoList> todoList = todoListMapper.queryToDoListByUserId(userDTOList);
        if (CollectionUtils.isEmpty(todoList)) {
            jsonObject.put("toDoCount", 0);
            jsonObject.put("unReadCount", 0);
            return jsonObject;
        }
        jsonObject.put("toDoCount", todoList.size());

        // 已读列表
        List<TodoRead> todoReadList = todoReadMapper.queryToDoReadByUserId(userDTOList);
        if (CollectionUtils.isEmpty(todoReadList)) {
            jsonObject.put("unReadCount", todoList.size());
            return jsonObject;
        }
        // 去重
        todoReadList = todoReadList.stream().collect(Collectors.collectingAndThen(Collectors.toCollection(() ->
                new TreeSet<>(Comparator.comparing(o -> o.getAppId() + ";" + o.getToDoId() + ";" + o.getToDoUser()))), ArrayList::new));

        // 通过外部待办唯一id计算交集，不在交集内的为未读数
        List<String> allTodoIdList = todoList.stream().map(TodoList::getToDoId).collect(Collectors.toList());
        List<String> readTodoIdList = todoReadList.stream().map(TodoRead::getToDoId).collect(Collectors.toList());
        List<String> respList = allTodoIdList.stream().filter(readTodoIdList::contains).collect(Collectors.toList());
        jsonObject.put("unReadCount", Math.max((allTodoIdList.size() - respList.size()), 0));
        return jsonObject;
    }

    /**
     * 查询三方用户id
     *
     * @param commonSystemDataList 应用列表
     * @return
     */
    private List<AppUserDTO> queryVerifyUseridList(List<LabelSystemData> commonSystemDataList) {
        commonSystemDataList = commonSystemDataList.stream()
                .filter(x -> StringUtils.isNotBlank(x.getAppCode())).collect(Collectors.collectingAndThen(Collectors.toCollection(() -> new TreeSet<>(Comparator.comparing(LabelSystemData::getAppCode))), ArrayList::new));
        List<AppUserDTO> appVerifyUserIdList = Lists.newArrayList();
        commonSystemDataList.forEach(x -> {
            String verifyUserId = iamService.queryMappingEmpId(x.getAppCode());
            if (StringUtils.isNotBlank(verifyUserId)) {
                AppUserDTO appUserDTO = new AppUserDTO();
                appUserDTO.setAppCode(x.getAppCode());
                appUserDTO.setAppId(x.getAppId());
                appUserDTO.setVerifyUserId(verifyUserId);
                appVerifyUserIdList.add(appUserDTO);
            }
        });
        return appVerifyUserIdList;
    }

    @Override
    public ResponseEntity<?> syncThirdTodoData(PreSystemToDoUpdateReq req) {

        List<Map<String, Object>> todoList = new ArrayList<>();
        List<Map<String, Object>> todoLists = new ArrayList<>();
        List<Map<String, Object>> todoFieldList = new ArrayList<>();

        String tenantId = Utils.getTenantId();
        String token = Utils.getUserToken();
        String appId = req.getAppId();

        //查询源应用对应的模式
        LabelSystemData labelSystemData = labelSystemDataMapper.selectById(req.getId());

        if (null == labelSystemData) {
            return ResponseEntityWrapper.wrapperOk(messageUtils.getMessage("error.message.third.data.updated"));
        }

        QueryWrapper<ThirdTodoConfig> condition2 = new QueryWrapper<>();
        condition2.eq("app_primary_id", labelSystemData.getAppCode());
        condition2.eq("app_source", labelSystemData.getDataType());
        List<ThirdTodoConfig> thirdTodoConfigList = thirdTodoConfigService.getBaseMapper().selectList(condition2);

        if (CollectionUtils.isEmpty(thirdTodoConfigList)) {
            return ResponseEntityWrapper.wrapperOk(messageUtils.getMessage("error.message.third.data.updated"));
        }
        ThirdTodoConfig thirdTodoConfig = thirdTodoConfigList.get(0);

        if (thirdTodoConfig.getValidStatus() == 0 || thirdTodoConfig.getAppAccessModel() == null) {
            return ResponseEntityWrapper.wrapperOk(messageUtils.getMessage("error.message.third.data.updated"));
        }

        RedisLock redisLock = new RedisLock(appId + tenantId);
        //分布式锁
        try {
            if (!redisLock.lock()) {
                return ResponseEntityWrapper.wrapperOk();
            }
        } catch (InterruptedException e) {
            return ResponseEntityWrapper.wrapperOk();
        }
        try {
            log.info("syncThirdTodoData start  appId:{}", appId);
            //混合云
            if (thirdTodoConfig.getAppAccessModel() == 0) {
                //默认设置中文简体

                String middleSystemName = thirdTodoConfig.getMiddleSystemName();
                String middleSystemUid = thirdTodoConfig.getMiddleSystemUid();
                Map<String, String> extHeader = new HashMap<>();
                extHeader.put("digi-userToken", token);
                extHeader.put("digi-appToken", envProperties.getAppToken());
                // 通过ESP查询三方系统的待办数据
                for (int i = 1; ; i++) {
                    Map<String, Object> parameter = new HashMap<>();
                    parameter.put("page_no", i);
                    parameter.put("page_size", Constants.TODO_MAX_SIZE);
                    parameter.put("tenant_id", tenantId);
                    Map<String, Object> result = espService.queryByEsp(middleSystemName,
                            middleSystemUid, EAIServiceNameEnum.TO_DO_LIST_QUERY.getServiceName(), extHeader, parameter, null, null
                    );
                    // 没有显示字段时，返回异常
                    if (Objects.isNull(result.get("to_do_list"))) {
                        log.info("syncThirdTodoData 混合云未查询到待办数据");
                        break;
                    }
                    todoList = (List<Map<String, Object>>) result.get("to_do_list");
                    todoFieldList = (List<Map<String, Object>>) result.get("display_field_list");
                    log.info("syncThirdTodoData  混合云待办数据:{}", todoList);
                    //插入待办数据
                    if (CollectionUtils.isNotEmpty(todoList)) {
                        todoLists.addAll(todoList);
                    }
                    if (!(Boolean) result.get("has_next")) {
                        break;
                    }
                }

            }
            //非混合云
            if (thirdTodoConfig.getAppAccessModel() == 1) {
                for (int i = 1; ; i++) {
                    String domain = thirdTodoConfig.getDomain();
                    Map<String, Object> bodyMap = new HashMap<>();
                    bodyMap.put("page_no", i);
                    bodyMap.put("page_size", Constants.TODO_MAX_SIZE);
                    bodyMap.put("tenant_id", tenantId);
                    Map<String, Object> jobData = tripartiteService.queryThirdData(domain + Constants.QUERY_TODOJOB_LIST_URL, "", bodyMap);
//                        Map<String, Object> jobData = tripartiteService.queryThirdData(domain + "/eai/mocksys/todo/queryToDoData", "", bodyMap);
                    if (null == jobData) {
                        log.info("syncThirdTodoData 非混合云未查询到待办数据");
                        break;
                    }
                    todoList = (List<Map<String, Object>>) jobData.get("to_do_list");
                    todoFieldList = (List<Map<String, Object>>) jobData.get("display_field_list");
                    log.info("syncThirdTodoData 非混合云待办数据:{}", todoList);
                    //插入待办数据
                    if (CollectionUtils.isNotEmpty(todoList)) {
                        todoLists.addAll(todoList);
                    }
                    if (!(Boolean) jobData.get("has_next")) {
                        break;
                    }
                }
            }

            //查询地中台用户id
            String verifyUserId = "";
            LabelSystemDataDto labelSystemDataDto = labelSystemDataService.getSystemDataSso(req.getId());
            if (null != labelSystemDataDto) {
                verifyUserId = iamService.queryMappingEmpId(labelSystemDataDto.getAppCode());
            }

            //处理插入数据
            handleTodoData(todoLists, todoFieldList, appId, verifyUserId);
            //删除read数据
            List<TodoRead> todoReadList = todoReadMapper.selectList(getToDoReadWrapepr(appId, verifyUserId));
            if (CollectionUtils.isNotEmpty(todoReadList) && CollectionUtils.isNotEmpty(todoLists)) {
                for (TodoRead todoRead : todoReadList) {
                    List<Map<String, Object>> result = todoLists.stream().filter(it -> it.get("to_do_id").toString().equals(todoRead.getToDoId())
                            && it.get("to_do_owner").toString().equals(todoRead.getToDoUser())).collect(Collectors.toList());
                    if (result.size() == 0) {
                        QueryWrapper deleteWrapper = new QueryWrapper<TodoRead>();
                        deleteWrapper.eq("app_id", appId);
                        deleteWrapper.eq("to_do_id", todoRead.getToDoId());
                        deleteWrapper.eq("to_do_user", todoRead.getToDoUser());
                        todoFieldMapper.delete(deleteWrapper);
                    }
                }
            }
        } catch (Exception e) {
            log.error("syncThirdTodoData error  e:{}", e);
            return ResponseEntityWrapper.wrapperFail(ErrorCodeConstant.SYSTEM_ERROR, messageUtils.getMessage("error.message.third.data.updated.error"));
        } finally {
            redisLock.unlock();
        }

        return ResponseEntityWrapper.wrapperOk(messageUtils.getMessage("error.message.third.data.updated"));
    }

    public QueryWrapper getToDoFieldWrapepr(String appId) {
        QueryWrapper queryWrapper = new QueryWrapper<TodoField>();
        //应用查询
        if (StringUtils.isNotEmpty(appId)) {
            queryWrapper.eq("app_id", appId);
        }
        return queryWrapper;
    }

    public QueryWrapper getToDoReadWrapepr(String appId, String verifyUserId) {
        QueryWrapper queryWrapper = new QueryWrapper<TodoRead>();
        //应用查询
        if (StringUtils.isNotEmpty(appId)) {
            queryWrapper.eq("app_id", appId);
        }

        if (StringUtils.isNotEmpty(verifyUserId)) {
            queryWrapper.eq("to_do_user", verifyUserId);
        }
        return queryWrapper;
    }

    public void handleTodoData(List<Map<String, Object>> todoList, List<Map<String, Object>> todoFieldList, String appId, String verifyUserId) {
        List<TodoList> todoListDtos = new ArrayList<>();
        List<TodoField> todoFieldListDtos = new ArrayList<>();
        List<TodoList> todoAllList = new ArrayList<>();
        for (Map<String, Object> map : todoList) {
            TodoList todo = new TodoList();
            String todoId = map.get("to_do_id").toString();
            String todoUser = "";
            if (map.get("to_do_owner") != null) {
                todoUser = map.get("to_do_owner").toString();
            }
            if (StringUtils.isEmpty(todoId) || StringUtils.isEmpty(todoUser)) {
                continue;
            }
            todo.setToDoSource(2);
            todo.setAppName("");
            todo.setAppId(appId);
            todo.setToDoId(todoId);
            String toDoUrl = map.get("to_do_url") == null ? "" : map.get("to_do_url").toString();
            todo.setToDoUrl(toDoUrl);
            String mobileUrl = map.get("to_do_mobile_url") == null ? "" : map.get("to_do_mobile_url").toString();
            todo.setMobileUrl(mobileUrl);
            todo.setToDoUser(todoUser);
            String toDoData = map.get("to_do_data") == null ? "" : map.get("to_do_data").toString();
            todo.setToDoData(toDoData);
            todo.setIsCompleted(0);
            todo.setIsRead(0);
            String mobileMainTitle = map.get("mobile_main_title") == null ? "" : map.get("mobile_main_title").toString();
            todo.setMobileMainTitle(mobileMainTitle);
            String mobileSubTitle = map.get("mobile_sub_title") == null ? "" : map.get("mobile_sub_title").toString();
            todo.setMobileSubTitle(mobileSubTitle);
            String mobileLabel = map.get("mobile_label") == null ? "" : map.get("mobile_label").toString();
            todo.setMobileLabel(mobileLabel);
            todo.setTenantId("");
            todo.setCreateTime(DateUtils.getNowTime(DateUtils.DATE_TIME_NORMAL_FORMATTER));
            //记录所有待办
            todoAllList.add(todo);

            //筛选当前登陆人待办数据
            if ((!StringUtils.isEmpty(verifyUserId) && !todoUser.equals(verifyUserId))) {
                continue;
            }
            todoListDtos.add(todo);
        }
        //保存待办标题
        for (Map<String, Object> fieldmap : todoFieldList) {
            TodoField field = new TodoField();
            field.setAppId(appId);
            String displayName = fieldmap.get("display_name") == null ? "" : fieldmap.get("display_name").toString();
            field.setDisplayName(displayName);
            Integer displayOrder = fieldmap.get("display_order") == null ? null : Integer.parseInt(fieldmap.get("display_order").toString());
            field.setDisplayOrder(displayOrder);
            String bindingDataKey = fieldmap.get("binding_data_key") == null ? "" : fieldmap.get("binding_data_key").toString();
            field.setBindingDataKey(bindingDataKey);
            field.setCreateTime(DateUtils.getNowTime(DateUtils.DATE_TIME_NORMAL_FORMATTER));
            todoFieldListDtos.add(field);
        }

        //删除待办数据
        todoListMapper.delete(getToDoWrapeprByUserId(appId, new ArrayList<>(), verifyUserId));
        //保存待办数据
        if (CollectionUtils.isNotEmpty(todoListDtos)) {
            //超过1000笔分批插入
            if (todoListDtos.size() > 1000) {
                for (int i = 0; i < todoListDtos.size(); i += 1000) {
                    List<TodoList> batchEntities = todoListDtos.subList(i, Math.min(i + 1000, todoListDtos.size()));
                    this.saveBatch(batchEntities);
                }
            } else {
                this.saveBatch(todoListDtos);
            }
        } else {
            //当前人没有归户，则将所有待办消息再新增回数据库
            if (StringUtils.isEmpty(verifyUserId) && CollectionUtils.isNotEmpty(todoAllList)) {
                //超过1000笔分批插入
                if (todoAllList.size() > 1000) {
                    for (int i = 0; i < todoAllList.size(); i += 1000) {
                        List<TodoList> batchEntities = todoAllList.subList(i, Math.min(i + 1000, todoAllList.size()));
                        this.saveBatch(batchEntities);
                    }
                } else {
                    this.saveBatch(todoAllList);
                }
            }
        }


//        //删除该应用展示的标题数据
        todoFieldMapper.delete(getToDoFieldWrapepr(appId));
        //保存待办标题
        if (CollectionUtils.isNotEmpty(todoFieldListDtos)) {
            todoFieldService.saveBatch(todoFieldListDtos);
        }
    }

    /**
     * 推送通知给PC用户
     *
     * @param appCode          应用id，sso配置中的appid
     * @param verifyUserldList 地端用户id
     */
    public void pushNoticeToUser(String appCode, List<String> verifyUserldList) {
        //给PC通知更新
        for (String verifyUserld : verifyUserldList) {
            try {
                MessageDO payload = new MessageDO();
                payload.setAppId(appCode);
                payload.setUserId(verifyUserld);
                payload.setType(Constants.VALID_STATUS_UNUSABLE);
                messageSendService.sendToClient(verifyUserld, payload, Constants.SEMC_TODO_USER);
            } catch (Exception e) {
                log.info("【pushToUser-sendToClient发送MQTT异常】verifyUserld：{},error：{}", verifyUserld, e);
            }
        }
    }

    /**
     * 获取地端用户ID
     * @param appCode 应用编码
     * @return
     */
    public String getVerifyUserId(String appCode) {
        String verifyUserId = "";
        if (MaycurCodeEnum.MAYCUR_APPROVALS.getType().equals(appCode)) {
            // 获取员工工号
            verifyUserId = this.eocService.getEmpJobNum();
        } else {
            //查询当前鼎捷云用户对应地端用户
            verifyUserId = iamService.queryMappingEmpId(appCode);
        }
        return verifyUserId;
    }

    @Override
    public List<LabelSystemData> searchTodoApp(PreSystemToDoSearchAppReq req) {
        //查询三方待办组件下数据源的应用系统
        QueryLabelSystemReq querySystemReq = new QueryLabelSystemReq();
        querySystemReq.setDataType(Constants.DataTypeEnum.TYPE_THIRD_TODO.getVal());
        //搜索条件是否为空
        boolean isEmptyQuery = StringUtils.isBlank(req.getQryCondition());
        List<Long> dataIds = labelSystemSourceService.selectPreSystem(querySystemReq.getDataType(), Constants.LabelTypeEnum.SYSTEM_PRE.getVal());
        List<LabelSystemData> oneLabelSystemDataList = labelSystemDataService.queryDataInfoBy(dataIds, isEmptyQuery);
        if (CollectionUtils.isEmpty(oneLabelSystemDataList)) {
            log.info("三方待办搜索应用对应的数据源为空");
            return Lists.newArrayList();
        }
        List<LabelSystemData> labelSystemDataList = new ArrayList<>();
        //条件为空的话就不调用底层三方待办查询
        if (isEmptyQuery) {
            labelSystemDataList = oneLabelSystemDataList;
        } else {
            Map<String, LabelSystemData> oneLabelSystemDataMap = oneLabelSystemDataList.stream().collect(Collectors.toMap(LabelSystemData::getAppId, Function.identity(), (a, b) -> a));
            LambdaQueryWrapper<TodoList> queryWrapper = new LambdaQueryWrapper<>();
            queryWrapper.in(TodoList::getAppId, oneLabelSystemDataMap.keySet());
            queryWrapper.like(StringUtils.isNotBlank(req.getQryCondition()), TodoList::getToDoData, req.getQryCondition());
            queryWrapper.groupBy(TodoList::getAppId);
            List<TodoList> todoLists = todoListMapper.selectList(queryWrapper);
            if (CollectionUtils.isEmpty(todoLists)) {
                log.info("三方待办搜索应用结果为空");
                return Lists.newArrayList();
            }

            for (TodoList todoList : todoLists) {
                LabelSystemData labelSystemData = oneLabelSystemDataMap.get(todoList.getAppId());
                if (labelSystemData != null) {
                    labelSystemDataList.add(labelSystemData);
                }
            }
            //查询每个应用对应的count（后续可以考虑用异步处理）
            for (LabelSystemData labelSystemData : labelSystemDataList) {
                PreSystemToDoCountReq countReq = new PreSystemToDoCountReq();
                countReq.setAppId(labelSystemData.getAppId());
                countReq.setAppName(labelSystemData.getAppCode());
                countReq.setAppCode(labelSystemData.getAppCode());
                countReq.setId(labelSystemData.getId());
                countReq.setQryCondition(req.getQryCondition());
                TodoCountResp todoCountResp = selectTodoCount(countReq);
                labelSystemData.setToDoNum(todoCountResp.getToDoNum());
                labelSystemData.setToDoneNum(todoCountResp.getToDoneNum());
                labelSystemData.setToSumNum(todoCountResp.getToSumNum());
            }
            if (CollectionUtils.isNotEmpty(labelSystemDataList)){
                Iterator<LabelSystemData> iterator = labelSystemDataList.iterator();
                while (iterator.hasNext()){
                    LabelSystemData next = iterator.next();
                    if (Objects.equals(0,next.getToDoNum()) && Objects.equals(0,next.getToDoneNum()) && Objects.equals(0,next.getToSumNum())){
                        iterator.remove();
                    }
                }
            }

        }
        List<LabelSystemData> labelSystemData = todoAppSortService.sortLabelSystemData(labelSystemDataList);
        if (org.apache.commons.collections4.CollectionUtils.isEmpty(labelSystemData)){
            return Collections.emptyList();
        }
        List<LabelSystemData> distinctLabelSystemDatas = labelSystemData.stream()
                .collect(Collectors.collectingAndThen(
                        Collectors.toCollection(() ->
                                new TreeSet<>(Comparator.comparing(LabelSystemData::getAppCode))),
                        ArrayList::new
                ));
        return distinctLabelSystemDatas;
    }

    public boolean exists(Wrapper<TodoField> queryWrapper) {
        Long count = todoFieldMapper.selectCount(queryWrapper);
        return null != count && count > 0;
    }
}
