package com.digiwin.athena.semc.service.sso.impl;


import com.google.common.collect.Lists;
import com.google.common.collect.Maps;

import com.alibaba.fastjson.JSONObject;
import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.baomidou.mybatisplus.core.toolkit.CollectionUtils;
import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.digiwin.athena.appcore.auth.AppAuthContextHolder;
import com.digiwin.athena.appcore.auth.domain.AuthoredUser;
import com.digiwin.athena.semc.common.Constants;
import com.digiwin.athena.semc.common.ErpSsoConstants;
import com.digiwin.athena.semc.common.enums.ErpAppNameEnum;
import com.digiwin.athena.semc.dto.erpsso.AddSsoInfoReq;
import com.digiwin.athena.semc.dto.erpsso.QueryDisplayReq;
import com.digiwin.athena.semc.dto.erpsso.QuerySsoListReq;
import com.digiwin.athena.semc.entity.sso.ErpSsoInfo;
import com.digiwin.athena.semc.entity.sso.ErpSsoParam;
import com.digiwin.athena.semc.mapper.sso.ErpSsoInfoMapper;
import com.digiwin.athena.semc.mapper.sso.ErpSsoParamMapper;
import com.digiwin.athena.semc.service.sso.IErpSsoInfoService;

import org.apache.commons.collections.MapUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.i18n.LocaleContextHolder;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.nio.charset.StandardCharsets;
import java.util.Base64;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.Function;
import java.util.stream.Collectors;

/**
 * @author zhangtuo
 * @since 2022-10-28
 */
@Service
public class ErpSsoInfoServiceImpl extends ServiceImpl<ErpSsoInfoMapper, ErpSsoInfo> implements IErpSsoInfoService {

    @Autowired
    ErpSsoParamMapper erpSsoParamMapper;

    @Autowired
    ErpSsoInfoMapper erpSsoInfoMapper;
    /**
     * 查询稳态配置列表
     *
     * @param tenantId 租户id
     * @return
     */
    public List<ErpSsoInfo> querySsoList(String tenantId, Integer protocolType) {
        // 默认带tenantId查询，全局组件配置的
        QuerySsoListReq querySsoListReq = new QuerySsoListReq();
        if (StringUtils.isNotBlank(tenantId)) {
            querySsoListReq.setTenantId(tenantId);
        }
        querySsoListReq.setProtocolType(protocolType);
        List<ErpSsoInfo> ssoInfoList = erpSsoInfoMapper.querySsoListByScope(querySsoListReq);
        if (CollectionUtils.isEmpty(ssoInfoList)) {
            return ssoInfoList;
        }
        List<ErpSsoParam> ssoParamList = erpSsoParamMapper.selectList(Wrappers.emptyWrapper());
        if (CollectionUtils.isEmpty(ssoParamList)) {
            return ssoInfoList;
        }
        Map<Long, List<ErpSsoParam>> infoIdMap = ssoParamList.stream().collect(Collectors.groupingBy(ErpSsoParam::getSsoInfoId));
        ssoInfoList.forEach(x -> {
            List<ErpSsoParam> paramList = infoIdMap.get(x.getId());
            if (CollectionUtils.isNotEmpty(paramList)) {
                HashMap<String, Object> paramMap = Maps.newHashMap();
                paramList.forEach(y -> paramMap.put(y.getParam(), y.getValue()));
                if (MapUtils.isEmpty(x.getExt())) {
                    x.setExt(paramMap);
                } else {
                    x.getExt().putAll(paramMap);
                }
            }
        });
        return ssoInfoList;
    }

    /**
     * 查询code、name、appToken已存在
     *
     * @param id       应用主键id
     * @param appCode  应用code
     * @param appName  应用名称
     * @param appToken 应用token
     * @return 返回
     */
    @Override
    public List<ErpSsoInfo> existByNameOrAppToken(Long id, String appCode, String appName, String appToken) {
        QueryWrapper<ErpSsoInfo> infoQueryWrapper = new QueryWrapper<>();
        // 修改时排除当前配置
        if (id != null) {
            infoQueryWrapper.ne("id", id);
        }
        if (StringUtils.isBlank(appToken)) {
            infoQueryWrapper.and(x -> x.eq("name", appName)
                    .or().eq("code", appCode));
        } else {
            infoQueryWrapper.and(x -> x.eq("name", appName)
                    .or().eq("code", appCode)
                    .or().eq("app_token", appToken));
        }
        return erpSsoInfoMapper.selectList(infoQueryWrapper);
    }

    /**
     * 新增或修改稳态配置列表
     *
     * @param addSsoInfoReq 新增实体类
     * @return
     */
    @Override
    @Transactional
    public Long saveOrUpdateSsoInfo(AddSsoInfoReq addSsoInfoReq) {
        String tenantId = AppAuthContextHolder.getContext().getAuthoredUser().getTenantId();
        ErpSsoInfo erpSsoInfo = new ErpSsoInfo();
        erpSsoInfo.setId(addSsoInfoReq.getId());
        erpSsoInfo.setAppSid(addSsoInfoReq.getAppSid());
        erpSsoInfo.setTenantId(tenantId);
        erpSsoInfo.setCode(addSsoInfoReq.getCode());
        erpSsoInfo.setName(addSsoInfoReq.getName());
        erpSsoInfo.setAppDesc(addSsoInfoReq.getDesc());
        erpSsoInfo.setLocationType(addSsoInfoReq.getLocationType());
        erpSsoInfo.setPrefix(addSsoInfoReq.getPrefix());
        erpSsoInfo.setClientLocation(addSsoInfoReq.getClientLocation());
        erpSsoInfo.setProtocolType(addSsoInfoReq.getProtocolType());
        erpSsoInfo.setCasServerUrl(addSsoInfoReq.getCasServerUrl());
        erpSsoInfo.setBase64Encode(addSsoInfoReq.getBase64Encode() == null ? Constants.ERP_URL_NO_NEEDBASE64 : addSsoInfoReq.getBase64Encode());
        erpSsoInfo.setAppToken(addSsoInfoReq.getAppToken());
        erpSsoInfo.setCallBackUrl(String.format(Constants.CS_APP_CALLBACK_URL, addSsoInfoReq.getCode()));
        erpSsoInfo.setAppId(addSsoInfoReq.getAppId());
        erpSsoInfo.setAppSecret(addSsoInfoReq.getAppSecret());
        erpSsoInfo.setUserBindFlag(addSsoInfoReq.getUserBindFlag());
        // 新增
        if (erpSsoInfo.getId() == null) {
            erpSsoInfoMapper.insert(erpSsoInfo);
            saveSsoParam(addSsoInfoReq, tenantId, erpSsoInfo.getId());
        } else {    // 修改
            erpSsoInfoMapper.updateById(erpSsoInfo);
            deleteErpSsoParams(erpSsoInfo.getId());
            saveSsoParam(addSsoInfoReq, tenantId, erpSsoInfo.getId());
        }
        // 保存动态参数
        return erpSsoInfo.getId();
    }

    /**
     * 新增稳态参数列表
     *
     * @param addSsoInfoReq 请求实体
     * @param tenantId      租户id
     * @param ssoId         配置id
     */
    private void saveSsoParam(AddSsoInfoReq addSsoInfoReq, String tenantId, Long ssoId) {
        List<ErpSsoParam> erpSsoParamList = Lists.newArrayList();
        JSONObject jsonObject = JSONObject.parseObject(addSsoInfoReq.getExt());
        if (jsonObject != null) {
            jsonObject.forEach((k, v) -> {
                ErpSsoParam erpSsoParam = new ErpSsoParam();
                erpSsoParam.setSsoInfoId(ssoId);
                erpSsoParam.setTenantId(tenantId);
                erpSsoParam.setParam(k);
                erpSsoParam.setValue(String.valueOf(v));
                erpSsoParamList.add(erpSsoParam);
            });
            erpSsoParamMapper.batchInsertSsoParamList(erpSsoParamList);
        }
    }

    /**
     * 删除稳态参数信息
     *
     * @param ssoId 配置id
     */
    @Override
    public void deleteErpSsoParams(Long ssoId) {
        ErpSsoParam param = new ErpSsoParam();
        param.setSsoInfoId(ssoId);
        QueryWrapper<ErpSsoParam> queryWrapper = new QueryWrapper<>();
        queryWrapper.setEntity(param);
        erpSsoParamMapper.delete(queryWrapper);
    }

    /**
     * 根据主键id查询配置信息
     *
     * @param ssoId 稳态配置id
     * @return
     */
    @Override
    public ErpSsoInfo querySsoInfo(Long ssoId) {
        QueryWrapper<ErpSsoInfo> infoQueryWrapper = new QueryWrapper<>();
        infoQueryWrapper.eq("id", ssoId);
        ErpSsoInfo erpSsoInfo = erpSsoInfoMapper.selectOne(infoQueryWrapper);

        // 动态参数
        QueryWrapper<ErpSsoParam> paramWrapper = new QueryWrapper<>();
        paramWrapper.eq("sso_info_id", ssoId);
        List<ErpSsoParam> ssoParamList = erpSsoParamMapper.selectList(paramWrapper);
        if (CollectionUtils.isNotEmpty(ssoParamList)) {
            HashMap<String, Object> paramMap = Maps.newHashMap();
            ssoParamList.forEach(y -> paramMap.put(y.getParam(), y.getValue()));
            erpSsoInfo.setExt(paramMap);
        }
        return erpSsoInfo;
    }

    /**
     * 修改生效状态
     *
     * @param ssoInfoId 配置id
     * @param status    状态
     * @return
     */
    @Override
    public int updateSsoStatus(Long ssoInfoId, Integer status) {
        return this.baseMapper.updateStatusById(ssoInfoId, status, AppAuthContextHolder.getContext().getAuthoredUser().getUserId());
    }

    /**
     * 更新应用的归户模式
     *
     * @param ssoIdList    应用id列表
     * @param userBindFlag 归户模式
     * @return
     */
    @Override
    public void updateBindFlag(List<Long> ssoIdList, Integer userBindFlag) {
        this.baseMapper.updateBindFlag(ssoIdList, userBindFlag, AppAuthContextHolder.getContext().getAuthoredUser().getUserId());
    }

    /**
     * 查询所有稳态的应用列表
     *
     * @param querySsoListReq 请求实体
     * @return
     */
    @Override
    public List<ErpSsoInfo> querySsoListByScope(QuerySsoListReq querySsoListReq) {
        return this.baseMapper.querySsoListByScope(querySsoListReq);
    }

    /**
     * 根据appId查询租户
     *
     * @param appId 应用id
     * @return 返回
     */
    @Override
    public List<String> queryTenantIdByAppId(String appId){
        return this.baseMapper.queryTenantIdByAppId(appId);
    }

    @Override
    public Map<Long, ErpSsoInfo> getErpInfoMap(List<String> csIdList) {
        Map<Long, ErpSsoInfo> erpInfoMap = new HashMap<>();
        if (org.apache.commons.collections.CollectionUtils.isEmpty(csIdList)) {
            return erpInfoMap;
        }
        QuerySsoListReq querySsoListReq = new QuerySsoListReq();
        List<Long> idList = csIdList.stream().map(Long::parseLong).collect(Collectors.toList());
        querySsoListReq.setIdList(idList);
        List<ErpSsoInfo> erpInfoList = erpSsoInfoMapper.querySsoListByScope(querySsoListReq);
        erpInfoMap = erpInfoList.stream().collect(Collectors.toMap(ErpSsoInfo::getId, Function.identity(), (a, b) -> a));
        return erpInfoMap;
    }

    /*
    查询租户下所有产品，并拼装url，参数部分包括固定的appToken、userToken，以及可变部分存储于ErpSsoParam中，参数部分支持base64加密
    最终拼装格式：prefix+clientLocation+suffix+param1=value1&param2=value2，如digiwine10sso://appToken=xxx&userToken=xxxxx&aaa=xxx
    以上废弃，更新如下：
    查询租户下所有产品，并拼装url，参数部分包括固定的appToken、callBackUrl，以及可变部分存储于ErpSsoParam中，参数部分支持base64加密
     最终拼装格式：prefix+clientLocation+suffix+param1=value1&param2=value2，如digiwine10sso://appToken=xxx&callBackUrl=xxx&appId=xxx&aaa=xxx&code=xxx
     callBackUrl，appId用于前端调用IAM获取授权码code，拼接在链接后面
     授权码code，appToken，callBackUrl，用于三方应用后续调用IAM获取accessToken使用
     */
    @Override
    public List<ErpSsoInfo> erpSsoInfoWithParams() {
        QueryWrapper<ErpSsoInfo> queryWrapper = new QueryWrapper<>();
        //只查询有效链接
        queryWrapper.eq("valid_status", Constants.VALID_STATUS_ENABLE);
        List<ErpSsoInfo> infos = erpSsoInfoMapper.selectList(queryWrapper);
        List<ErpSsoParam> params = erpSsoParamMapper.selectList(Wrappers.emptyWrapper());
        for (ErpSsoInfo info : infos) {
            // 生成_fullpath
            generateFullPath(info, params);
        }
        return infos;
    }

    /*
  查询租户下指定产品，并拼装url，参数部分包括固定的appToken、callBackUrl，以及可变部分存储于ErpSsoParam中，参数部分支持base64加密
   最终拼装格式：prefix+clientLocation+suffix+param1=value1&param2=value2，如digiwine10sso://appToken=xxx&callBackUrl=xxx&appId=xxx&aaa=xxx&code=xxx
   callBackUrl，appId用于调用IAM获取授权码code，拼接在链接后面
   授权码code，appToken，callBackUrl，用于三方应用后续调用IAM获取accessToken使用
   */
    @Override
    public ErpSsoInfo erpSsoInfoWithParams(AuthoredUser user, QueryDisplayReq req) {
        QueryWrapper<ErpSsoInfo> queryWrapper = new QueryWrapper<>();
        //只查询有效链接
        queryWrapper.eq("valid_status", Constants.VALID_STATUS_ENABLE);
        queryWrapper.eq("code", req.getErpAppName());
        // 1、查询erp_sso_info
        List<ErpSsoInfo> infos = erpSsoInfoMapper.selectList(queryWrapper);
        if(CollectionUtils.isEmpty(infos)){
            return null;
        }
        ErpSsoInfo info = infos.get(0);
        QueryWrapper<ErpSsoParam> queryParamWrapper = new QueryWrapper<>();
        queryParamWrapper.eq("sso_info_id", info.getId());

        // 2、查询erp_sso_param
        List<ErpSsoParam> params = erpSsoParamMapper.selectList(queryParamWrapper);

        // E10的 报表、公司别key需要做转换处理
        if(ErpAppNameEnum.E10.getValue().equalsIgnoreCase(req.getErpAppName())){
            preDealForE10(req.getDynamicParams());
        }

        // 3、添加动态参数
        addDynamicParams(info,params,req.getDynamicParams());

        // 4、生成_fullpath
        generateFullPath(info, params);
        return info;
    }

    /**
     * E10的 报表、公司别key需要做转换处理
     * @param dynamicParams
     */
    private void preDealForE10(Map<String,String> dynamicParams){
        Map<String,String> params = new HashMap<>();
        Iterator<Map.Entry<String, String>> iterator = dynamicParams.entrySet().iterator();
        while (iterator.hasNext()){
            Map.Entry<String, String> entry = iterator.next();
            if(ErpSsoConstants.DYNAMIC_PARAM_PRID.equalsIgnoreCase(entry.getKey())){
                iterator.remove();
                params.put(ErpSsoConstants.E10_PROGRAM_ID, entry.getValue());
            }else if(ErpSsoConstants.DYNAMIC_PARAM_COMPANY_ID.equalsIgnoreCase(entry.getKey())){
                iterator.remove();
                params.put(ErpSsoConstants.E10_COMPANY_ID, entry.getValue());
            }
        }

        if(CollectionUtils.isNotEmpty(params)){
            dynamicParams.putAll(params);
        }
    }

    /**
     * 添加动态参数：如果存在则替换，如果不存在则新增
     * @param info
     * @param params
     * @param dynamicParams
     */
    private void addDynamicParams(ErpSsoInfo info,List<ErpSsoParam> params, Map<String,String> dynamicParams){
        if(CollectionUtils.isEmpty(dynamicParams)){
            return;
        }

        dynamicParams.entrySet().forEach(dynamicPara ->{
            Optional<ErpSsoParam> paramOptional = params.stream().filter(param -> param.getParam().equalsIgnoreCase(dynamicPara.getKey())).findFirst();
            if(paramOptional.isPresent()){
                // 如果存在则替换
                paramOptional.get().setValue(dynamicPara.getValue());
            } else {
                // 如果不存在则新增
                ErpSsoParam param = new ErpSsoParam();
                param.setSsoInfoId(info.getId());
                param.setParam(dynamicPara.getKey());
                param.setValue(dynamicPara.getValue());
                params.add(param);
            }
        });
    }

    /**
     * 生成_fullpath
     *
     * @param info
     * @param params
     */
    private void generateFullPath(ErpSsoInfo info, List<ErpSsoParam> params){
        if (info.getProtocolType().equals(1)) {
            StringBuilder stringBuilder = new StringBuilder();
            StringBuilder stringBuilder2 = new StringBuilder();
            stringBuilder.append(StringUtils.stripToEmpty(info.getPrefix()))
                    .append(StringUtils.stripToEmpty(info.getClientLocation()))
                    .append(StringUtils.stripToEmpty(info.getSuffix()));
            if (info.getLocationType() != null && Constants.PRESCRIBED_ROUTE.equals(info.getLocationType())) {
                stringBuilder.append("?");
            }
            stringBuilder2.append("appToken").append("=").append(info.getAppToken()).append("&");
            stringBuilder2.append("curLocale=").append(LocaleContextHolder.getLocale()).append("&");
            stringBuilder2.append("appCode=").append(info.getCode()).append("&");
            if (StringUtils.isBlank(info.getAppId())) { // 兼容已集成的应用
                stringBuilder2.append("userToken").append("=").append(AppAuthContextHolder.getContext().getAuthoredUser().getToken());
            } else {
                stringBuilder2.append("tenantId").append("=").append(AppAuthContextHolder.getContext().getAuthoredUser().getTenantId()).append("&");
                stringBuilder2.append("callBackUrl").append("=").append(info.getCallBackUrl()).append("&")
                        .append("appId").append("=").append(info.getAppId()).append("&").append("bindMode").append("=").append(info.getUserBindFlag());
            }
            // 无需归户只需要用userToken，适用于鼎捷云账套体系
            if (Constants.BindFlagEnum.NO.getFlag().equals(info.getUserBindFlag())) {
                stringBuilder2.append("userToken").append("=").append(AppAuthContextHolder.getContext().getAuthoredUser().getToken());
            }
            for (ErpSsoParam param : params) {
                if (Objects.equals(info.getId(), param.getSsoInfoId()) && null != param.getParam() && null != param.getValue()) {
                    stringBuilder2.append("&");
                    stringBuilder2.append(param.getParam()).append("=").append(param.getValue());
                }
            }
            if (null == info.getExt()) {
                info.setExt(new HashMap<>());
            }
            String str1 = stringBuilder.toString();
            String str2 = stringBuilder2.toString();
            // 是否需要安全的base64加密
            if (Constants.ERP_URL_NEEDBASE64.equals(info.getBase64Encode())) {
                str2 = Base64.getEncoder().encodeToString(str2.getBytes(StandardCharsets.UTF_8));
            }
            String fullpath = str1 + str2;
            info.getExt().put("_fullpath", fullpath);
        }
        if (info.getProtocolType().equals(2)) {
            if (null == info.getExt()) {
                info.setExt(new HashMap<>());
            }
            info.getExt().put("_fullpath", info.getCallBackUrl());
        }
        if (info.getProtocolType().equals(3)) {
            if (null == info.getExt()) {
                info.setExt(new HashMap<>());
            }
            info.getExt().put("_fullpath", info.getCallBackUrl());
        }
    }
}
