package com.digiwin.athena.semc.service.sso.impl;


import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.digiwin.athena.appcore.auth.AppAuthContextHolder;
import com.digiwin.athena.semc.common.Constants;
import com.digiwin.athena.semc.entity.sso.MobileSsoInfo;
import com.digiwin.athena.semc.mapper.sso.MobileSsoInfoMapper;
import com.digiwin.athena.semc.service.sso.IMobileSsoInfoService;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.ObjectUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.i18n.LocaleContextHolder;
import org.springframework.stereotype.Service;

import java.util.List;

/**
 * 移动端SSO应用配置
 *
 * @author sungq
 * @since 2024-10-24
 */
@Service
public class MobileSsoInfoServiceImpl extends ServiceImpl<MobileSsoInfoMapper, MobileSsoInfo> implements IMobileSsoInfoService {

    @Autowired
    MobileSsoInfoMapper mobileSsoInfoMapper;

    /**
     * 校验应用是否存在
     *
     * @param mobileSsoInfo 请求实体
     * @return 返回
     */
    @Override
    public List<MobileSsoInfo> existMobileApp(MobileSsoInfo mobileSsoInfo) {
        QueryWrapper<MobileSsoInfo> queryWrapper = new QueryWrapper<>();
        // 修改时排除当前配置
        if (ObjectUtils.isNotEmpty(mobileSsoInfo.getId())) {
            queryWrapper.ne("id", mobileSsoInfo.getId());
        }
        if (StringUtils.isBlank(mobileSsoInfo.getAppToken())) {
            queryWrapper.and(x -> x.eq("app_name", mobileSsoInfo.getAppName())
                    .or().eq("app_code", mobileSsoInfo.getAppCode()));
        } else {
            queryWrapper.and(x -> x.eq("app_name", mobileSsoInfo.getAppName())
                    .or().eq("app_code", mobileSsoInfo.getAppCode())
                    .or().eq("app_token", mobileSsoInfo.getAppToken()));
        }
        return mobileSsoInfoMapper.selectList(queryWrapper);
    }

    /**
     * 保存应用配置
     *
     * @param mobileSsoInfoReq 请求入参
     * @return 返回
     */
    @Override
    public Long saveMobileSso(MobileSsoInfo mobileSsoInfoReq) {
        mobileSsoInfoReq.setTenantId(AppAuthContextHolder.getContext().getAuthoredUser().getTenantId());
        // 新增
        if (ObjectUtils.isEmpty(mobileSsoInfoReq.getId())) {
            mobileSsoInfoMapper.insert(mobileSsoInfoReq);
        } else {
            mobileSsoInfoMapper.updateById(mobileSsoInfoReq);
        }
        return mobileSsoInfoReq.getId();
    }

    /**
     * 查询应用配置
     *
     * @param id 主键id
     * @return 返回
     */
    @Override
    public MobileSsoInfo queryMobileSsoInfo(Long id) {
        QueryWrapper<MobileSsoInfo> infoQueryWrapper = new QueryWrapper<>();
        infoQueryWrapper.eq("id", id);
        return mobileSsoInfoMapper.selectOne(infoQueryWrapper);
    }

    /**
     * 构建移动端应用SSO链接
     *
     * @return 返回
     */
    @Override
    public List<MobileSsoInfo> queryMobileLinkSsoList(Long id) {
        QueryWrapper<MobileSsoInfo> wrapper = new QueryWrapper<>();
        if (ObjectUtils.isNotEmpty(id)) {
            wrapper.eq("id", id);
        }
        List<MobileSsoInfo> mobileSsoInfoList = mobileSsoInfoMapper.selectList(wrapper);
        if (CollectionUtils.isEmpty(mobileSsoInfoList)) {
            return mobileSsoInfoList;
        }

        for (MobileSsoInfo info : mobileSsoInfoList) {
            StringBuilder stringBuilder = new StringBuilder();
            stringBuilder.append(StringUtils.stripToEmpty(info.getCallBackUrl()));
            stringBuilder.append(info.getCallBackUrl().contains("?") ? "&" : "?");
            stringBuilder.append("appToken").append("=").append(info.getAppToken()).append("&");
            stringBuilder.append("curLocale=").append(LocaleContextHolder.getLocale()).append("&");
            stringBuilder.append("appCode=").append(info.getAppCode()).append("&");
            stringBuilder.append("tenantId").append("=").append(AppAuthContextHolder.getContext().getAuthoredUser().getTenantId()).append("&");
            stringBuilder.append("appId").append("=").append(info.getAppId()).append("&");
            stringBuilder.append("bindMode").append("=").append(info.getUserBindFlag());
            if (Constants.BindFlagEnum.NO.getFlag().equals(info.getUserBindFlag())) {
                stringBuilder.append("&").append("userToken").append("=").append(AppAuthContextHolder.getContext().getAuthoredUser().getToken());
            }
            info.setMobileLinkUrl(stringBuilder.toString());
        }
        return mobileSsoInfoList;
    }
}
