package com.digiwin.athena.semc.service.sso.impl;


import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.digiwin.athena.appcore.auth.AppAuthContextHolder;
import com.digiwin.athena.semc.common.Constants;
import com.digiwin.athena.semc.common.enums.ProtocolTypeEnum;
import com.digiwin.athena.semc.dto.erpsso.QuerySsoListReq;
import com.digiwin.athena.semc.dto.erpsso.RegisterDTO;
import com.digiwin.athena.semc.dto.erpsso.ThirdSsoInfoDto;
import com.digiwin.athena.semc.entity.sso.ThirdSsoInfo;
import com.digiwin.athena.semc.mapper.sso.ThirdSsoInfoMapper;
import com.digiwin.athena.semc.proxy.iam.service.IamService;
import com.digiwin.athena.semc.service.sso.IThirdSsoInfoService;

import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.i18n.LocaleContextHolder;
import org.springframework.stereotype.Service;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.function.Function;
import java.util.stream.Collectors;

import javax.annotation.Resource;

import io.github.linpeilie.Converter;
import lombok.extern.slf4j.Slf4j;

/**
 * @author zhangtuo
 * @since 2022-10-28
 */
@Service
@Slf4j
public class ThirdSsoInfoServiceImpl extends ServiceImpl<ThirdSsoInfoMapper, ThirdSsoInfo> implements IThirdSsoInfoService {

    @Resource
    private IamService iamService;

    @Autowired
    ThirdSsoInfoMapper thirdSsoInfoMapper;

    @Resource
    private Converter converter;

    /**
     * 查询code、name、appToken已存在
     *
     * @param id       应用主键id
     * @param appCode  应用code
     * @param appName  应用名称
     * @param appToken 应用token
     * @return 返回
     */
    @Override
    public List<ThirdSsoInfo> existThirdSso(Long id, String appCode, String appName, String appToken) {
        QueryWrapper<ThirdSsoInfo> infoQueryWrapper = new QueryWrapper<>();
        // 修改时排除当前配置
        if (id != null) {
            infoQueryWrapper.ne("id", id);
        }
        if (StringUtils.isBlank(appToken)) {
            infoQueryWrapper.and(x -> x.eq("app_name", appName)
                    .or().eq("app_code", appCode));
        } else {
            infoQueryWrapper.and(x -> x.eq("app_name", appName)
                    .or().eq("app_code", appCode)
                    .or().eq("app_token", appToken));
        }
        return thirdSsoInfoMapper.selectList(infoQueryWrapper);
    }

    /**
     * 新增或修改配置信息
     *
     * @param thirdSsoInfo 实体类
     * @return
     */
    @Override
    public Long saveOrUpdateThirdSso(ThirdSsoInfo thirdSsoInfo) {
        thirdSsoInfo.setTenantId(AppAuthContextHolder.getContext().getAuthoredUser().getTenantId());
        // 新增
        if (thirdSsoInfo.getId() == null) {
            thirdSsoInfoMapper.insert(thirdSsoInfo);
        } else {    // 修改
            thirdSsoInfoMapper.updateById(thirdSsoInfo);
        }
        return thirdSsoInfo.getId();
    }

    /**
     * 根据主键id查询配置信息
     *
     * @param ssoId 配置id
     * @return
     */
    @Override
    public ThirdSsoInfo queryThirdSsoInfo(Long ssoId) {
        QueryWrapper<ThirdSsoInfo> infoQueryWrapper = new QueryWrapper<>();
        infoQueryWrapper.eq("id", ssoId);
        return thirdSsoInfoMapper.selectOne(infoQueryWrapper);
    }

    /**
     * 修改生效状态
     *
     * @param ssoInfoId 配置id
     * @param status    状态
     * @return
     */
    @Override
    public int updateThirdSsoStatus(Long ssoInfoId, Integer status) {
        return this.baseMapper.updateThirdStatusById(ssoInfoId, status, AppAuthContextHolder.getContext().getAuthoredUser().getUserId());
    }

    /**
     * 更新应用的归户模式
     *
     * @param ssoIdList    应用id列表
     * @param userBindFlag 归户模式
     * @return
     */
    @Override
    public void updateBindFlag(List<Long> ssoIdList, Integer userBindFlag) {
        this.baseMapper.updateBindFlag(ssoIdList, userBindFlag, AppAuthContextHolder.getContext().getAuthoredUser().getUserId());
    }

    /**
     * 注册应用
     *
     * @param registerDTO 请求体
     * @return
     */
    @Override
    public RegisterDTO registerApp(RegisterDTO registerDTO) throws Exception {
        // 先获取appSid
        Long appSid = iamService.registerApp(registerDTO);
        // 再通过appSid获取appId，appSecret
        return iamService.queryAppInfo(appSid, registerDTO.getAppToken());
    }

    /*
   查询租户下所有产品，并拼装url，callBackUrl、appToken、appId
    appId和callBackUrl用于前端调用IAM获取授权码code，拼接在链接后面
    授权码code，appToken，callBackUrl，用于三方应用后续调用IAM获取accessToken使用
    */
    @Override
    public List<ThirdSsoInfoDto> erpSsoInfoWithParams() {
        QueryWrapper queryWrapper = new QueryWrapper();
        //只查询有效链接
        queryWrapper.eq("valid_status", Constants.VALID_STATUS_ENABLE);
        List<ThirdSsoInfo> infos = thirdSsoInfoMapper.selectList(queryWrapper);
        //转为dto
        List<ThirdSsoInfoDto> dtos = CollectionUtils.isEmpty(infos) ? new ArrayList<>() :
                infos.stream().map(info -> {
                    ThirdSsoInfoDto dto = converter.convert(info, ThirdSsoInfoDto.class);;
                    dto.setName(info.getAppName());
                    return dto;
                }).collect(Collectors.toList());
        for (ThirdSsoInfoDto info : dtos) {
            if (ProtocolTypeEnum.OAUTH.getType().equals(info.getProtocolType())) {
                StringBuilder stringBuilder = new StringBuilder();
                stringBuilder.append(StringUtils.stripToEmpty(info.getCallBackUrl()));
                stringBuilder.append(info.getCallBackUrl().contains("?") ? "&" : "?");
                stringBuilder.append("appToken").append("=").append(info.getAppToken()).append("&");
                stringBuilder.append("curLocale=").append(LocaleContextHolder.getLocale()).append("&");
                stringBuilder.append("appCode=").append(info.getAppCode()).append("&");
                stringBuilder.append("tenantId").append("=").append(AppAuthContextHolder.getContext().getAuthoredUser().getTenantId()).append("&");
                stringBuilder.append("appId").append("=").append(info.getAppId()).append("&");
                stringBuilder.append("bindMode").append("=").append(info.getUserBindFlag());
                if (Constants.BindFlagEnum.NO.getFlag().equals(info.getUserBindFlag())) {
                    stringBuilder.append("&").append("userToken").append("=").append(AppAuthContextHolder.getContext().getAuthoredUser().getToken());
                }
                if (null == info.getExt()) {
                    info.setExt(new HashMap<>());
                }
                String fullpath = stringBuilder.toString();
                info.getExt().put("_fullpath", fullpath);
                info.setCode(info.getAppCode());
            }
            if (ProtocolTypeEnum.CAS.getType().equals(info.getProtocolType())) {
                if (null == info.getExt()) {
                    info.setExt(new HashMap<>());
                }
                info.getExt().put("_fullpath", info.getCallBackUrl());
                info.setCode(info.getAppCode());
            }
            if (ProtocolTypeEnum.SAML.getType().equals(info.getProtocolType())) {
                if (null == info.getExt()) {
                    info.setExt(new HashMap<>());
                }
                info.getExt().put("_fullpath", info.getCallBackUrl());
                info.setCode(info.getAppCode());
            }
        }

        return dtos;

    }

    /**
     * 查询敏态的应用列表
     *
     * @param querySsoListReq 请求实体
     * @return
     */
    @Override
    public List<ThirdSsoInfo> queryThirdListByScope(QuerySsoListReq querySsoListReq) {
        return this.baseMapper.queryThirdListByScope(querySsoListReq);
    }

    @Override
    public Map<Long, ThirdSsoInfo> getSsoInfoMap(List<String> bsIdList) {
        Map<Long, ThirdSsoInfo> thirdSsoInfoMap = new HashMap<>();
        if (CollectionUtils.isEmpty(bsIdList)) {
            return thirdSsoInfoMap;
        }
        QuerySsoListReq querySsoListReq = new QuerySsoListReq();
        List<Long> idList = bsIdList.stream().map(Long::parseLong).collect(Collectors.toList());
        querySsoListReq.setIdList(idList);
        List<ThirdSsoInfo> thirdSsoInfoList = thirdSsoInfoMapper.queryThirdListByScope(querySsoListReq);
        thirdSsoInfoMap = thirdSsoInfoList.stream().collect(Collectors.toMap(ThirdSsoInfo::getId, Function.identity(), (a, b) -> a));
        return thirdSsoInfoMap;
    }
}
