package com.digiwin.athena.semc.service.tenant.impl;

import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.digiwin.athena.appcore.domain.BaseResultDTO;
import com.digiwin.athena.appcore.util.MessageUtils;
import com.digiwin.athena.semc.annotate.AttributeExtraService;
import com.digiwin.athena.semc.common.ErrorCodeConstant;
import com.digiwin.athena.semc.common.I18NKey;
import com.digiwin.athena.semc.common.enums.AttributeDataTypeEnum;
import com.digiwin.athena.semc.dto.config.TenantAttributeDto;
import com.digiwin.athena.semc.entity.tenant.TenantAttributeDef;
import com.digiwin.athena.semc.entity.tenant.TenantAttributeValue;
import com.digiwin.athena.semc.mapper.tenant.TenantAttributeDefMapper;
import com.digiwin.athena.semc.mapper.tenant.TenantAttributeValueMapper;
import com.digiwin.athena.semc.proxy.trans.service.TranslateService;
import com.digiwin.athena.semc.service.tenant.TenantAttributeExtraService;
import com.digiwin.athena.semc.util.DateUtils;
import com.digiwin.athena.semc.util.ResponseEntityWrapperUtil;
import com.digiwin.athena.semc.util.Utils;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.BooleanUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.math.NumberUtils;
import org.springframework.http.HttpStatus;
import org.springframework.http.ResponseEntity;

import javax.annotation.Resource;
import java.lang.reflect.Method;
import java.math.BigDecimal;
import java.sql.Date;
import java.util.Objects;

/**
 * @author guijf
 */
@Slf4j
public class BaseTenantAttributeService extends ServiceImpl<TenantAttributeValueMapper, TenantAttributeValue> {

    @Resource
    protected TenantAttributeDefMapper tenantAttributeDefMapper;

    @Resource
    protected MessageUtils messageUtils;

    @Resource
    private TenantAttributeExtraService tenantAttributeExtraService;

    @Resource
    private TranslateService translateService;

    protected TenantAttributeDef queryAttributeDef(String attributeCode) {
        LambdaQueryWrapper<TenantAttributeDef> queryWrapper = new LambdaQueryWrapper<>();
        queryWrapper.eq(TenantAttributeDef::getAttributeCode, attributeCode);
        return  tenantAttributeDefMapper.selectOne(queryWrapper);
    }

    /**
     * 根据属性代码查询属性值
     * 此方法首先根据属性代码获取属性定义，然后根据属性定义查询属性值，
     * 如果未定义属性值，则使用默认值
     *
     * @param attributeCode 属性代码，用于标识特定的租户属性
     * @return TenantAttributeDto 包含属性值的租户属性DTO对象
     */
    protected TenantAttributeDto<Object> queryAttributeValue(String attributeCode) {
        // 初始化租户属性DTO对象，并设置属性代码
        TenantAttributeDto<Object> tenantAttribute = new TenantAttributeDto<>();
        tenantAttribute.setAttributeCode(attributeCode);

        // 查询属性定义
        TenantAttributeDef tenantAttributeDef = queryAttributeDef(attributeCode);

        // 如果属性定义为空，则直接返回初始化的租户属性DTO对象
        if(Objects.isNull(tenantAttributeDef)){
            return tenantAttribute;
        }

        // 翻译并设置属性名称
        String attributeName = translateService.translateTextCache(tenantAttributeDef.getAttributeName(), StringUtils.EMPTY);
        tenantAttribute.setAttributeName(attributeName);

        // 设置属性是否必填
        tenantAttribute.setRequired(tenantAttributeDef.getRequired());

        // 根据属性定义ID和租户ID查询属性值
        TenantAttributeValue tenantAttributeValue = getOne(new LambdaQueryWrapper<TenantAttributeValue>()
                .eq(TenantAttributeValue::getAttributeId, tenantAttributeDef.getId()));
        // 根据查询结果确定属性值
        Object attributeValue;
        if(Objects.isNull(tenantAttributeValue)){
            // 如果未查询到属性值，则使用默认值
            attributeValue = defaultValue(tenantAttributeDef);
        }else {
            tenantAttribute.setAttributeId(tenantAttributeValue.getId());
            // 根据属性定义的数据类型解析属性值
            AttributeDataTypeEnum attributeDataTypeEnum = AttributeDataTypeEnum.fromCode(tenantAttributeDef.getDataType());
            attributeValue = Objects.nonNull(attributeDataTypeEnum)?extractAttributeValue(tenantAttributeValue,attributeDataTypeEnum):null;
        }
        // 设置属性值并返回租户属性DTO对象
        tenantAttribute.setAttributeValue(attributeValue);
        return tenantAttribute;
    }

    private Object defaultValue(TenantAttributeDef tenantAttributeDef){
        try{
            AttributeDataTypeEnum attributeDataTypeEnum = AttributeDataTypeEnum.fromCode(tenantAttributeDef.getDataType());
            if(Objects.nonNull(tenantAttributeDef.getDefaultValue()) && Objects.nonNull(attributeDataTypeEnum)){
                switch (attributeDataTypeEnum){
                    case BOOLEAN:
                        return BooleanUtils.toBooleanObject(tenantAttributeDef.getDefaultValue());
                    case VARCHAR :
                    case TEXT:
                        return tenantAttributeDef.getDefaultValue();
                    case INT:
                        return NumberUtils.createInteger(tenantAttributeDef.getDefaultValue());
                    case DECIMAL:
                        return new BigDecimal(tenantAttributeDef.getDefaultValue());
                    case DATETIME:
                        return DateUtils.StringToDate(tenantAttributeDef.getDefaultValue());
                    default:
                        return null;
                }
            }
        } catch (Exception e) {
            log.error("BaseTenantAttributeService.defaultValue error defaultValue:{}",tenantAttributeDef.getDefaultValue(),e);
        }
        return null;
    }

    private Object extractAttributeValue(TenantAttributeValue value, AttributeDataTypeEnum type) {
        switch (type){
            case BOOLEAN:
                return value.getBooleanValue();
            case VARCHAR:
                return value.getVarcharValue();
            case INT:
                return value.getIntValue();
            case DATETIME:
                return value.getDatetimeValue();
            case DECIMAL:
                return value.getDecimalValue();
            default:
                return null;
        }
    }

    /**
     * 保存或更新属性值
     * 此方法首先验证属性定义是否存在，然后根据属性数据类型进行类型检查和值的转换
     * 如果属性值有效，则根据情况更新或保存属性值，并返回结果
     *
     * @param attribute 租户属性DTO，包含要保存或更新的属性信息
     * @return 返回保存或更新结果的HTTP响应实体
     */
    public ResponseEntity<BaseResultDTO<Object>> saveOrUpdateAttributeValue(TenantAttributeDto<?> attribute) {
        // 查询属性定义
        TenantAttributeDef tenantAttributeDef = queryAttributeDef(attribute.getAttributeCode());
        // 如果属性定义不存在，返回错误信息
        if(Objects.isNull(tenantAttributeDef)){
            return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.PARAM_ILLEGAL_ERROR, messageUtils.getMessage(I18NKey.COMMON_PARAM_ILLEGAL));
        }
        // 获取属性数据类型枚举
        AttributeDataTypeEnum attributeDataTypeEnum = AttributeDataTypeEnum.fromCode(tenantAttributeDef.getDataType());
        // 如果数据类型不合法，记录错误日志并返回错误信息
        if(Objects.isNull(attributeDataTypeEnum)){
            log.error("attributeDataType is error DataType:{}",tenantAttributeDef.getDataType());
            return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.PARAM_ILLEGAL_ERROR, messageUtils.getMessage(I18NKey.COMMON_PARAM_ILLEGAL));
        }
        Object attributeValue = convertAttributeValue(attribute, attributeDataTypeEnum);

        // 检查属性值是否符合预期数据类型
        if(Objects.isNull(attributeValue) ||!attributeDataTypeEnum.getDataType().isInstance(attributeValue)){
            return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.PARAM_ILLEGAL_ERROR, messageUtils.getMessage(I18NKey.COMMON_PARAM_ILLEGAL));
        }

        // 获取当前操作用户
        TenantAttributeValue tenantAttributeValue;
        // 根据属性ID查询或创建属性值对象
        if(Objects.nonNull(attribute.getAttributeId())){
            tenantAttributeValue = getById(attribute.getAttributeId());
            // 如果属性值对象不存在，返回错误信息
            if(Objects.isNull(tenantAttributeValue)){
                return ResponseEntityWrapperUtil.wrapperFail(ErrorCodeConstant.PARAM_ILLEGAL_ERROR, messageUtils.getMessage(I18NKey.COMMON_PARAM_ILLEGAL));
            }
        }else {
            tenantAttributeValue = TenantAttributeValue.builder().attributeId(tenantAttributeDef.getId()).build();
        }
        // 根据属性数据类型，设置相应的值
        buildAttributeValue(tenantAttributeValue,attributeDataTypeEnum,attributeValue);
        // 执行属性值校验
        BaseResultDTO<Object> attributedCheck = attributeCheck(tenantAttributeDef, tenantAttributeValue);
        // 如果校验不通过，返回校验结果
        if(!attributedCheck.isOK()){
            return ResponseEntity.ok(attributedCheck);
        }
        // 更新或保存属性值
        String nowTime = DateUtils.getNowTime("");
        if(Objects.nonNull(tenantAttributeValue.getId())){
            tenantAttributeValue.setModifyTime(nowTime);
            tenantAttributeValue.setModifyUserId(Utils.getUserId());
            updateById(tenantAttributeValue);
        }else {
            tenantAttributeValue.setTenantId(Utils.getTenantId());
            tenantAttributeValue.setCreateUserId(Utils.getUserId());
            tenantAttributeValue.setCreateTime(nowTime);
            tenantAttributeValue.setModifyTime(nowTime);
            tenantAttributeValue.setModifyUserId(Utils.getUserId());
            save(tenantAttributeValue);
        }
        // 更新属性ID并返回成功信息
        attribute.setAttributeId(tenantAttributeValue.getId());
        return ResponseEntityWrapperUtil.wrapperOk(attribute);
    }

    private void buildAttributeValue(TenantAttributeValue attribute, AttributeDataTypeEnum dataType,Object value) {
        switch (dataType){
            case BOOLEAN:
                attribute.setBooleanValue((Boolean) value);
                break;
            case VARCHAR:
                attribute.setVarcharValue((String) value);
                break;
            case INT:
                attribute.setIntValue((Integer) value);
                break;
            case DATETIME:
                attribute.setDatetimeValue((Date) value);
                break;
            case DECIMAL:
                attribute.setDecimalValue((Double) value);
                break;
        }
    }

    private Object convertAttributeValue(TenantAttributeDto<?> attribute,AttributeDataTypeEnum dataType){
        Object attributeValue = attribute.getAttributeValue();
        try{
            if(Objects.isNull(attributeValue) || dataType.getDataType().isInstance(attribute.getAttributeValue())){
                return attributeValue;
            }
            if(attributeValue instanceof String){
                switch (dataType){
                    case BOOLEAN:
                        attributeValue = BooleanUtils.toBooleanObject((String)attributeValue);
                        break;
                    case INT:
                        attributeValue = NumberUtils.createInteger((String)attributeValue);
                        break;
                    case DECIMAL:
                        attributeValue = NumberUtils.createDouble((String)attributeValue);
                        break;
                    case DATETIME:
                        attributeValue = DateUtils.StringToDate((String)attributeValue);
                        break;
                }
            }
        } catch (Exception e) {
            log.error("BaseTenantAttributeService.convertValue error attributeValue:{}",attributeValue,e);
        }
        return attributeValue;
    }

    private BaseResultDTO<Object> attributeCheck(TenantAttributeDef tenantAttributeDef, TenantAttributeValue tenantAttributeValue){
        String code=null;
        try{
            code = tenantAttributeDef.getAttributeCode();
            Class<?> cls = tenantAttributeExtraService.getClass();
            Method[] methods = cls.getDeclaredMethods();
            for (Method method : methods) {
                AttributeExtraService attributeCheckService = method.getAnnotation(AttributeExtraService.class);
                if (Objects.nonNull(attributeCheckService) && attributeCheckService.code().equals(code) && attributeCheckService.type()==0) {
                    try {
                        return (BaseResultDTO<Object>) method.invoke(tenantAttributeExtraService, tenantAttributeDef,tenantAttributeValue);
                    } catch (Exception ex) {
                        log.error("BaseTenantAttributeService.attributeCheck.invoke is error code：{}",code,ex);
                        return checkError();
                    }
                }
            }
            BaseResultDTO<Object> result = new BaseResultDTO<>();
            result.setStatus(HttpStatus.OK.value());
            result.setStatusDescription(HttpStatus.OK.getReasonPhrase());
            return result;
        }catch (Exception ex){
            log.error("BaseTenantAttributeService.attributeCheck is error code：{}",code,ex);
            return checkError();
        }
    }
    private BaseResultDTO<Object> checkError(){
        BaseResultDTO<Object> result = new BaseResultDTO<>();
        result.setStatus(HttpStatus.INTERNAL_SERVER_ERROR.value());
        result.setStatusDescription(HttpStatus.INTERNAL_SERVER_ERROR.getReasonPhrase());
        result.setErrorCode(String.valueOf(ErrorCodeConstant.PARAM_ILLEGAL_ERROR));
        result.setErrorMessage(messageUtils.getMessage(I18NKey.COMMON_PARAM_ILLEGAL));
        return result;
    }
}
