package com.digiwin.athena.semc.service.tenant.impl;

import com.digiwin.athena.semc.common.RedisConstant;
import com.digiwin.athena.semc.dto.config.TenantAttributeDto;
import com.digiwin.athena.semc.service.cache.ICacheService;
import com.digiwin.athena.semc.service.tenant.TenantAppService;
import com.digiwin.athena.semc.service.tenant.TenantAttributeService;
import com.digiwin.athena.semc.vo.tenant.TenantSalesPlanVO;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.stereotype.Service;

import javax.annotation.Resource;

@Slf4j
@Service
public class TenantAppServiceImpl implements TenantAppService {
    private static final String OM_PRODUCT_CODE_ASA = "ASA";

    private static final String TENANT_ATTR_CODE_SALESPLAN = "tenantSalesPlan";

    @Resource
    private TenantAttributeService tenantAttributeService;

    @Resource
    private ICacheService cacheService;

    @Override
    public TenantSalesPlanVO queryTenantSalesPlan(String tenantId) {
        TenantSalesPlanVO tenantSalesPlan = null;

        String cacheTenantSalesPlan = cacheService.getValue(RedisConstant.TENANT_SALES_PLAN + tenantId);

        // 缓存项为null，表示缓存中缺失该租户的销售方案信息，需要从数据库load
        if (null != cacheTenantSalesPlan) {
            // 缓存项为空，表示该租户没有指定销售方案，默认视作全功能销售方案
            if (StringUtils.isNotBlank(cacheTenantSalesPlan)) {
                tenantSalesPlan = new TenantSalesPlanVO();
                tenantSalesPlan.setTenantId(tenantId);
                tenantSalesPlan.setSalesPlan(cacheTenantSalesPlan);
            }
        } else {
            TenantAttributeDto<Object> tenantSalesPlanAttr = tenantAttributeService.queryTenantAttributeByCode(TENANT_ATTR_CODE_SALESPLAN);

            Object tenantAttrValue = tenantSalesPlanAttr.getAttributeValue();

            if (tenantAttrValue != null && StringUtils.isNotBlank(tenantAttrValue.toString())) {
                String salesPlan = tenantAttrValue.toString();

                tenantSalesPlan = new TenantSalesPlanVO();
                tenantSalesPlan.setTenantId(tenantId);
                tenantSalesPlan.setSalesPlan(salesPlan);

                cacheService.cache(RedisConstant.TENANT_SALES_PLAN + tenantId, salesPlan);
            } else {
                // 销售方案默认只对个别租户设置，未配置销售方案的租户，设置空的缓存项，避免大部分租户缓存穿透，频繁访问数据库
                cacheService.cache(RedisConstant.TENANT_SALES_PLAN + tenantId, "");
            }
        }

        return tenantSalesPlan;
    }

    @Override
    public boolean isOnlyAsaSalesPlan(String tenantId) {
        TenantSalesPlanVO tenantSalesPlan = this.queryTenantSalesPlan(tenantId);

        boolean isOnlyAsa = false;

        // 如果租户没有配置salesPlan信息，则默认为包含工作台的完整销售包
        if (tenantSalesPlan != null && OM_PRODUCT_CODE_ASA.equals(tenantSalesPlan.getSalesPlan())) {
            isOnlyAsa = true;
        }

        return isOnlyAsa;
    }
}
