package com.digiwin.athena.semc.service.utils;

import com.digiwin.athena.appcore.constant.ErrorTypeEnum;
import com.digiwin.athena.appcore.domain.BaseResultDTO;
import com.digiwin.athena.semc.common.ErrorCodeConstant;
import com.google.common.base.Joiner;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.ArrayUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.tuple.Pair;
import org.springframework.context.i18n.LocaleContextHolder;
import org.springframework.http.HttpStatus;
import org.springframework.stereotype.Component;
import org.springframework.validation.beanvalidation.LocalValidatorFactoryBean;

import javax.validation.*;
import javax.validation.constraints.NotNull;
import javax.validation.constraints.Size;
import javax.validation.metadata.ConstraintDescriptor;
import javax.validation.metadata.ValidateUnwrappedValue;
import java.util.*;
import java.util.stream.Collectors;

/**
 * @author jf gui
 */
@Component
public class ValidationMsgUtil {

    private static final String REQUIRED = "Required";
    private static final String LENGTH_MAX = "LengthMax";
    private static final String LENGTH_MIN = "LengthMin";
    private static final String LENGTH_RANGE = "LengthRange";

    private static MessageInterpolator interpolator;

    private static Validator validator;

    public ValidationMsgUtil(LocalValidatorFactoryBean localValidatorFactoryBean ) {
        interpolator = localValidatorFactoryBean.getMessageInterpolator();
        validator = localValidatorFactoryBean.getValidator();
    }


    public static <T> Set<ConstraintViolation<T>> validate(T obj) {
        return validator.validate(obj);
    }
    
    public static <T> BaseResultDTO<String> validateFirst(T obj){
        List<String> actionedValidate = actionValidate(obj);
        BaseResultDTO<String> result = new BaseResultDTO<>();
        if(CollectionUtils.isNotEmpty(actionedValidate)){
            result.setErrorType(ErrorTypeEnum.BUSINESS.getValue());
            result.setErrorCode(String.valueOf(ErrorCodeConstant.PARAM_MISSING_ERROR));
            result.setErrorMessage(actionedValidate.get(0));
            result.setStatus(HttpStatus.INTERNAL_SERVER_ERROR.value());
        }else {
            result.setStatus(HttpStatus.OK.value());
            result.setStatusDescription(HttpStatus.OK.getReasonPhrase());
        }
        return result;
    }

    public static <T> BaseResultDTO<String> validateAll(T obj,String splice){
        List<String> actionedValidate = actionValidate(obj);
        BaseResultDTO<String> result = new BaseResultDTO<>();
        if(CollectionUtils.isNotEmpty(actionedValidate)){
            result.setErrorType(ErrorTypeEnum.BUSINESS.getValue());
            result.setErrorCode(String.valueOf(ErrorCodeConstant.PARAM_MISSING_ERROR));
            result.setErrorMessage(Joiner.on(Objects.isNull(splice)?";":splice).join(actionedValidate));
            result.setStatus(HttpStatus.INTERNAL_SERVER_ERROR.value());
        }else {
            result.setStatus(HttpStatus.OK.value());
            result.setStatusDescription(HttpStatus.OK.getReasonPhrase());
        }
        return result;
    }

    private static <T> List<String> actionValidate(T obj){
        Set<ConstraintViolation<T>> violations = validate(obj);
        if (!violations.isEmpty()) {
            return violations.stream()
                    .map(ConstraintViolation::getMessage)
                    .filter(StringUtils::isNotBlank)
                    .collect(Collectors.toList());
        }
        return Collections.emptyList();
    }

    /**
     * 获取指定键的消息内容，并根据提供的属性进行插值替换
     * ValidationMsgUtil.getMessage("{Device.Id}{Required}", null)
     *
     * @param template 消息键，用于标识特定的消息模板 {Device.Id}{Required}
     * @param attributes 包含替换参数的键值对映射，用于消息模板中的变量替换
     * @return 插值替换后的消息字符串
     */
    @SafeVarargs
    public static String getMessage(String template, Pair<String,Object> ... attributes) {
        return interpolate(template, buildAttributes(attributes));
    }

    /**
     * 获取指定键的有效消息，通过插值替换上下文中的占位符
     *
     * @param key 消息键，用于查找对应的ValidationMessages模板
     * @param attributes 属性映射，包含用于消息插值的键值对
     * @return 插值处理后的完整消息字符串
     */
    @SafeVarargs
    public static String getValidMsg(String key, Pair<String,Object> ... attributes) {
        // 构建非空属性上下文并进行消息插值
        return interpolate(buildKey(key), buildContext(buildNotNull(buildAttributes(attributes))));
    }

    /**
     * 获取指定键的有效消息，通过插值替换上下文中的占位符
     *
     * @param key 消息键，用于查找对应的ValidationMessages模板
     * @param attributes 属性映射，包含用于消息插值的键值对
     * @return 插值处理后的完整消息字符串
     */
    @SafeVarargs
    public static String validRequired(String key, Pair<String,Object> ... attributes) {
        return interpolate(buildKeys(key,REQUIRED), buildContext(buildNotNull(buildAttributes(attributes))));
    }


    /**
     * 验证字符串长度是否不超过指定最大值
     *
     * @param key 需要验证的键名
     * @param max 允许的最大长度
     * @param attributes 包含验证属性的映射表
     * @return 插值处理后的验证结果字符串
     */
    @SafeVarargs
    public static String validLengthMax(String key, int max, Pair<String,Object> ... attributes) {
        return interpolate(buildKeys(key,LENGTH_MAX), buildContext(buildSize(null,max,buildAttributes(attributes))));
    }


    /**
     * 验证字符串长度是否满足最小长度要求
     *
     * @param key 需要验证的键名
     * @param min 最小长度限制
     * @param attributes 包含验证属性的映射表
     * @return 验证结果的插值字符串
     */
    @SafeVarargs
    public static String validLengthMin(String key, int min, Pair<String,Object> ... attributes) {
        return interpolate(buildKeys(key,LENGTH_MIN), buildContext(buildSize(min,null,buildAttributes(attributes))));
    }


    /**
     * 验证字符串长度是否在指定范围内
     * ValidationMsgUtil.validLengthRange("Device.Id",1,2,null)->
     *
     * @param key 需要验证的字符串键值
     * @param min 最小长度限制
     * @param max 最大长度限制
     * @param attributes 包含验证属性的映射表
     * @return 返回插值处理后的验证结果字符串
     */
    @SafeVarargs
    public static String validLengthRange(String key, int min, int max, Pair<String,Object> ... attributes) {
        return interpolate(buildKeys(key,LENGTH_RANGE), buildContext(buildSize(min,max,buildAttributes(attributes))));
    }

    public static String buildKeys(String... keys){
        if(ArrayUtils.isNotEmpty(keys)){
            StringBuilder key = new StringBuilder();
            for (String s : keys) {
                key.append(buildKey(s));
            }
            return key.toString();
        }
        return null;
    }
    public static String buildKey(String key){
        return "{" + key + "}";
    }

    @SafeVarargs
    public static Map<String,Object> buildAttributes(Pair<String,Object> ... attributes){
        Map<String, Object> params = new HashMap<>();
        if(ArrayUtils.isNotEmpty(attributes)){
            for(Pair<String,Object> attribute :attributes){
                if(Objects.nonNull(attribute)&& StringUtils.isNotBlank(attribute.getKey())){
                    params.put(attribute.getKey(),attribute.getValue());
                }
            }
        }
        return params;
    }

    private static MessageInterpolator.Context buildContext(ConstraintDescriptor<?> constraintDescriptor){
        return new MessageInterpolator.Context() {
            @Override
            public ConstraintDescriptor<?> getConstraintDescriptor() {
                return constraintDescriptor;
            }
            @Override
            public Object getValidatedValue() {
                return null;
            }
            @Override
            public <T> T unwrap(Class<T> type) {
                return null;
            }
        };
    }

    private static String interpolate(String template, MessageInterpolator.Context context){
        return interpolator.interpolate(template, context, LocaleContextHolder.getLocale());
    }

    private static String interpolate(String template,Map<String, Object> attributes){
        return interpolator.interpolate(template, buildContext(buildNotNull(attributes)), LocaleContextHolder.getLocale());
    }

    public static ConstraintDescriptor<javax.validation.constraints.NotNull> buildNotNull(Map<String, Object> attributes){
        return new ConstraintDescriptor<javax.validation.constraints.NotNull>() {
            @Override
            public NotNull getAnnotation() {
                return null;
            }

            @Override
            public String getMessageTemplate() {
                return "";
            }

            @Override
            public Map<String, Object> getAttributes() {
                return Objects.isNull(attributes) ? new HashMap<String,Object>() : attributes;
            }

            @Override
            public Set<ConstraintDescriptor<?>> getComposingConstraints() {
                return Collections.emptySet();
            }
            @Override
            public boolean isReportAsSingleViolation() {
                return false;
            }
            @Override
            public ValidateUnwrappedValue getValueUnwrapping() {
                return null;
            }
            @Override
            public Set<Class<?>> getGroups() {
                return Collections.emptySet();
            }
            @Override
            public Set<Class<? extends javax.validation.Payload>> getPayload() {
                return Collections.emptySet();
            }

            @Override
            public ConstraintTarget getValidationAppliesTo() {
                return null;
            }
            @Override
            public List<Class<? extends ConstraintValidator<NotNull, ?>>> getConstraintValidatorClasses() {
                return Collections.emptyList();
            }
            @Override
            public <T> T unwrap(Class<T> type) {
                return null;
            }
        };
    }

    public static ConstraintDescriptor<javax.validation.constraints.Size> buildSize(Integer min,Integer max,Map<String, Object> attributes){
        return new ConstraintDescriptor<javax.validation.constraints.Size>() {
            @Override
            public Size getAnnotation() {
                return null;
            }

            @Override
            public String getMessageTemplate() {
                return "";
            }

            @Override
            public Map<String, Object> getAttributes() {
                Map<String, Object> objectMap = Objects.isNull(attributes) ? new HashMap<String,Object>() : attributes;
                if(Objects.nonNull(min)){
                    objectMap.put("min",min);
                }
                if(Objects.nonNull(max)){
                    objectMap.put("max",max);
                }
                return objectMap;
            }

            @Override
            public Set<ConstraintDescriptor<?>> getComposingConstraints() {
                return Collections.emptySet();
            }
            @Override
            public boolean isReportAsSingleViolation() {
                return false;
            }
            @Override
            public ValidateUnwrappedValue getValueUnwrapping() {
                return null;
            }
            @Override
            public Set<Class<?>> getGroups() {
                return Collections.emptySet();
            }
            @Override
            public Set<Class<? extends javax.validation.Payload>> getPayload() {
                return Collections.emptySet();
            }

            @Override
            public ConstraintTarget getValidationAppliesTo() {
                return null;
            }
            @Override
            public List<Class<? extends ConstraintValidator<Size, ?>>> getConstraintValidatorClasses() {
                return Collections.emptyList();
            }
            @Override
            public <T> T unwrap(Class<T> type) {
                return null;
            }
        };
    }
}
