package com.digiwin.athena.semc.service.workbench.designer.service;

import cn.hutool.core.collection.CollectionUtil;
import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.digiwin.athena.appcore.util.SnowflakeIdWorker;
import com.digiwin.athena.semc.common.Constants;
import com.digiwin.athena.semc.dto.portal.PortalContentCarouselImage;
import com.digiwin.athena.semc.dto.workbench.req.WorkbenchDeployAppReqDto;
import com.digiwin.athena.semc.entity.menu.CustomizedMenuTemplate;
import com.digiwin.athena.semc.entity.menu.CustomizedMenuWork;
import com.digiwin.athena.semc.entity.portal.AppCustomSetting;
import com.digiwin.athena.semc.entity.portal.LabelSystemCustom;
import com.digiwin.athena.semc.entity.portal.PortalContentCarousel;
import com.digiwin.athena.semc.entity.portal.PortalDeployMessage;
import com.digiwin.athena.semc.entity.portal.PortalInfo;
import com.digiwin.athena.semc.entity.portal.PortalInfoContent;
import com.digiwin.athena.semc.entity.portal.RelatedApp;
import com.digiwin.athena.semc.service.menu.CustomizedMenuTemplateService;
import com.digiwin.athena.semc.service.menu.CustomizedMenuWorkService;
import com.digiwin.athena.semc.service.portal.AppCustomSettingService;
import com.digiwin.athena.semc.service.portal.LabelSystemCustomService;
import com.digiwin.athena.semc.service.portal.PortalDeployMessageService;
import com.digiwin.athena.semc.service.portal.PortalInfoContentService;
import com.digiwin.athena.semc.service.portal.PortalInfoService;
import com.digiwin.athena.semc.service.portal.RelatedAppService;
import com.digiwin.athena.semc.service.workbench.constant.EnvOperateEnum;
import com.digiwin.athena.semc.service.workbench.constant.MenuItemTypeEnum;
import com.digiwin.athena.semc.service.workbench.constant.WorkBenchConstant;
import com.digiwin.athena.semc.service.workbench.designer.mongo.WorkBenchAppCustomSetting;
import com.digiwin.athena.semc.service.workbench.designer.mongo.WorkBenchComponent;
import com.digiwin.athena.semc.service.workbench.designer.mongo.WorkBenchMenu;
import com.digiwin.athena.semc.service.workbench.designer.mongo.WorkBenchMenuItem;
import com.digiwin.athena.semc.service.workbench.designer.mongo.WorkBenchPortal;
import com.digiwin.athena.semc.service.workbench.designer.mongo.WorkBenchPortalContent;
import com.digiwin.athena.semc.service.workbench.designer.mongo.WorkBenchRelatedApp;
import com.digiwin.athena.semc.service.workbench.designer.po.WorkbenchDesignerData;
import com.digiwin.athena.semc.service.workbench.util.WorkbenchSyncDataConverter;
import com.digiwin.athena.semc.service.workbench.util.WorkbenchUtils;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import java.io.File;
import java.util.ArrayList;
import java.util.Comparator;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.function.Function;
import java.util.stream.Collectors;

@Component
@Slf4j
public class WorkbenchDeployHandler extends AbstractWorkbenchDeployHandler {
    @Autowired
    private AppCustomSettingService appCustomSettingService;

    @Autowired
    private RelatedAppService relatedAppService;

    @Autowired
    private LabelSystemCustomService customComponentService;

    @Autowired
    private PortalInfoService portalService;

    @Autowired
    private PortalInfoContentService portalContentService;

    @Autowired
    private CustomizedMenuTemplateService menuService;

    @Autowired
    private CustomizedMenuWorkService menuItemService;

    @Autowired
    private PortalDeployMessageService portalDeployMessageService;

    @Override
    public void deploy(WorkbenchDeployAppReqDto deployAppReq, WorkbenchDesignerData designerData) {
        String application = deployAppReq.getApplication();
        // 数据版本，如：1.0、2.0
        String version = EnvOperateEnum.PUBLISH.getVersion();
        // 编译版本
        String compileVersion = deployAppReq.getCompileNo();

        // 发版工作台应用客制配置 workbench_app_custom_setting
        deployCustomSettings(application, version, designerData.getCustomSettings());

        // 发版关联应用表 workbench_related_app（删除，插入）
        deployRelatedApps(application, version, designerData.getRelatedApps());

        // 发版部件 t_label_system_custom（存在更新，不存在新增）
        deployComponents(application, compileVersion, version, designerData.getComponents());

        // 同步门户数据 t_portal_info、t_portal_info_content、t_customized_menu_template、t_customized_menu
        deployPortals(application, compileVersion, version, designerData);

        // 同步轮播图数据
        uploadCarousel(designerData.getPortalContents(),designerData.getCarouselFiles());

        //删除消息
        deletePortalDeployMessages(application, designerData.getPortals(), version);
    }

    private void uploadCarousel(List<WorkBenchPortalContent> portalContents,File[] carouselFiles) {
        Map<String,File> map = Maps.newHashMap();
        if(null != carouselFiles){
            for (File file : carouselFiles) {
                String fileName = file.getName();
                // 解析出 剔除 后缀名的文件名
                String fileId = StringUtils.substring(fileName, 0, StringUtils.lastIndexOf(fileName, '.'));
                map.put(fileId, file);
            }
        }
        for (WorkBenchPortalContent workBenchPortalContent : portalContents){
            PortalContentCarousel carouselInfo = workBenchPortalContent.getCarouselInfo();
            if (null != carouselInfo) {
                List<PortalContentCarouselImage> imageList = carouselInfo.getImageList();
                for (PortalContentCarouselImage portalContentCarouselImage : imageList){
                    if (Objects.equals(0,portalContentCarouselImage.getImgType())){
                        continue;
                    }else{
                        File file = map.get(portalContentCarouselImage.getFileId());
                        if (Objects.isNull(file)){
                            continue;
                        }
                        uploadThumbnail(file,portalContentCarouselImage.getFileId());
                    }
                }
            }
        }
    }

    /**
     * 入库未读消息
     * @param portals
     */
    private void deletePortalDeployMessages(String application,List<WorkBenchPortal> portals,String version){
        if (CollectionUtils.isEmpty(portals)) {
            log.info("{}无有效内容项、菜单项，跳出入库未读消息", application);
            return;
        }
        //设计时id集合
        Set<Long> portalDesignerIds = portals.stream().map(e -> e.getId()).collect(Collectors.toSet());
        if (CollectionUtils.isEmpty(portalDesignerIds)){
            return;
        }
        LambdaQueryWrapper<PortalDeployMessage> lambdaQueryWrappernew = new LambdaQueryWrapper<>();
        lambdaQueryWrappernew.in(PortalDeployMessage::getPortalId,portalDesignerIds).eq(PortalDeployMessage::getVersion,version);
        portalDeployMessageService.remove(lambdaQueryWrappernew);
    }

    private void deployPortals(String application, String compileVersion, String version,
            WorkbenchDesignerData designerData) {
        if (CollectionUtils.isEmpty(designerData.getPortals())) {
            log.info("编译包没有门户数据，无需发版门户");
            return;
        }

        // 发版门户
        List<PortalInfo> allPortal = deployPortal(application, compileVersion, version, designerData.getPortals());

        // 发版门户内容
        deployPortalContents(application, compileVersion, version, designerData.getPortalContents(), allPortal);

        // 发版菜单
        deployMenus(application, compileVersion, version, designerData.getMenus(), designerData.getMenuItems());

        // 上传缩略图
        uploadThumbnail(designerData.getThumbnailFiles(), allPortal);
    }

    /**
     * @description: 发版门户
     * @param
     * @return
     * @author: sunyfa
     */
    private List<PortalInfo> deployPortal(String application, String compileVersion, String version,
            List<WorkBenchPortal> portals) {
        // 1.0 门户设计时id，设计时门户对象
        Map<Long, WorkBenchPortal> desPortalMap = portals.stream().collect(Collectors.toMap(WorkBenchPortal::getId, Function.identity()));

        // 运行态比较设计态数据，如果有删除的，删除门户、门户内容、菜单模板、菜单项数据
        List<PortalInfo> existPortals = portalService.lambdaQuery()
                                                   .eq(PortalInfo::getWorkbenchAppCode, application)
                                                   .eq(PortalInfo::getVersion, version)
                                                   .list();

        // 门户设计时id，门户运行时对象
        Map<Long, PortalInfo> existPortalMap = existPortals.stream()
                                                           .collect(Collectors.toMap(PortalInfo::getPortalDesignerId, Function.identity()));

        // 过滤出删除的门户设计时id
        if (StringUtils.isNotEmpty(compileVersion)) {
            List<Long> deleteDesPortalIds = CollectionUtil.subtractToList(existPortalMap.keySet(), desPortalMap.keySet());
            if (CollectionUtils.isNotEmpty(deleteDesPortalIds)) {
                portalService.remove(Wrappers.<PortalInfo>lambdaQuery()
                                             .in(PortalInfo::getPortalDesignerId, deleteDesPortalIds)
                                             .eq(PortalInfo::getWorkbenchAppCode, application)
                                             .eq(PortalInfo::getVersion, version));
            }
        }

        List<PortalInfo> allPortal = new ArrayList<>();
        // 过滤出需要更新的部件设计时id
        Set<Long> updateDesPortalIdSet =
                CollectionUtil.intersectionDistinct(desPortalMap.keySet(), existPortalMap.keySet());

        if (CollectionUtils.isNotEmpty(updateDesPortalIdSet)) {
            List<PortalInfo> existPortalList = updateDesPortalIdSet.stream().map(
                    desPortalId -> WorkbenchSyncDataConverter.toUpdatePortal(existPortalMap.get(desPortalId),
                            desPortalMap.get(desPortalId), compileVersion)).collect(Collectors.toList());
            // 批量更新1.0门户，但不会改变菜单ID值
            portalService.updateBatchById(existPortalList);
            allPortal.addAll(existPortalList);
        }

        // 过滤出需要新增的部件设计时id
        List<Long> addDesPortalIdList = CollectionUtil.subtractToList(desPortalMap.keySet(), existPortalMap.keySet());
        if (CollectionUtils.isNotEmpty(addDesPortalIdList)) {
            List<PortalInfo> addPortalList = addDesPortalIdList.stream().map(
                    desPortalId -> WorkbenchSyncDataConverter.toNewPortal(desPortalMap.get(desPortalId), version,
                            compileVersion, desPortalMap.get(desPortalId).getMenuId())).collect(Collectors.toList());

            portalService.saveBatch(addPortalList);
            allPortal.addAll(addPortalList);
        }

        log.info("发版门户完成");
        return allPortal;
    }

    /**
     * @description: 发版门户内容
     * @param
     * @return
     * @author: sunyfa
     */
    private void deployPortalContents (String application, String compileVersion, String version,
            List<WorkBenchPortalContent> desPortalContents, List<PortalInfo> allPortal) {
        if (CollectionUtils.isEmpty(desPortalContents)) {
            log.info("编译包没有门户内容数据，无需发版门户内容");
            return;
        }

        if (StringUtils.isNotEmpty(compileVersion)) {
            // 删除所有门户内容
            portalContentService.remove(Wrappers.<PortalInfoContent>lambdaQuery()
                                                .eq(PortalInfoContent::getWorkbenchAppCode, application)
                                                .eq(PortalInfoContent::getVersion, version));
        }
        else {
            List<Long> portalContentDesIds = desPortalContents.stream().map(
                    portalContent -> portalContent.getId()).collect(Collectors.toList());

            // 删除当前门户内容
            portalContentService.remove(
                    Wrappers.<PortalInfoContent>lambdaQuery().eq(PortalInfoContent::getWorkbenchAppCode, application)
                            .eq(PortalInfoContent::getVersion, version)
                            .in(PortalInfoContent::getDesignerId, portalContentDesIds));
        }

        // 1.0 门户设计时id，门户运行时id
        Map<Long, PortalInfo> portalIdMap =
                allPortal.stream().collect(Collectors.toMap(PortalInfo::getPortalDesignerId,  Function.identity()));

        // 新增菜单内容
        List<PortalInfoContent> addNewPortalContentList = new ArrayList<>();
        // 根据设计时的id升序排序
        List<WorkBenchPortalContent> sortedList = desPortalContents.stream()
                                                                   .sorted(Comparator.comparing(WorkBenchPortalContent::getId))
                                                                   .collect(Collectors.toList());
        sortedList.forEach(desPortalContent -> {
            log.info("开始处理门户内容：{}", desPortalContent.getId());
            PortalInfoContent portalContent = WorkbenchSyncDataConverter.toNewPortalContent(application,
                    desPortalContent, portalIdMap.get(desPortalContent.getPortalId()).getId(), version);


            // 预设部件
            if (WorkbenchUtils.isPreComponent(desPortalContent)) {
                portalContent.setContentType(desPortalContent.getContentType());
                portalContent.setDataType(desPortalContent.getContentSubType());
            }
            else{
                if (Constants.ClientTypeEnum.PC.getValue() == portalIdMap.get(
                        desPortalContent.getPortalId()).getClientType()) {
                    if (WorkbenchUtils.isCustomComponent(desPortalContent)) {
                        portalContent.setContentId(desPortalContent.getContentId());
                    }
                }
                else {
                    if (WorkbenchUtils.isMobileCustomComponent(desPortalContent)) {
                        portalContent.setContentId(desPortalContent.getContentId());
                    }
                }
            }

            addNewPortalContentList.add(portalContent);
        });

        portalContentService.saveBatch(addNewPortalContentList);

        log.info("发版门户内容完成");
    }

    /**
     * @description: 发版菜单
     * @param
     * @return 
     * @author: sunyfa
     */
    private void deployMenus (String application, String compileVersion, String version, List<WorkBenchMenu> desMenus,
            List<WorkBenchMenuItem> desMenuItems) {
        // 发版菜单模板
        List<CustomizedMenuTemplate> allMenuTemplates = deployMenuTemplates(application, compileVersion, version, desMenus);

        // 发版菜单项
        deployMenuItems (application, compileVersion, version, desMenuItems, allMenuTemplates);
    }

    private void uploadThumbnail (File[] thumbnailFiles, List<PortalInfo> allPortal) {
        if (CollectionUtils.isEmpty(allPortal)) {
            log.info("编译包没有缩略图数据，无需下载缩略图");
            return;
        }

        Map<String,File> map = Maps.newHashMap();
        for (File file : thumbnailFiles) {
            String fileName = file.getName();
            // 解析出 剔除 后缀名的文件名
            String fileId = StringUtils.substring(fileName, 0, StringUtils.lastIndexOf(fileName, '.'));
            map.put(fileId, file);
        }
        for (PortalInfo portalInfo : allPortal) {
            String portalImgId = portalInfo.getPortalImgId();
            File file = map.get(portalImgId);
            if (Objects.isNull(file)) {
                continue;
            }
            uploadThumbnail(file, portalImgId);
        }
    }

    /**
     * @description: 发版菜单模板
     * @param
     * @return 
     * @author: sunyfa
     */
    private List<CustomizedMenuTemplate> deployMenuTemplates(String application, String compileVersion,
            String version, List<WorkBenchMenu> desMenus) {
        if (CollectionUtils.isEmpty(desMenus)) {
            log.info("编译包没有菜单数据，无需发版菜单");
            return null;
        }

        // 1.0 菜单设计时id，菜单设计时对象
        Map<Long, WorkBenchMenu> desMenuMap = desMenus.stream()
                                                           .collect(Collectors.toMap(WorkBenchMenu::getId, Function.identity()));

        // 比较新数据中是否有删除的，有删除的，删除部件数据
        List<CustomizedMenuTemplate> existMenuTemplates;
        if (StringUtils.isNotEmpty(compileVersion)) {
            existMenuTemplates = menuService.lambdaQuery()
                                            .eq(CustomizedMenuTemplate::getWorkbenchAppCode, application)
                                            .eq(CustomizedMenuTemplate::getVersion, version).list();
        }
        else {
            existMenuTemplates = menuService.lambdaQuery()
                                            .eq(CustomizedMenuTemplate::getWorkbenchAppCode, application)
                                            .eq(CustomizedMenuTemplate::getVersion, version)
                                            .in(CustomizedMenuTemplate::getMenuDesignerId, desMenuMap.keySet()).list();
        }

        // 已发版的id映射 1.0 菜单设计时id，菜单运行时id
        Map<Long, CustomizedMenuTemplate> existMenuTemplateMap = existMenuTemplates
                .stream()
                .collect(Collectors.toMap(CustomizedMenuTemplate::getMenuDesignerId, Function.identity()));

        // 过滤出需要删除的菜单
        List<Long> delDesMenuIdList = CollectionUtil.subtractToList(existMenuTemplateMap.keySet(), desMenuMap.keySet());
        if (CollectionUtils.isNotEmpty(delDesMenuIdList)) {
            menuService.remove(
                    Wrappers.<CustomizedMenuTemplate>lambdaQuery()
                            .in(CustomizedMenuTemplate::getMenuDesignerId, delDesMenuIdList)
                            .eq(CustomizedMenuTemplate::getVersion, version));
        }

        List<CustomizedMenuTemplate> allMenuTemplate = new ArrayList<>();
        // 过滤出需要更新的菜单设计时id
        Set<Long> updateDesMenuIdList =
                CollectionUtil.intersectionDistinct(existMenuTemplateMap.keySet(), desMenuMap.keySet());
        if (CollectionUtils.isNotEmpty(updateDesMenuIdList)) {
            List<CustomizedMenuTemplate> existMenuTemplateList = updateDesMenuIdList.stream().map(
                    desMenuId -> WorkbenchSyncDataConverter.toUpdatePublishMenu(existMenuTemplateMap.get(desMenuId),
                            desMenuMap.get(desMenuId), compileVersion)).collect(Collectors.toList());

            menuService.updateBatchById(existMenuTemplateList);
            allMenuTemplate.addAll(existMenuTemplateList);
        }

        // 过滤出需要新增的部件设计时id
        List<Long> addDesMenuIdList = CollectionUtil.subtractToList(desMenuMap.keySet(), existMenuTemplateMap.keySet());
        if (CollectionUtils.isNotEmpty(addDesMenuIdList)) {
            List<CustomizedMenuTemplate> addMenuTemplateList = addDesMenuIdList.stream().map(
                    desMenuId -> WorkbenchSyncDataConverter.toNewMenu(desMenuMap.get(desMenuId), version, application,
                            compileVersion)).collect(Collectors.toList());

            menuService.saveBatch(addMenuTemplateList);
            allMenuTemplate.addAll(addMenuTemplateList);
        }

        log.info("发版菜单模板完成");
        return allMenuTemplate;
    }

    private void deployMenuItems (String application, String compileVersion, String version, List<WorkBenchMenuItem> desMenuItems,
            List<CustomizedMenuTemplate> allMenuTemplates) {
        if (CollectionUtils.isEmpty(desMenuItems)) {
            log.info("编译包没有菜单项数据，无需发版菜单项");
            return;
        }

        if (StringUtils.isNotEmpty(compileVersion)) {
            // 删除已发版的菜单项
            menuItemService.remove(Wrappers.<CustomizedMenuWork>lambdaQuery()
                                           .eq(CustomizedMenuWork::getWorkbenchAppCode, application)
                                           .eq(CustomizedMenuWork::getVersion, version));
        }
        else {
            List<Long> desMenuTemplateIdList = desMenuItems.stream().map(
                    menuTemplate -> menuTemplate.getMenuId()).collect(Collectors.toList());

            // 删除已发版的菜单项
            menuItemService.remove(Wrappers.<CustomizedMenuWork>lambdaQuery()
                                           .eq(CustomizedMenuWork::getWorkbenchAppCode, application)
                                           .in(CustomizedMenuWork::getDesTemplateId, desMenuTemplateIdList)
                                           .eq(CustomizedMenuWork::getVersion, version));
        }

        // 过滤出所有的根节点
        List<WorkBenchMenuItem> rootDesMenuItems =
                desMenuItems.stream().filter(WorkbenchUtils::isRoot).collect(Collectors.toList());

        if (CollectionUtils.isEmpty(rootDesMenuItems)) {
            log.error("发版时菜单项数据没有根节点");
            return;
        }

        // 过滤出所有的叶子节点
        List<WorkBenchMenuItem> childrenDesMenuItemList = Lists.newArrayList(desMenuItems);
        childrenDesMenuItemList.removeAll(rootDesMenuItems);

        // 1.0 叶子节点 菜单设计时id，菜单项列表
        Map<Long, List<WorkBenchMenuItem>> newDesMenuItemMap = new HashMap<>();
        childrenDesMenuItemList.forEach(desMenuItem -> newDesMenuItemMap.putIfAbsent(desMenuItem.getMenuId(), Lists.newArrayList(desMenuItem)));

        // 过滤出所有的自定义部件的运行时id
        // 收集菜单项中所有自定义部件
        List<Long> desCustomCompIds = desMenuItems.stream().filter(WorkbenchUtils::isCustomComponent).map(
                WorkBenchMenuItem::getMenuItemSubId).collect(Collectors.toList());

        // 1.0 组件设计时id，组件运行时id
        Map<Long, Long> customCompIdMap = new HashMap<>();
        if (CollectionUtils.isNotEmpty(desCustomCompIds)) {
            customCompIdMap = toCustomCompId(version, desCustomCompIds);
        }

        // 菜单模板id映射 菜单设计时id，菜单运行时id
        Map<Long, Long> menuTemplateIdMap = allMenuTemplates
                .stream()
                .collect(Collectors.toMap(CustomizedMenuTemplate::getMenuDesignerId, CustomizedMenuTemplate::getId));

        List<CustomizedMenuWork> menuWorkList = new ArrayList<>();
        Map<Long, Long> finalCustomCompIdMap = customCompIdMap;
        rootDesMenuItems.stream().forEach(desMenuItem -> {
            Map<String, Long> idMapping = Maps.newHashMap();

            // 运行态menuid
            Long menuId = menuTemplateIdMap.get(desMenuItem.getMenuId());

            Long menuItemId = SnowflakeIdWorker.getInstance().newId();
            idMapping.put(desMenuItem.getId(), menuItemId);

            // 构建父节点对象
            CustomizedMenuWork parentMenuItem = WorkbenchSyncDataConverter.toMenuItem(application, desMenuItem,
                    menuItemId, menuId, null, version);

            // 自定义组件
            if (WorkbenchUtils.isCustomComponent(desMenuItem)) {
                parentMenuItem.setMenuLabelId(finalCustomCompIdMap.get(desMenuItem.getMenuItemSubId()));
                parentMenuItem.setMenuType(MenuItemTypeEnum.COMPONENT.getType());
                parentMenuItem.setMenuLabelType(Constants.LabelTypeEnum.SYSTEM_CUSTOM.getVal());
            }
            // 预设组件
            else if (WorkbenchUtils.isPreComponent(desMenuItem)) {
                parentMenuItem.setMenuType(MenuItemTypeEnum.COMPONENT.getType());
                parentMenuItem.setMenuLabelType(Constants.LabelTypeEnum.SYSTEM_PRE.getVal());
                parentMenuItem.setDataType(desMenuItem.getSubCategory());
            }

            menuWorkList.add(parentMenuItem);

            if (CollectionUtils.isNotEmpty(childrenDesMenuItemList)) {
                // 构建叶子对象
                buildChildMenuItem(application, desMenuItem.getId(), childrenDesMenuItemList, menuWorkList, idMapping,
                        menuId, finalCustomCompIdMap, version);
            }
        });

        menuItemService.saveBatch(menuWorkList);
        log.info("发版菜单完成");
    }

    private void buildChildMenuItem(String application, String parentId, List<WorkBenchMenuItem> workBenchMenuItemList,
            List<CustomizedMenuWork> menuWorkList, Map<String, Long> idMapping, Long menuId,
            Map<Long, Long> customCompIdMap,String version) {
        workBenchMenuItemList.forEach(workBenchMenuItem -> {
            if (StringUtils.isNotEmpty(workBenchMenuItem.getMenuItemParentId()) && workBenchMenuItem.getMenuItemParentId().equals(parentId)) {
                String desMenuItemId = workBenchMenuItem.getId();

                Long menuItemId = SnowflakeIdWorker.getInstance().newId();
                idMapping.put(desMenuItemId, menuItemId);

                if (null != workBenchMenuItem.getMenuItemParentId() && null == idMapping.get(workBenchMenuItem.getMenuItemParentId())) {
                    Long parentMenuItemId = SnowflakeIdWorker.getInstance().newId();
                    idMapping.put(workBenchMenuItem.getMenuItemParentId(), parentMenuItemId);
                }

                CustomizedMenuWork menuItem = WorkbenchSyncDataConverter.toMenuItem(application,workBenchMenuItem,
                        menuItemId, menuId, idMapping.get(workBenchMenuItem.getMenuItemParentId()), version);

                // 自定义组件
                if (WorkbenchUtils.isCustomComponent(workBenchMenuItem)) {
                    menuItem.setMenuLabelId(customCompIdMap.get(workBenchMenuItem.getMenuItemSubId()));
                    menuItem.setMenuType(MenuItemTypeEnum.COMPONENT.getType());
                    menuItem.setMenuLabelType(Constants.LabelTypeEnum.SYSTEM_CUSTOM.getVal());
                }
                // 预设组件
                else if (WorkbenchUtils.isPreComponent(workBenchMenuItem)) {
                    menuItem.setMenuType(MenuItemTypeEnum.COMPONENT.getType());
                    menuItem.setMenuLabelType(Constants.LabelTypeEnum.SYSTEM_PRE.getVal());
                    menuItem.setDataType(workBenchMenuItem.getSubCategory());
                }

                menuWorkList.add(menuItem);

                buildChildMenuItem(application, workBenchMenuItem.getId(), workBenchMenuItemList, menuWorkList,
                        idMapping, menuId, customCompIdMap, version);
            }
        });
    }

    private Map<Long, Long> toCustomCompId(String version, List<Long> designerIds) {
        List<LabelSystemCustom> customComps = customComponentService.lambdaQuery()
                .in(LabelSystemCustom::getComponentDesignerId, designerIds)
                .eq(LabelSystemCustom::getVersion, version).select(LabelSystemCustom::getComponentDesignerId, LabelSystemCustom::getId)
                .list();
        return customComps.stream().collect(Collectors.toMap(LabelSystemCustom::getComponentDesignerId, LabelSystemCustom::getId));
    }

    /**
     * 同步自定义部件
     *
     * @param application    工作台应用code
     * @param compileVersion 编译版本
     * @param version        数据版本
     * @param desComponents  ISV自定义部件列表
     */
    private void deployComponents(String application, String compileVersion, String version, List<WorkBenchComponent> desComponents) {
        if (CollectionUtils.isEmpty(desComponents)) {
            log.info("编译包没有部件数据，无需发版部件");
            return;
        }

        // 过滤掉预设部件
        List<WorkBenchComponent> customDesComponentList = filterCustomDesComponents(desComponents);
        if (CollectionUtils.isEmpty(customDesComponentList)) {
            log.info("编译包没有自定义部件，无需发版部件");
            return;
        }

        // 1.0 部件设计时id，部件设计时对象
        Map<Long, WorkBenchComponent> desComponentMap = customDesComponentList.stream()
                                                                              .collect(Collectors.toMap(WorkBenchComponent::getId, Function.identity()));

        // 比较新数据中是否有删除的，有删除的，删除部件数据
        List<LabelSystemCustom> existComponents = customComponentService.lambdaQuery()
                                                                      .eq(LabelSystemCustom::getWorkbenchAppCode, application)
                                                                      .eq(LabelSystemCustom::getVersion, version)
                                                                      .list();
        // 已发版的id映射 1.0 部件设计时id，部件运行时id
        Map<Long, LabelSystemCustom> existComponentMap = existComponents.stream()
                               .collect(Collectors.toMap(LabelSystemCustom::getComponentDesignerId, Function.identity()));

        // 过滤出需要删除的部件
        if (StringUtils.isNotEmpty(compileVersion)) {
            List<Long> delDesCompIdList = CollectionUtil.subtractToList(existComponentMap.keySet(),
                    desComponentMap.keySet());
            if (CollectionUtils.isNotEmpty(delDesCompIdList)) {
                customComponentService.remove(
                        Wrappers.<LabelSystemCustom>lambdaQuery().in(LabelSystemCustom::getComponentDesignerId,
                                delDesCompIdList).eq(LabelSystemCustom::getVersion, version));
            }
        }

        // 过滤出需要更新的部件设计时id
        Set<Long> updateDesCompIdList =
                CollectionUtil.intersectionDistinct(desComponentMap.keySet(), existComponentMap.keySet());
        if (CollectionUtils.isNotEmpty(updateDesCompIdList)) {
            List<LabelSystemCustom> existCompList = updateDesCompIdList.stream().map(
                    desCompId -> WorkbenchSyncDataConverter.toUpdateCustomComponent(compileVersion,
                            existComponentMap.get(desCompId), desComponentMap.get(desCompId))).collect(
                    Collectors.toList());

            customComponentService.updateBatchById(existCompList);
        }

        // 过滤出需要新增的部件设计时id
        List<Long> addDesCompIdList = CollectionUtil.subtractToList(desComponentMap.keySet(), existComponentMap.keySet());
        if (CollectionUtils.isNotEmpty(addDesCompIdList)) {
            List<LabelSystemCustom> addCompList = addDesCompIdList.stream().map(
                    desCompId -> WorkbenchSyncDataConverter.toNewCustomComponent(compileVersion, version,
                            desComponentMap.get(desCompId))).collect(Collectors.toList());

            customComponentService.saveBatch(addCompList);
        }
    }

    /**
     * @description: 过滤掉发版部件中的预设部件
     * @param
     * @return 
     * @author: sunyfa
     */
    private List<WorkBenchComponent> filterCustomDesComponents (List<WorkBenchComponent> desComponents) {
        List<WorkBenchComponent> filterDesComponents = new ArrayList<>();
        desComponents.stream().forEach(desComponent -> {
            if (WorkBenchConstant.PC.equals(desComponent.getClientType())) {
                if(desComponent.getCategory() != Constants.DataCategoryEnum.CATEGORY_TODO.getVal()
                        && desComponent.getCategory() != Constants.DataCategoryEnum.CATEGORY_REPORT.getVal()
                        && desComponent.getCategory() != Constants.DataCategoryEnum.CATEGORY_NOTICE.getVal()
                        && desComponent.getCategory() != Constants.DataCategoryEnum.CATEGORY_CURRENCY.getVal()){
                    filterDesComponents.add(desComponent);
                }
            }else if (WorkBenchConstant.MOBILE.equals(desComponent.getClientType())) {
                if(desComponent.getCategory() != Constants.PrtalMobileContentTypeEnum.PRE_KING_KONG_DISTRICT.getVal()
                        && desComponent.getCategory() != Constants.PrtalMobileContentTypeEnum.PRE_TASK.getVal()
                        && desComponent.getCategory() != Constants.PrtalMobileContentTypeEnum.PRE_PROJECT.getVal()
                        && desComponent.getCategory() != Constants.PrtalMobileContentTypeEnum.KING_KONG_BALL.getVal()
                        && desComponent.getCategory() != Constants.PrtalMobileContentTypeEnum.CONTENT_CAROUSEL.getVal()
                        && desComponent.getCategory() != Constants.PrtalMobileContentTypeEnum.CONTENT_NEWS.getVal()
                        && desComponent.getCategory() != Constants.PrtalMobileContentTypeEnum.TITLE.getVal()
                        && desComponent.getCategory() != Constants.PrtalMobileContentTypeEnum.DIVIDER.getVal()
                        && desComponent.getCategory() != Constants.PrtalMobileContentTypeEnum.THIRD_TO_DO.getVal()
                        && desComponent.getCategory() != Constants.PrtalMobileContentTypeEnum.PICTURE.getVal()
                        && desComponent.getCategory() != Constants.PrtalMobileContentTypeEnum.BUSINESS_ENTRANCE.getVal()){
                    filterDesComponents.add(desComponent);
                }
            }
        });

        return filterDesComponents;
    }

    /**
     * 同步关联应用表
     *
     * @param application    工作台应用code
     * @param version        数据版本
     * @param desRelatedApps 关联应用数据列表
     */
    private void deployRelatedApps(String application, String version, List<WorkBenchRelatedApp> desRelatedApps) {
        if (CollectionUtils.isEmpty(desRelatedApps)) {
            log.info("编译包没有关联应用数据，无需发版关联应用");
            return;
        }

        // 构建运行时关联应用对象
        List<RelatedApp> relatedApps =
                desRelatedApps.stream()
                                 .map(relatedApp -> WorkbenchSyncDataConverter.toRelatedApp(relatedApp, version))
                                 .collect(Collectors.toList());

        // 根据门户id、application、version 删除已发版的数据
        Set<String> eleIds = relatedApps.stream().map(RelatedApp::getEleId).collect(Collectors.toSet());
        relatedAppService.remove(Wrappers.<RelatedApp>lambdaQuery()
                                         .in(RelatedApp::getEleId, eleIds)
                                         .eq(RelatedApp::getApplication, application)
                                         .eq(RelatedApp::getVersion, version));

        // 批量新增application的1.0数据
        relatedAppService.saveBatch(relatedApps);

        log.info("发版关联应用完成");
    }

    /**
     * 同步应用客制配置数据
     *
     * @param application       工作台应用code
     * @param version           数据版本
     * @param desCustomSettings ISV客制配置数据列表
     */
    private void deployCustomSettings(String application, String version,
            List<WorkBenchAppCustomSetting> desCustomSettings) {
        if (CollectionUtils.isEmpty(desCustomSettings)) {
            log.info("编译包没有应用客制数据，无需发版客制包");
            return;
        }

        // 构建运行时客制包对象
        List<AppCustomSetting> customSettings =
                desCustomSettings.stream()
                                 .map(desCustomSetting -> WorkbenchSyncDataConverter.toCustomSetting(desCustomSetting, version))
                                 .collect(Collectors.toList());

        // 删除已发版的客制包数据
        appCustomSettingService.remove(Wrappers.<AppCustomSetting>lambdaQuery()
                                               .eq(AppCustomSetting::getApplication, application)
                                               .eq(AppCustomSetting::getVersion, version));
        // 批量新增数据
        appCustomSettingService.saveBatch(customSettings);
        log.info("发版客制包完成");
    }
}
