package com.digiwin.athena.semc.service.workbench.designer.service;

import cn.hutool.core.io.FileUtil;
import cn.hutool.core.util.ArrayUtil;
import cn.hutool.core.util.StrUtil;
import cn.hutool.core.util.ZipUtil;
import com.alibaba.fastjson.JSONObject;
import com.baomidou.mybatisplus.core.toolkit.CollectionUtils;
import com.digiwin.athena.appcore.exception.BusinessException;
import com.digiwin.athena.semc.dto.workbench.req.WorkbenchDeployAppReqDto;
import com.digiwin.athena.semc.service.workbench.constant.WorkBenchConstant;
import com.digiwin.athena.semc.service.workbench.designer.mongo.WorkBenchAppCustomSetting;
import com.digiwin.athena.semc.service.workbench.designer.mongo.WorkBenchComponent;
import com.digiwin.athena.semc.service.workbench.designer.mongo.WorkBenchMenu;
import com.digiwin.athena.semc.service.workbench.designer.mongo.WorkBenchMenuItem;
import com.digiwin.athena.semc.service.workbench.designer.mongo.WorkBenchPortal;
import com.digiwin.athena.semc.service.workbench.designer.mongo.WorkBenchPortalContent;
import com.digiwin.athena.semc.service.workbench.designer.mongo.WorkBenchRelatedApp;
import com.digiwin.athena.semc.service.workbench.designer.mongo.WorkBenchThumbnail;
import com.digiwin.athena.semc.service.workbench.designer.po.WorkbenchDesignerData;
import com.digiwin.athena.semc.util.SafeFileUtils;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections4.MapUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Service;
import org.springframework.web.multipart.MultipartFile;

import java.io.File;
import java.io.InputStream;
import java.nio.charset.StandardCharsets;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

@Service
@Slf4j
public class WorkbenchDeployService {
    @Value("${compile.zipPath}")
    private String compileZipPath;

    @Value("${compile.dataPath}")
    private String compileDataPath;

    @Autowired
    private WorkbenchDeployHandler workbenchDeployHandler;

    /**
     * 发版工作台应用
     *
     * @param deployAppReqDto 工作台应用code
     * @param compileDataFile 应用编译数据
     */
    public void deployApp(WorkbenchDeployAppReqDto deployAppReqDto, MultipartFile compileDataFile) {
        try {
            // 解析出设计时数据
            WorkbenchDesignerData designerData = parseDesignerData(deployAppReqDto, compileDataFile);
            if (null == designerData) {
                log.warn("没有编译数据，结束发版，applicationCode:{},deployNo:{}", deployAppReqDto.getApplication(),
                        deployAppReqDto.getDeployNo());
                return;
            }

            workbenchDeployHandler.deploy(deployAppReqDto, designerData);
        }
        catch (Exception ex) {
            ex.printStackTrace();
            throw BusinessException.create("部署工作台失败", ex);
        }
        finally {
            removeUnzipCompileDirectory(deployAppReqDto.getApplication(), deployAppReqDto.getCompileVersion());
        }
    }

    public WorkbenchDesignerData parseDesignerData(WorkbenchDeployAppReqDto deployAppReqDto, MultipartFile compareDataFile) {
        try (InputStream inStream = compareDataFile.getInputStream()) {
            log.info("获取到编译包");
            // 从编译包中解析出json、缩略图等文件
            File compileDataDirector = unzipCompileFile(deployAppReqDto.getApplication(), deployAppReqDto.getCompileVersion(), inStream);
            log.info("开始解析编译数据,compileDataDirector:{}", compileDataDirector.list());

            Map<String, File[]> compileFileMap = new HashMap<>();
            loopFiles(compileDataDirector, 1, compileFileMap);
            log.info("compileFileMap:{}", compileFileMap);
            if (MapUtils.isEmpty(compileFileMap)) {
                return null;
            }

            WorkbenchDesignerData designerData = new WorkbenchDesignerData();
            // 应用客制配置
            List<WorkBenchAppCustomSetting> customSettings = readLines(compileFileMap.get(WorkBenchConstant.WB_APP_CUSTOM_SETTING), WorkBenchAppCustomSetting.class);
            designerData.setCustomSettings(customSettings);

            // 关联应用
            List<WorkBenchRelatedApp> relatedApps = readLines(compileFileMap.get(WorkBenchConstant.WB_RELATED_APP), WorkBenchRelatedApp.class);
            designerData.setRelatedApps(relatedApps);

            // 自定义部件
            List<WorkBenchComponent> components = readLines(compileFileMap.get(WorkBenchConstant.WB_COMPONENT), WorkBenchComponent.class);
            designerData.setComponents(components);

            // 门户
            List<WorkBenchPortal> portals = readLines(compileFileMap.get(WorkBenchConstant.WB_PORTAL), WorkBenchPortal.class);
            designerData.setPortals(portals);

            // 门户内容
            List<WorkBenchPortalContent> portalContents = readLines(compileFileMap.get(WorkBenchConstant.WB_PORTAL_CONTENT), WorkBenchPortalContent.class);
            designerData.setPortalContents(portalContents);

            // 菜单
            List<WorkBenchMenu> menus = readLines(compileFileMap.get(WorkBenchConstant.WB_MENU), WorkBenchMenu.class);
            designerData.setMenus(menus);

            // 菜单项
            List<WorkBenchMenuItem> menuItems = readLines(compileFileMap.get(WorkBenchConstant.WB_MENU_ITEM), WorkBenchMenuItem.class);
            designerData.setMenuItems(menuItems);

            // 缩略图
            List<WorkBenchThumbnail> thumbnails = readLines(compileFileMap.get(WorkBenchConstant.WB_THUMBNAIL), WorkBenchThumbnail.class);
            designerData.setThumbnails(thumbnails);

            designerData.setThumbnailFiles(compileFileMap.get(WorkBenchConstant.THUMBNAIL_FOLDER));

            designerData.setCarouselFiles(compileFileMap.get(WorkBenchConstant.CAROUSEL_FOLDER));

            return designerData;
        } catch (Exception ex) {
            ex.printStackTrace();
            throw BusinessException.create("部署工作台失败", ex);
        }
    }
//
//    private <T> List<T> readLines (File[] files, Class<T> clz) throws IOException {
//        if (ArrayUtil.isEmpty(files)) {
//            return new ArrayList<>();
//        }
//
//        List<T> results = new ArrayList<>();
//
//        for (File file : files) {
//            readJsonFile(file, clz);
//        }
//        return results;
//    }

//    // 读取为单个对象
//    public static <T> T readJsonFile(File file, Class<T> clazz) throws IOException {
//        ObjectMapper objectMapper = new ObjectMapper();
//        return objectMapper.readValue(file, clazz);
//    }
//
//    // 读取为对象列表
//    public static <T> List<T> readJsonArrayFile(String filePath, Class<T> clazz) throws IOException {
//        ObjectMapper objectMapper = new ObjectMapper();
//        return objectMapper.readValue(new File(filePath),
//                objectMapper.getTypeFactory().constructCollectionType(List.class, clazz));
//    }

    private <T> List<T> readLines(File[] files, Class<T> clz) {
        if (ArrayUtil.isEmpty(files)) {
            return new ArrayList<>();
        }

        List<T> results = new ArrayList<>();
        for (File file : files) {
            T result = readFirstLine(file, clz);
            if (null != result) {
                results.add(result);
            }
        }
        return results;
    }

    private <T> T readFirstLine(File file, Class<T> clz) {
        List<String> lines = FileUtil.readLines(file, StandardCharsets.UTF_8);
        lines = null != lines ? lines.stream().filter(StringUtils::isNotBlank).collect(Collectors.toList()) : new ArrayList<>();
        return CollectionUtils.isNotEmpty(lines) ? JSONObject.parseObject(lines.get(0), clz) : null;
    }

    private void loopFiles(File file, int depth, Map<String, File[]> compileFileMap) {
        log.info("开始解析文件 {}", file.getAbsolutePath());
        if (!file.isDirectory()) {
            return;
        }

        String dirName = file.getName();
        if (WorkBenchConstant.WB_COLLECTION_LIST.contains(dirName) || StringUtils.equals(WorkBenchConstant.THUMBNAIL_FOLDER, dirName)
                || StringUtils.equals(WorkBenchConstant.CAROUSEL_FOLDER, dirName)) {
            File[] files = file.listFiles();
            if (ArrayUtil.isNotEmpty(files)) {
                compileFileMap.put(dirName, files);
            }
        } else if (depth < WorkBenchConstant.LOOP_FILE_MAX_DEPTH) {
            File[] subFiles = file.listFiles();
            if (ArrayUtil.isNotEmpty(subFiles)) {
                for (File subFile : subFiles) {
                    loopFiles(subFile, ++depth, compileFileMap);
                }
            }
        }
    }

    /**
     * 解压编译包至本地
     *
     * @param application    工作台应用code
     * @param compileVersion 编译版本
     * @param inStream       文件流
     * @return 解压后的本地文件夹
     */
    private File unzipCompileFile(String application, String compileVersion, InputStream inStream) {
        String applicationCompileZipPath = compileZipPath + File.separator + application + File.separator;
        log.info("applicationCompileZipPath {}", applicationCompileZipPath);
        String applicationCompileDataPath = compileDataPath + File.separator + application + File.separator;
        log.info("applicationCompileDataPath {}", applicationCompileDataPath);
        String compileDataZipPath = StrUtil.format(applicationCompileZipPath + application + "_{}.zip", compileVersion);
        log.info("compileDataZipPath {}", compileDataZipPath);
        FileUtil.writeFromStream(inStream, compileDataZipPath);
        ZipUtil.unzip(compileDataZipPath, applicationCompileDataPath);
        return SafeFileUtils.createFile(applicationCompileDataPath);
    }

    /**
     * 删除本地解压的编译文件目录
     *
     * @param application    工作台应用code
     * @param compileVersion 编译版本
     */
    private void removeUnzipCompileDirectory(String application, String compileVersion) {
        // 杀出zip包
        FileUtil.del(StrUtil.format(compileZipPath + File.separator + application + File.separator + application + "_{}.zip", compileVersion));
        // 删除解压文件夹
        FileUtil.del(compileDataPath + File.separator + application + File.separator);
    }
}
