package com.digiwin.athena.semc.service.workbench.designer.service;

import cn.hutool.core.collection.CollectionUtil;
import cn.hutool.core.lang.Tuple;
import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.digiwin.athena.appcore.util.SnowflakeIdWorker;
import com.digiwin.athena.semc.dto.workbench.req.TenantUser;
import com.digiwin.athena.semc.entity.menu.CustomizedMenuTemplate;
import com.digiwin.athena.semc.entity.menu.CustomizedMenuWork;
import com.digiwin.athena.semc.entity.portal.AppCustomSetting;
import com.digiwin.athena.semc.entity.portal.LabelSystemCustom;
import com.digiwin.athena.semc.entity.portal.PortalDeployMessage;
import com.digiwin.athena.semc.entity.portal.PortalInfo;
import com.digiwin.athena.semc.entity.portal.PortalInfoContent;
import com.digiwin.athena.semc.entity.portal.RelatedApp;
import com.digiwin.athena.semc.service.menu.CustomizedMenuTemplateService;
import com.digiwin.athena.semc.service.menu.CustomizedMenuWorkService;
import com.digiwin.athena.semc.service.portal.AppCustomSettingService;
import com.digiwin.athena.semc.service.portal.LabelSystemCustomService;
import com.digiwin.athena.semc.service.portal.PortalDeployMessageService;
import com.digiwin.athena.semc.service.portal.PortalInfoContentService;
import com.digiwin.athena.semc.service.portal.PortalInfoService;
import com.digiwin.athena.semc.service.portal.RelatedAppService;
import com.digiwin.athena.semc.service.portal.WorkbenchImageService;
import com.digiwin.athena.semc.service.workbench.constant.EnvOperateEnum;
import com.digiwin.athena.semc.service.workbench.util.WorkbenchSyncDataConverter;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.function.Function;
import java.util.stream.Collectors;

@Service
@Slf4j
public class WorkbenchSwitchHandler extends AbstractWorkbenchSwitchHandler {
    @Autowired
    private PortalInfoService portalService;

    @Autowired
    private PortalInfoContentService portalContentService;

    @Autowired
    private CustomizedMenuTemplateService menuService;

    @Autowired
    private CustomizedMenuWorkService menuItemService;

    @Autowired
    private LabelSystemCustomService customComponentService;

    @Autowired
    private RelatedAppService relatedAppService;

    @Autowired
    private WorkbenchImageService imageService;

    @Autowired
    private AppCustomSettingService appCustomSettingService;

    @Autowired
    private PortalDeployMessageService portalDeployMessageService;

    @Override
    public void switchWorkbench(String application, List<TenantUser> users) {
        // 查询 1.0版本的所有门户
        List<PortalInfo> portals = portalService
                .lambdaQuery()
                .eq(PortalInfo::getWorkbenchAppCode, application)
                .eq(PortalInfo::getVersion, EnvOperateEnum.PUBLISH.getVersion()).list();
        if (CollectionUtils.isEmpty(portals)) {
            log.warn("{}没有要切版的门户，跳出切版操作", application);
            return;
        }

        // 数据版本，如：1.0、2.0
        String version = EnvOperateEnum.SWITCH.getVersion();

        // 切版客制包
        switchCustomSetting(application);

        // 切版关联应用表
        switchRelatedApps(application);

        // 切版自定义部件数据，返回<ISV自定义部件ID，自定义部件ID>
        Tuple tuple = switchComponents(application);

        // 切版门户
        switchPortals(application, tuple.get(0), tuple.get(1));

        //删除消息
        deletePortalDeployMessages(application, portals, version);
    }

    /**
     * 入库未读消息
     * @param portals
     */
    private void deletePortalDeployMessages(String application, List<PortalInfo> portals, String version){
        if (CollectionUtils.isEmpty(portals)) {
            log.info("{}无有效内容项、菜单项，跳出入库未读消息", application);
            return;
        }
        Set<Long> portalIdset = portals.stream().map(e -> e.getPortalDesignerId()).collect(Collectors.toSet());
        LambdaQueryWrapper<PortalDeployMessage> lambdaQueryWrapper = new LambdaQueryWrapper<>();
        lambdaQueryWrapper.in(PortalDeployMessage::getPortalId,portalIdset).eq(PortalDeployMessage::getVersion,version);
        portalDeployMessageService.remove(lambdaQueryWrapper);
    }

    private void switchCustomSetting(String application) {
        log.info("切版客制包");
        // 查询1.0数据
        List<AppCustomSetting> publishCustomSettingList = appCustomSettingService.lambdaQuery()
                                                                             .eq(AppCustomSetting::getApplication, application)
                                                                             .eq(AppCustomSetting::getVersion, EnvOperateEnum.PUBLISH.getVersion()).list();
        if (CollectionUtils.isEmpty(publishCustomSettingList)) {
            return;
        }

        Set<AppCustomSetting> releaseAppCustomSettingList = publishCustomSettingList.stream()
                                                               .map(WorkbenchSyncDataConverter::toReleaseAppCustomSetting)
                                                               .collect(Collectors.toSet());

        // 根据application删除2.0数据
        appCustomSettingService.remove(Wrappers.<AppCustomSetting>lambdaQuery()
                                         .eq(AppCustomSetting::getApplication, application)
                                         .eq(AppCustomSetting::getVersion, EnvOperateEnum.SWITCH.getVersion())
        );
        // 新增application的2.0数据
        appCustomSettingService.saveBatch(releaseAppCustomSettingList);
    }

    private void switchRelatedApps(String application) {
        log.info("切版门户应用表");
        // 查询1.0数据
        List<RelatedApp> publishRelatedApps = relatedAppService.lambdaQuery()
                .eq(RelatedApp::getApplication, application)
                .eq(RelatedApp::getVersion, EnvOperateEnum.PUBLISH.getVersion())
                .list();
        if (CollectionUtils.isEmpty(publishRelatedApps)) {
            return;
        }

        Set<RelatedApp> releaseRelatedApps = publishRelatedApps.stream()
                .map(WorkbenchSyncDataConverter::toNewReleaseRelatedApp)
                .collect(Collectors.toSet());

        // 根据application删除2.0数据
        relatedAppService.remove(Wrappers.<RelatedApp>lambdaQuery()
                .eq(RelatedApp::getApplication, application)
                .eq(RelatedApp::getVersion, EnvOperateEnum.SWITCH.getVersion())
        );
        // 新增application的2.0数据
        relatedAppService.saveBatch(releaseRelatedApps);
    }

    /**
     * @description: TODO
     * @param publishCompIdMap 1.0 部件运行时id，部件设计时id
     * @param releaseCompDesIdMap 2.0 部件设计时id，部件运行时id
     * @return 
     * @author: sunyfa
     */
    private void switchPortals(String application, Map<Long, String> publishCompIdMap, Map<String, Long> releaseCompDesIdMap) {
        log.info("切版门户");
        // 根据application查询1.0门户数据
        List<PortalInfo> publishPortals = portalService.lambdaQuery()
                .eq(PortalInfo::getWorkbenchAppCode, application)
                .eq(PortalInfo::getVersion, EnvOperateEnum.PUBLISH.getVersion())
                .list();
        if (CollectionUtils.isEmpty(publishPortals)) {
            return;
        }

        // 切版门户
        Map<Long, PortalInfo> portalMap = switchPortal(application);

        // 切版门户内容
        switchPortalContent(application, portalMap);

        // 切版菜单模板 2.0 设计时菜单模板id，菜单模板运行时对象
        Map<Long, CustomizedMenuTemplate> menuTemplateMap = switchMenuTemplate(application);

        // 切版菜单项
        switchMenu (application, menuTemplateMap);
    }

    public Map<Long, PortalInfo> switchPortal (String application) {
        // 查询 1.0 门户
        List<PortalInfo> publishPortalList = portalService
                .lambdaQuery()
                .in(PortalInfo::getWorkbenchAppCode, application)
                .eq(PortalInfo::getVersion, EnvOperateEnum.PUBLISH.getVersion()).list();

        // 1.0 门户设计时id，门户运行时对象
        Map<Long, PortalInfo> publishPortalMap = publishPortalList.stream().collect(
                Collectors.toMap(PortalInfo::getPortalDesignerId, Function.identity()));

        // 查询 2.0 门户
        List<PortalInfo> releasePortalList = portalService.lambdaQuery()
                                                          .in(PortalInfo::getWorkbenchAppCode, application)
                                                          .eq(PortalInfo::getVersion, EnvOperateEnum.SWITCH.getVersion())
                                                          .list();

        // 2.0 门户设计时id，门户运行时对象
        Map<Long, PortalInfo> releasePortalMap = releasePortalList.stream().collect(
                Collectors.toMap(PortalInfo::getPortalDesignerId, Function.identity()));

        // 2.0 设计时id，运行时对象
        Map<Long, PortalInfo> portalMap = new HashMap<>();

        // 过滤出需要更新的门户设计时id
        Set<Long> updateDesPortalIdSet =
                CollectionUtil.intersectionDistinct(releasePortalMap.keySet(), publishPortalMap.keySet());
        if (CollectionUtils.isNotEmpty(updateDesPortalIdSet)) {
            List<PortalInfo> updatePortalList = new ArrayList<>();

            updateDesPortalIdSet.stream().forEach(desPortalId -> {
                PortalInfo releaseUpdatePortalInfo =
                        WorkbenchSyncDataConverter.toUpdateReleasePortal(releasePortalMap.get(desPortalId),
                                publishPortalMap.get(desPortalId));
                updatePortalList.add(releaseUpdatePortalInfo);
                portalMap.put(publishPortalMap.get(desPortalId).getId(), releaseUpdatePortalInfo);
            });

            // 更新门户
            portalService.updateBatchById(updatePortalList);
        }

        // 过滤出需要新增的菜单模板设计时id
        List<Long> addDesPortalIdList = CollectionUtil.subtractToList(publishPortalMap.keySet(), releasePortalMap.keySet());
        if(CollectionUtils.isNotEmpty(addDesPortalIdList)){
            List<PortalInfo> addPortalList = new ArrayList<>();

            addDesPortalIdList.stream().forEach(desPortalId -> {
                PortalInfo releaseUpdatePortalInfo =
                        WorkbenchSyncDataConverter.toNewReleasePortal(publishPortalMap.get(desPortalId));
                addPortalList.add(releaseUpdatePortalInfo);
                portalMap.put(publishPortalMap.get(desPortalId).getId(), releaseUpdatePortalInfo);
            });

            // 更新菜单模板
            portalService.saveBatch(addPortalList);
        }

        log.info("切版门户完成");
        return portalMap;
    }

    public void switchPortalContent (String application, Map<Long, PortalInfo> portalMap) {
        // 查询 1.0 门户内容
        List<PortalInfoContent> publishPortalContentList = portalContentService.lambdaQuery().in(
                PortalInfoContent::getWorkbenchAppCode, application).eq(PortalInfoContent::getVersion,
                EnvOperateEnum.PUBLISH.getVersion()).list();

        // 1.0 门户设计时id，门户运行时对象
        Map<Long, PortalInfoContent> publishPortalContentMap = publishPortalContentList.stream().collect(
                Collectors.toMap(PortalInfoContent::getDesignerId, Function.identity()));

        // 删除所有2.0门户内容
        portalContentService.remove(Wrappers.<PortalInfoContent>lambdaQuery()
                                            .eq(PortalInfoContent::getWorkbenchAppCode, application)
                                            .eq(PortalInfoContent::getVersion, EnvOperateEnum.SWITCH.getVersion()));

        // 2.0 部件设计时id，部件运行时id
        if (null != publishPortalContentMap && !publishPortalContentMap.isEmpty()) {
            List<PortalInfoContent> portalInfoContentList = new ArrayList<>();

            publishPortalContentMap.entrySet().forEach(entry -> {
                PortalInfoContent releaseUpdatePortalContentInfo = WorkbenchSyncDataConverter.toNewReleasePortalContent(
                        entry.getValue(), portalMap.get(entry.getValue().getPortalId()));
                portalInfoContentList.add(releaseUpdatePortalContentInfo);
            });

            portalContentService.saveBatch(portalInfoContentList);
        }

        log.info("切版门户完成");
    }

    /**
     * @description: 切版菜单模板
     * @param
     * @return 
     * @author: sunyfa
     */
    public Map<Long, CustomizedMenuTemplate> switchMenuTemplate (String application) {
        log.info("切版菜单模板");

        // 查询 2.0 菜单模板
        List<CustomizedMenuTemplate> releaseMenuTemplateList = menuService.lambdaQuery()
                .in(CustomizedMenuTemplate::getWorkbenchAppCode, application)
                .eq(CustomizedMenuTemplate::getVersion, EnvOperateEnum.SWITCH.getVersion())
                .list();

        // 设计时id，运行时对象
        Map<Long, CustomizedMenuTemplate> releaseMenuTemplateMap = releaseMenuTemplateList.stream().collect(
                Collectors.toMap(CustomizedMenuTemplate::getMenuDesignerId, Function.identity()));

        // 查询 1.0 菜单模板
        List<CustomizedMenuTemplate> publishMenuTemplateList = menuService.lambdaQuery().in(
                CustomizedMenuTemplate::getWorkbenchAppCode, application).eq(CustomizedMenuTemplate::getVersion,
                EnvOperateEnum.PUBLISH.getVersion()).list();

        Map<Long, CustomizedMenuTemplate> publishMenuTemplateMap = publishMenuTemplateList.stream().collect(
                Collectors.toMap(CustomizedMenuTemplate::getMenuDesignerId, Function.identity()));

        // 2.0 运行时id，运行时对象
        Map<Long, CustomizedMenuTemplate> menuTemplateMap = new HashMap<>();

        // 过滤出需要更新的菜单模板设计时id
        Set<Long> updateDesMenuIdSet =
                CollectionUtil.intersectionDistinct(releaseMenuTemplateMap.keySet(), publishMenuTemplateMap.keySet());
        if (CollectionUtils.isNotEmpty(updateDesMenuIdSet)) {
            List<CustomizedMenuTemplate> updateMenuTemplateList = new ArrayList<>();

            updateDesMenuIdSet.stream().forEach(desMenuCompId -> {
                CustomizedMenuTemplate releaseUpdateMenuTemplate = WorkbenchSyncDataConverter.toUpdateReleaseMenu(
                        publishMenuTemplateMap.get(desMenuCompId), releaseMenuTemplateMap.get(desMenuCompId).getId());
                updateMenuTemplateList.add(releaseUpdateMenuTemplate);
                menuTemplateMap.put(publishMenuTemplateMap.get(desMenuCompId).getId(), releaseUpdateMenuTemplate);
            });

            // 更新菜单模板
            menuService.updateBatchById(publishMenuTemplateList);
        }

        // 过滤出需要新增的菜单模板设计时id
        List<Long> addDesMenuIdList = CollectionUtil.subtractToList(publishMenuTemplateMap.keySet(),
                releaseMenuTemplateMap.keySet());
        if(CollectionUtils.isNotEmpty(addDesMenuIdList)){
            List<CustomizedMenuTemplate> addMenuTemplateList = new ArrayList<>();

            addDesMenuIdList.stream().forEach(desMenuCompId -> {
                CustomizedMenuTemplate releaseAddMenuTemplate =
                        WorkbenchSyncDataConverter.toNewReleaseMenu(publishMenuTemplateMap.get(desMenuCompId));
                addMenuTemplateList.add(releaseAddMenuTemplate);
                menuTemplateMap.put(publishMenuTemplateMap.get(desMenuCompId).getId(), releaseAddMenuTemplate);
            });

            // 更新菜单模板
            menuService.saveBatch(addMenuTemplateList);
        }

        return menuTemplateMap;
    }

    public void switchMenu (String application, Map<Long, CustomizedMenuTemplate> menuTemplateMap) {
        // 删除已发版的菜单项
        menuItemService.remove(Wrappers.<CustomizedMenuWork>lambdaQuery()
                                       .eq(CustomizedMenuWork::getWorkbenchAppCode, application)
                                       .eq(CustomizedMenuWork::getVersion, EnvOperateEnum.SWITCH.getVersion()));

        // 查询 1.0 菜单项
        List<CustomizedMenuWork> releaseMenuList = menuItemService.lambdaQuery()
                                                                          .eq(CustomizedMenuWork::getWorkbenchAppCode, application)
                                                                          .eq(CustomizedMenuWork::getVersion, EnvOperateEnum.PUBLISH.getVersion())
                                                                          .list();

        if (CollectionUtils.isNotEmpty(releaseMenuList)) {
            List<CustomizedMenuWork> addReleaseMenuList = new ArrayList<>();

            Map<Long, Long> releaseMenuIdMap = new HashMap<>();
            Map<Long, String> releaseMenuNameMap = new HashMap<>();
            releaseMenuList.stream().forEach(releaseMenu -> {
                Long menuId = SnowflakeIdWorker.getInstance().newId();
                releaseMenuIdMap.put(releaseMenu.getId(), menuId);
                releaseMenuNameMap.put(releaseMenu.getId(), releaseMenu.getMenuName());

                CustomizedMenuWork newReleaseMenuItem = WorkbenchSyncDataConverter.toNewReleaseMenuItem(releaseMenu, menuId);
                newReleaseMenuItem.setTemplateId(menuTemplateMap.get(releaseMenu.getTemplateId()).getId());

                addReleaseMenuList.add(newReleaseMenuItem);
            });

            // 填充新的parentId，parentName
            addReleaseMenuList.stream().forEach(newReleaseMenuItem -> {
                newReleaseMenuItem.setParentName(releaseMenuNameMap.get(newReleaseMenuItem.getParentId()));
                newReleaseMenuItem.setParentId(releaseMenuIdMap.get(newReleaseMenuItem.getParentId()));
            });

            menuItemService.saveBatch(addReleaseMenuList);
        }

        log.info("切版菜单项完成");
    }

    private Tuple switchComponents(String application) {
        log.info("切版部件");
        // 发版的1.0客制组件列表
        List<LabelSystemCustom> publishCustomComps = customComponentService.lambdaQuery()
                .eq(LabelSystemCustom::getWorkbenchAppCode, application)
                .eq(LabelSystemCustom::getVersion, EnvOperateEnum.PUBLISH.getVersion())
                .list();
        if (CollectionUtils.isEmpty(publishCustomComps)) {
            return new Tuple(new HashMap<>(), new HashMap<>());
        }

        // 1.0 部件运行时id，部件设计时id
        Map<Long, Long> publishCompIdMap = publishCustomComps.stream()
                .collect(Collectors.toMap(LabelSystemCustom::getId, LabelSystemCustom::getComponentDesignerId));

        // 2.0 部件列表
        List<LabelSystemCustom> releaseCustomComps = queryReleaseCustomCompIds(publishCompIdMap.values());
        // 2.0 设计时id，运行时部件
        Map<Long, LabelSystemCustom> releaseCustomCompMap = new HashMap<>();
        if (CollectionUtils.isNotEmpty(releaseCustomComps)) {
            releaseCustomCompMap = releaseCustomComps.stream().collect(Collectors.toMap(LabelSystemCustom::getComponentDesignerId, Function.identity()));
        }

        // 分类新增、更新的1.0自定义部件
        List<LabelSystemCustom> addCustomComps = new ArrayList<>();
        List<LabelSystemCustom> updateCustomComps = new ArrayList<>();
        for (LabelSystemCustom publishCustomComp : publishCustomComps) {
            if (releaseCustomCompMap.keySet().contains(publishCustomComp.getComponentDesignerId())) {
                updateCustomComps.add(publishCustomComp);
            } else {
                addCustomComps.add(publishCustomComp);
            }
        }

        // 切过版的更新
        if (CollectionUtils.isNotEmpty(updateCustomComps)) {
            Set<LabelSystemCustom> toUpdateCustomComps = new HashSet<>();
            for(LabelSystemCustom publishCustomComp : updateCustomComps){
                LabelSystemCustom releaseCustomComp = releaseCustomCompMap.get(publishCustomComp.getComponentDesignerId());
                toUpdateCustomComps.add(WorkbenchSyncDataConverter.toUpdateReleaseCustomComponent(releaseCustomComp, publishCustomComp));
            }

            customComponentService.updateBatchById(toUpdateCustomComps);
        }

        if (CollectionUtils.isNotEmpty(addCustomComps)) {
            Set<LabelSystemCustom> releaseCustomComp = addCustomComps.stream()
                    .map(WorkbenchSyncDataConverter::toNewReleaseCustomComponent).collect(Collectors.toSet());
            customComponentService.saveBatch(releaseCustomComp);
        }

        // 需要返回：1.0版本部件  <运行时部件id，设计时部件id>
        // 需要返回：2.0版本部件  <设计时部件id，运行部件id>
        List<LabelSystemCustom> allReleaseCustomComps = queryReleaseCustomCompIds(publishCompIdMap.values());

        // 2.0 部件设计时id，部件运行时id
        Map<Long, Long> releaseCompDesIdMap = allReleaseCustomComps.stream()
                .collect(Collectors.toMap(LabelSystemCustom::getComponentDesignerId, LabelSystemCustom::getId));
        return new Tuple(publishCompIdMap, releaseCompDesIdMap);
    }

    private List<LabelSystemCustom> queryReleaseCustomCompIds(Collection<Long> customCompDesIds) {
        return customComponentService.lambdaQuery()
                .in(LabelSystemCustom::getComponentDesignerId, customCompDesIds)
                .eq(LabelSystemCustom::getVersion, EnvOperateEnum.SWITCH.getVersion())
                .list();
    }
}
