package com.digiwin.athena.semc.service.workbench.impl;

import com.digiwin.athena.appcore.auth.AppAuthContextHolder;
import com.digiwin.athena.semc.dto.workbench.req.WorkbenchDeployAppReqDto;
import com.digiwin.athena.semc.dto.workbench.req.WorkbenchSwitchAppReqDto;
import com.digiwin.athena.semc.entity.portal.AppCustomSetting;
import com.digiwin.athena.semc.entity.portal.PortalInfo;
import com.digiwin.athena.semc.service.portal.AppCustomSettingService;
import com.digiwin.athena.semc.service.portal.PortalInfoService;
import com.digiwin.athena.semc.service.workbench.WorkbenchAppService;
import com.digiwin.athena.semc.service.workbench.constant.WorkBenchConstant;
import com.digiwin.athena.semc.service.workbench.designer.service.WorkbenchDeleteService;
import com.digiwin.athena.semc.service.workbench.designer.service.WorkbenchDeployService;
import com.digiwin.athena.semc.service.workbench.designer.service.WorkbenchSwitchService;
import com.digiwin.athena.semc.service.workbench.util.WorkbenchDataConverter;
import com.digiwin.athena.semc.vo.workbench.AppCustomSettingVO;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.web.multipart.MultipartFile;

import java.util.ArrayList;
import java.util.List;
import java.util.stream.Collectors;

@Slf4j
@Service
public class WorkbenchAppServiceImpl implements WorkbenchAppService {
    @Autowired
    private WorkbenchDeployService appDeployService;

    @Autowired
    private WorkbenchSwitchService workbenchSwitchService;

    @Autowired
    private AppCustomSettingService appCustomSettingService;

    @Autowired
    WorkbenchDeleteService workbenchDeleteService;

    @Autowired
    private PortalInfoService portalInfoService;

    @Override
    public void deleteApp(String appCode) {
        AppAuthContextHolder.getContext().getAuthoredUser().setTenantId(WorkBenchConstant.SYSTEM_TENANT);
        workbenchDeleteService.delete(appCode);
    }

    @Override
    public void deletePortal(String appCode, Long portalId) {
        AppAuthContextHolder.getContext().getAuthoredUser().setTenantId(WorkBenchConstant.SYSTEM_TENANT);
        workbenchDeleteService.deletePortal(appCode, portalId);
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void deployApp(WorkbenchDeployAppReqDto deployAppReq, MultipartFile compileDataFile) {
        AppAuthContextHolder.getContext().getAuthoredUser().setTenantId(WorkBenchConstant.SYSTEM_TENANT);

        log.info("开始发版工作台{}-{}", deployAppReq.getApplication(), deployAppReq.getDeployNo());
        long startTime = System.currentTimeMillis();

        appDeployService.deployApp(deployAppReq, compileDataFile);
        log.info("部署工作台{}-{}结束，耗时 {}ms", deployAppReq.getApplication(), deployAppReq.getDeployNo(), System.currentTimeMillis() - startTime);
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void switchApp(WorkbenchSwitchAppReqDto switchAppReq) {
        AppAuthContextHolder.getContext().getAuthoredUser().setTenantId(WorkBenchConstant.SYSTEM_TENANT);

        log.info("开始切版工作台{}-{}", switchAppReq.getApplication(), switchAppReq.getDeployNo());
        long startTime = System.currentTimeMillis();

        workbenchSwitchService.switchWorkbench(switchAppReq.getApplication(), switchAppReq.getTenantUsers());
        log.info("切版工作台{}-{}结束，耗时 {}ms", switchAppReq.getApplication(), switchAppReq.getDeployNo(), System.currentTimeMillis() - startTime);
    }

    @Override
    public List<AppCustomSettingVO> getAppCustomSetting(String application) {
        List<AppCustomSetting> appCustomSettings = appCustomSettingService.lambdaQuery()
                .eq(AppCustomSetting::getApplication, application)
                .list();
        if (CollectionUtils.isEmpty(appCustomSettings)) {
            return new ArrayList<>();
        }

        return appCustomSettings.stream()
                .map(WorkbenchDataConverter.INSTANCE::toAppCustomSetting)
                .collect(Collectors.toList());
    }

    @Override
    public List<PortalInfo> getPortalInfos(List<Long> portalIdList) {
        AppAuthContextHolder.getContext().getAuthoredUser().setTenantId(WorkBenchConstant.SYSTEM_TENANT);
        List<PortalInfo> portalInfoList = portalInfoService.lambdaQuery().in(PortalInfo::getPortalDesignerId,
                portalIdList).list();

        if (CollectionUtils.isEmpty(portalInfoList)) {
            return new ArrayList<>();
        }

        return portalInfoList;
    }
}
