package com.digiwin.athena.semc.util;

import com.digiwin.athena.appcore.util.TimeUtils;
import org.apache.commons.lang3.StringUtils;

import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.time.LocalDateTime;
import java.time.format.DateTimeFormatter;
import java.util.Calendar;
import java.util.Date;

/**
 * @ClassName DateUtils
 * @Description 时间处理类
 * @Author sungq
 * @Date 2022/11/16
 **/
public class DateUtils {
    public static final String DATE_TIME_NORMAL_FORMATTER = "yyyy-MM-dd HH:mm:ss";

    public static final String DATE_NORMAL_FORMATTER = "yyyy-MM-dd";

    public static final String DATE_NORMAL_FORMATTER_YYYYMMDD = "yyyyMMdd";
    public static final String DATE_NORMAL_FORMATTER_YYYYMM = "yyyyMM";

    public static final String DATE_NORMAL_FORMATTER_YYYY_MM = "yyyy-MM";

    public static final String TIME_NORMAL_FORMATTER = "HH:mm:ss";

    public static final String DATE_TIME_FORMATTER = "yyyyMMddHHmmss";

    public static final String YEAR = "year";

    public static final String MONTH = "month";

    public static final String DAY = "day";

    public static final String HOUR = "hour";

    public static final String MINUTE = "minute";

    public static final String SECOND = "second";

    public static final String WEEK = "week";

    public static final String START_TIME = " 00:00:00";

    public static final String END_TIME =  " 23:59:59";

    public static String getNowTime() {
        return getNowTime(null);
    }

    /**
     * 根据指定格式获取当前时间
     *
     * @param format 日期目标格式
     * @return 时间
     */
    public static String getNowTime(String format) {
        if (StringUtils.isBlank(format)) {
            format = DATE_TIME_NORMAL_FORMATTER;
        }
        return LocalDateTime.now().format(DateTimeFormatter.ofPattern(format));
    }

    /**
     * @param dateStr 时间基点
     * @param pattern 时间格式
     * @param step    相加的时长
     * @param type    相加的类型 "year"年、"month"月、"day"日、”hour“时、”minute“分、”second“秒、"week"周
     * @return 相加后的时间
     * @description: 按时间格式相加
     * 输入要相加的时间基点（字符串型或时间类型），相加的时长（整型），格式（"year"年、"month"月、"day"日、”hour“时、”minute“分、”second“秒、"week"周）
     * 输出相加后的时间（字符串型或时间类型）
     * @author: sunyfa
     */
    public static String addDate(String dateStr, String pattern, int step, String type) {
        if (dateStr == null) {
            return dateStr;
        } else {
            Date date1 = TimeUtils.parseDate(dateStr, pattern);

            Calendar cal = Calendar.getInstance();
            cal.setTime(date1);

            switch (type) {
                case YEAR:
                    cal.add(Calendar.YEAR, step);
                    break;
                case MONTH:
                    cal.add(Calendar.MONTH, step);
                    break;
                case WEEK:
                    cal.add(Calendar.WEEK_OF_MONTH, step);
                    break;
                case DAY:
                    cal.add(Calendar.DAY_OF_MONTH, step);
                    break;
                case HOUR:
                    cal.add(Calendar.HOUR, step);
                    break;
                case MINUTE:
                    cal.add(Calendar.MINUTE, step);
                    break;
                case SECOND:
                    cal.add(Calendar.SECOND, step);
                    break;
                default:
            }

            return TimeUtils.format(cal.getTime(), pattern);
        }
    }

    /**
     * @param beginDate 开始时间
     * @param endDate   结束时间
     * @return 0 beginTime < endTime； 1 beginTime == endTime； 2 beginTime > endTime
     * @description: 比较两个时间
     * @author: sunyfa
     */
    public static int compare(Date beginDate, Date endDate) {
        int ret = 1;
        long beginTime = beginDate.getTime();
        long endTime = endDate.getTime();

        if (beginTime > endTime) {
            ret = 2;
        }
        if (beginTime == endTime) {
            ret = 1;
        }
        if (beginTime < endTime) {
            ret = 0;
        }
        return ret;
    }

    /**
     * 校验时间格式是否正确
     *
     * @param dateParam 校验的日期
     * @param format    校验的格式
     * @return
     */
    public static boolean validateDate(String dateParam, String format) {
        SimpleDateFormat formatter = new SimpleDateFormat(format);
        try {
            Date date = formatter.parse(dateParam);
        } catch (Exception e) {
            return false;
        }
        return true;
    }

    /**
     * 计算两个时间相差的时间
     *
     * @param nowDate 开始日期
     * @param endDate 截止日期
     * @return
     */
    public static String calDateDifference(Date nowDate, Date endDate) {
        long nd = 1000L * 24L * 60L * 60L;
        long nh = 1000L * 60L * 60L;
        long nm = 1000L * 60L;
        // long ns = 1000;
        // 获得两个时间的毫秒时间差异
        long diff = endDate.getTime() - nowDate.getTime();
        // 计算差多少天
        long day = diff / nd;
        // 计算差多少小时
        long hour = diff % nd / nh;
        // 计算差多少分钟
        long min = diff % nd % nh / nm;
        // 计算差多少秒//输出结果
        // long sec = diff % nd % nh % nm / ns;
        return day + "天" + hour + "小时" + min + "分钟";
    }

    /**
     * date 转 String
     *
     * @param date
     * @return
     */
    public static String DateToString(Date date) {
        SimpleDateFormat sdf = new SimpleDateFormat(DATE_TIME_NORMAL_FORMATTER);
        String dateString = sdf.format(date);
        return dateString;
    }

    /**
     * String转date
     *
     * @param dateTime
     * @return
     */
    public static Date StringToDate(String dateTime) {
        SimpleDateFormat format = new SimpleDateFormat(DATE_TIME_NORMAL_FORMATTER);
        try {
            Date date = format.parse(dateTime);
            return date;
        } catch (ParseException e) {
            throw new RuntimeException(e);
        }
    }

    /**
     * 将String类型的字符串日期转为util.Date类型的日期
     * @param date 传入的String类型的日期
     * @return util.Date类型的日期
     */
    public static java.util.Date strToDate(String date) {
        SimpleDateFormat sdf = new SimpleDateFormat(DATE_TIME_NORMAL_FORMATTER);
        java.util.Date d = null;
        try {
            d = sdf.parse(date);
        } catch (ParseException e) {
            throw new RuntimeException(e);
        }
        return d;

    }

    /**
     * 获取当前时间的年和月份
     * @return
     */
    public static  String getCurrentDateYM(){
        // 获取当前日期
        Date currentDate = new Date();
        // 创建SimpleDateFormat实例，指定日期格式
        SimpleDateFormat sdf = new SimpleDateFormat(DATE_NORMAL_FORMATTER_YYYY_MM);
        // 格式化日期为字符串
        String formattedDate = sdf.format(currentDate);
        return  formattedDate;
    }

    /**
     * 获取当前时间的年月日
     * @return
     */
    public static  String getCurrentDateYMD(){
        // 获取当前日期
        Date currentDate = new Date();
        // 创建SimpleDateFormat实例，指定日期格式
        SimpleDateFormat sdf = new SimpleDateFormat(DATE_NORMAL_FORMATTER_YYYYMMDD);
        // 格式化日期为字符串
        String formattedDate = sdf.format(currentDate);
        return  formattedDate;
    }

    /**
     * 获取当前时间的年月
     * @return
     */
    public static String getCurrentDateYYYYMM(){
        // 获取当前日期
        Date currentDate = new Date();
        // 创建SimpleDateFormat实例，指定日期格式
        SimpleDateFormat sdf = new SimpleDateFormat(DATE_NORMAL_FORMATTER_YYYYMM);
        // 格式化日期为字符串
        String formattedDate = sdf.format(currentDate);
        return  formattedDate;
    }


    /**
     *日期格式转换
     * @param input 输入的值
     * @param nowFormat 当前日期格式
     * @param targetFormat 日期目标格式
     * @return
     */
    public static  String getDateFormat(String input,String nowFormat,String targetFormat) {
        // 输入日期时间字符串的格式
        String output = "";
        try {
            SimpleDateFormat formatIn = new SimpleDateFormat(nowFormat);
            Date date = formatIn.parse(input); // 解析输入字符串得到Date对象
            // 输出日期时间字符串的格式
            SimpleDateFormat formatOut = new SimpleDateFormat(targetFormat);
            // 格式化Date对象为指定格式的字符串
            output = formatOut.format(date);
        } catch (ParseException e) {
            throw new RuntimeException(e);
        }
        return output;
    }

    /**
     * <一段时间后（年，月，日，时，分，秒）>
     * <功能详细描述>
     *
     * @param date
     * @param number
     * @return
     * @see [类、类#方法、类#成员]
     */
    public static Date getNewDate(Date date, int... number) {
        Calendar calendar = Calendar.getInstance();
        calendar.setTime(date);
        if (number.length > 0) {
            calendar.set(Calendar.YEAR, calendar.get(Calendar.YEAR) + number[0]);
            if (number.length > 1) {
                calendar.set(Calendar.MONTH, calendar.get(Calendar.MONTH) + number[1]);
                if (number.length > 2) {

                    calendar.set(Calendar.DAY_OF_MONTH, calendar.get(Calendar.DAY_OF_MONTH) + number[2]);
                    if (number.length > 3) {
                        calendar.set(Calendar.HOUR_OF_DAY, calendar.get(Calendar.HOUR_OF_DAY) + number[3]);
                        if (number.length > 4) {
                            calendar.set(Calendar.MINUTE, calendar.get(Calendar.MINUTE) + number[4]);
                            if (number.length > 5) {
                                calendar.set(Calendar.SECOND, calendar.get(Calendar.SECOND) + number[5]);
                            }
                        }
                    }
                }
            }
        }
        return calendar.getTime();
    }

}