package com.digiwin.athena.semc.util;

import lombok.extern.slf4j.Slf4j;

import java.io.File;
import java.nio.file.InvalidPathException;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.Arrays;
import java.util.List;

@Slf4j
public class SafeFileUtils {

    // 定义黑名单列表，包含系统关键目录
    private static final List<String> BLACKLISTED_DIRECTORIES =
            Arrays.asList("/etc", "/bin", "/sbin", "/root", "/dev", "/proc");


    // 校验路径是否合法，防止路径穿越和非法字符
    public static boolean isValidPath(String path) {
        if (path == null || path.isEmpty()) {
            log.error("current absolute path is empty ,path is:{}", path);
            return true;
        }

        try {
            // 使用 Java NIO 的 Path 解析来确保路径是有效的
            Path resolvedPath = Paths.get(path).toAbsolutePath();
            // 检查解析后的路径是否有意外的外部路径穿越
            if (!resolvedPath.normalize().toString().equals(resolvedPath.toString())) {
                return true;
            }

            // 检查路径是否包含黑名单目录
            for (String blacklistedDir : BLACKLISTED_DIRECTORIES) {
                if (resolvedPath.startsWith(blacklistedDir)) {
                    log.error("Path starts with a blacklisted directory: {}", blacklistedDir);
                    return true;
                }
            }
        } catch (InvalidPathException e) {
            log.error("Invalid path format: {}", path, e);
            return true;
        }

        return false;
    }


    // 提供统一的文件创建接口
    public static File createFile(String path) {
        if (isValidPath(path)) {
            throw new IllegalArgumentException("Invalid file path: " + path);
        }

        return new File(path);
    }


    public static String filterPath(String path) {
        if (isValidPath(path)) {
            throw new IllegalArgumentException("Invalid file path: " + path);
        }
        return path;
    }

}
