package com.digiwin.athena.semc.util;



import com.alibaba.fastjson.JSONObject;
import com.alibaba.fastjson.serializer.SerializerFeature;

import org.apache.commons.io.FileUtils;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.MDC;

import java.io.File;
import java.nio.charset.StandardCharsets;
import java.util.Arrays;
import java.util.Base64;
import java.util.Comparator;
import java.util.Date;
import java.util.Map;

import javax.servlet.http.HttpServletRequest;

import cn.hutool.core.date.DateUtil;
import lombok.Data;
import lombok.extern.slf4j.Slf4j;


/**
 * 安全日志工具类
 */
@Slf4j
public class SecurityLogUtil {

    @Data
    public static class SecurityLog {
        private String errorType = "[cyberSecurity]";

        private String timestamp = DateUtil.now();

        private Integer code = 500;

        /**
         * 安全告警类别：纵向越权或横向越权
         */
        private String message;

        /**
         * 用户尝试调用的接口
         */
        private String path;

        private Boolean success = false;

        /**
         * 应用
         */
        private String sourceId="semc";

        /**
         * 安全的告警码，5位数字，本期需求都是10001
         */
        private Integer errorCode = 10001;

        private SecurityLogErrorMessage errorMessage;

        private Map<String, Object> errorInstructors;

        private String traceId;

    }

    @Data
    public static class SecurityLogErrorMessage {
        private String type = "[cyberSecurity]";

        private String userId;

        private String userName;

        private String tenantId;

        private String userIp;

        private String rawRequest;

        private String userToken;

        private String appKey;

        private String logEnd = "[cyberSecurityLogEnd]";
    }


    public enum MessageEnum {
        HORIZONTAL(1, "横向越权"),
        VERTICAL(2, "纵向越权");

        private Integer code;
        private String value;

        MessageEnum(Integer code, String value) {
            this.code = code;
            this.value = value;
        }

        public Integer getCode() {
            return code;
        }

        public String getValue() {
            return value;
        }
    }

    /**
     * 写入纵向越权安全日志
     * @param request    请求
     * @param rawRequest 请求参数
     */
    public static void writeSecurityLog(HttpServletRequest request, String rawRequest) {
        writeSecurityLog(MessageEnum.VERTICAL, request, rawRequest);
    }


    /**
     * 写入安全日志
     * @param message 告警类别
     * @param request 请求
     * @param rawRequest 请求参数
     */
    public static void writeSecurityLog(MessageEnum message, HttpServletRequest request,String rawRequest){
        //写入安全日志
        SecurityLog securityLog = new SecurityLog();
        securityLog.setMessage(message.getValue());
        securityLog.setPath(request.getServletPath());
        securityLog.setTraceId(MDC.get("PtxId"));
        SecurityLogErrorMessage securityLogErrorMessage = new SecurityLogErrorMessage();
        securityLogErrorMessage.setUserId(Utils.getUserId());
        securityLogErrorMessage.setUserName(Utils.getUserName());
        securityLogErrorMessage.setUserIp(WebToolUtils.getRemortIP(request));
        securityLogErrorMessage.setTenantId(Utils.getTenantId());
        securityLogErrorMessage.setRawRequest(rawRequest);
        String userToken = request.getHeader("Token");
        if (StringUtils.isBlank(userToken)) {
            userToken = request.getHeader("digi-middleware-auth-user");
        }
        securityLogErrorMessage.setUserToken(userToken);
        securityLogErrorMessage.setAppKey(request.getHeader("digi-middleware-auth-app"));
        securityLog.setErrorMessage(securityLogErrorMessage);
        writeSecurityLog(securityLog);
    }


    public static void writeSecurityLog(SecurityLog securityLog) {
        //头尾顺序一定要保持一致，为了避免低版本不支持顺序，使用手动方式指定
        JSONObject jsonObject = new JSONObject(true);
        jsonObject.put("errorType", securityLog.getErrorType());
        jsonObject.put("timestamp", securityLog.getTimestamp());
        jsonObject.put("code", securityLog.getCode());
        jsonObject.put("Message", securityLog.getMessage());
        jsonObject.put("Path", securityLog.getPath());
        jsonObject.put("Success", securityLog.getSuccess());
        jsonObject.put("sourceId", securityLog.getSourceId());
        jsonObject.put("traceId", securityLog.getTraceId());
        jsonObject.put("errorCode", securityLog.getErrorCode());
        jsonObject.put("errorInstructors", securityLog.getErrorInstructors());
        SecurityLogErrorMessage errorMessage = securityLog.getErrorMessage();
        JSONObject errorMessageJson = new JSONObject(true);
        errorMessageJson.put("type", errorMessage.getType());
        errorMessageJson.put("userID", errorMessage.getUserId());
        errorMessageJson.put("userName", errorMessage.getUserName());
        errorMessageJson.put("userIp", errorMessage.getUserIp());
        if (StringUtils.isNotBlank(errorMessage.getRawRequest())) {
            errorMessageJson.put("rawRequest", Base64.getEncoder().encodeToString(errorMessage.getRawRequest().getBytes(StandardCharsets.UTF_8)));
        } else {
            errorMessageJson.put("rawRequest", errorMessage.getRawRequest());
        }
        errorMessageJson.put("userToken", errorMessage.getUserToken());
        errorMessageJson.put("appKey", errorMessage.getAppKey());
        errorMessageJson.put("logEnd", errorMessage.getLogEnd());
        jsonObject.put("errorMessage", errorMessageJson);
        writeSecurityLog(jsonObject.toString(SerializerFeature.WriteMapNullValue));


    }


    private static File getSecurityFile() {
        String logPath = "/usr/local/semc/.logs/error/";
        String logName = "log-security-";
        File logDirectory = new File(logPath);
        if (!logDirectory.exists()) {
            logDirectory.mkdirs();
        }
        String nowDate = DateUtil.formatDate(new Date());
        File[] files = logDirectory.listFiles(file -> file.isFile() && file.getName().startsWith(logName + nowDate));
        if (files == null || files.length == 0) {
            File newLog = new File(logPath + logName + nowDate + ".0.log");
            return newLog;
        }
        File file = Arrays.stream(files).max(Comparator.comparing(File::getName)).get();
        long fileSize = file.length();
        if (fileSize < 20 * 1024 * 1024) {
            return file;
        }
        String seq = file.getName().split("\\.")[1];
        return new File(logPath + logName + nowDate + "." + (Integer.parseInt(seq)+1) + ".log");
    }

    private static void writeSecurityLog(String content) {
        File logFile = getSecurityFile();
        try {
            FileUtils.writeStringToFile(logFile, content + System.lineSeparator(), StandardCharsets.UTF_8, true);
        } catch (Exception e) {
            log.error("安全日志写入异常：{}，{}", logFile.getName(),content, e);
        }
    }


}
