package com.digiwin.athena.semc.util;

import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.digiwin.athena.appcore.auth.AppAuthContextHolder;
import com.digiwin.athena.appcore.auth.domain.AuthoredUser;
import com.digiwin.athena.semc.common.Constants;
import com.digiwin.athena.semc.common.enums.BizAuthTypeEnum;
import com.digiwin.athena.semc.dto.news.SaveAnnouncementReq;
import com.digiwin.athena.semc.dto.news.UpdateNewsStatusReq;
import com.digiwin.athena.semc.entity.news.NewsAnnouncementAut;
import com.digiwin.athena.semc.mapper.news.NewsAnnouncementAuthMapper;
import com.digiwin.athena.semc.proxy.athena.service.audc.AudcService;
import com.digiwin.athena.semc.proxy.athena.service.audc.model.UserTrackExtendDTO;

import org.apache.commons.collections.CollectionUtils;
import org.springframework.stereotype.Component;

import java.time.LocalDateTime;
import java.time.ZoneId;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.Executor;
import java.util.stream.Collectors;

import javax.annotation.Resource;

import lombok.extern.slf4j.Slf4j;

/**
 * 埋点工具类
 *
 * @author chenpeng
 */
@Slf4j
@Component
public class TrackingUtil {
    @Resource
    private AudcService audcService;
    @Resource
    private NewsAnnouncementAuthMapper newsAnnouncementAuthMapper;

    @Resource(name = "semcAsyncExecutor")
    private Executor asyncTaskExecutor;

    /**
     * 保存公告埋点
     *
     * @param saveAnnouncementReq
     */
    public void reportUserTrackForSaveNews(SaveAnnouncementReq saveAnnouncementReq) {
        AuthoredUser authoredUser = AppAuthContextHolder.getContext().getAuthoredUser();
        CompletableFuture.runAsync(() -> {
            // 不是发布状态不埋点
            if (!Constants.NewsAnnouncementStatusEnum.PUBLISHED.getFlag().equals(saveAnnouncementReq.getNewsStatus())) {
                return;
            }
            // 不是全员可见，并且没有外部用户 不埋点
            if (!Constants.NEWS_ALL_AUTH_FLAG_YES.equals(saveAnnouncementReq.getAllAuthFlag()) && CollectionUtils.isEmpty(saveAnnouncementReq.getExternalUserList())) {
                return;
            }
            // 只有包含外部用户，并且是发布状态才埋点
            List<UserTrackExtendDTO> userTrackExtendDTOS = buildRequestParam(authoredUser, "function_analysis", "news", null, String.valueOf(saveAnnouncementReq.getId()), "外部用户公告");
            audcService.reportUserTrack(authoredUser, userTrackExtendDTOS);
        },asyncTaskExecutor);

    }

    /**
     * 更新公告发布状态埋点
     *
     * @param updateNewsStatusReq
     */
    public void reportUserTrackForUpdatePublishStatus(UpdateNewsStatusReq updateNewsStatusReq) {
        AuthoredUser authoredUser = AppAuthContextHolder.getContext().getAuthoredUser();
        CompletableFuture.runAsync(() -> {
            // 不是发布状态不埋点
            if (!Constants.NewsAnnouncementStatusEnum.PUBLISHED.getFlag().equals(updateNewsStatusReq.getNewsStatus())) {
                return;
            }
            // 公告id为空不埋点
            if (CollectionUtils.isEmpty(updateNewsStatusReq.getIdList())) {
                return;
            }
            // 公告id查询公告发布范围
            QueryWrapper<NewsAnnouncementAut> condition = new QueryWrapper<>();
            condition.in("news_id", updateNewsStatusReq.getIdList());
            List<NewsAnnouncementAut> authList = newsAnnouncementAuthMapper.selectList(condition);
            if (CollectionUtils.isEmpty(authList)) {
                return;
            }
            // newsId分组
            Map<Long, List<NewsAnnouncementAut>> authMap = authList.stream().collect(Collectors.groupingBy(NewsAnnouncementAut::getNewsId));
            // 循环map，公告发布范围包含外部用户的调用埋点接口
            List<UserTrackExtendDTO> requestParamList = new ArrayList<>();
            for (Map.Entry<Long, List<NewsAnnouncementAut>> entry : authMap.entrySet()) {
                List<NewsAnnouncementAut> value = entry.getValue();
                Optional<NewsAnnouncementAut> any = value.stream().filter(e -> Constants.NEWS_ALL_AUTH_FLAG_YES.equals(e.getAllAuthFlag()) || BizAuthTypeEnum.EXTERNAL_USER.getValue().equals(e.getAuthType()))
                        .findAny();
                if (any.isPresent()) {
                    requestParamList.addAll(buildRequestParam(authoredUser, "function_analysis", "news", null, String.valueOf(entry.getKey()), "外部用户公告"));
                }
            }
            audcService.reportUserTrack(authoredUser, requestParamList);
        },asyncTaskExecutor);

    }

    /**
     * 组装请求参数
     * @param authoredUser
     * @param source       performance_analysis:性能分析, function_analysis:功能分析, null:用户行为上报
     * @param workType     作业类型，如：任务 task， 项目：Project，报表：report，基础资料录入：base-data
     * @param appCode      应用code
     * @param code         任务code/页面唯一标识/作业唯一标识
     * @param description  描述
     * @return
     */
    private List<UserTrackExtendDTO> buildRequestParam(AuthoredUser authoredUser, String source, String workType, String appCode, String code, String description) {
        List<UserTrackExtendDTO> list = new ArrayList<>();
        UserTrackExtendDTO extendDTO = new UserTrackExtendDTO();
        extendDTO.setSource(source);
        extendDTO.setWorkType(workType);
        extendDTO.setWorkCode(code);
        extendDTO.setUserId(authoredUser.getUserId());
        extendDTO.setTenantId(authoredUser.getTenantId());
        Long createTime = LocalDateTime.now().atZone(ZoneId.systemDefault()).toInstant().toEpochMilli();
        extendDTO.setTimestamp(createTime);
        Map<String, Object> workContent = new HashMap<>();
        workContent.put("code", code);
        workContent.put("description", description);
        Map<String, Object> attachData = new HashMap<>();
        attachData.put("appCode", appCode);
        extendDTO.setWorkContent(workContent);
        extendDTO.setAttachData(attachData);
        list.add(extendDTO);
        return list;
    }
}
