package com.digiwin.dap.middle.cloud.conditional;

import org.springframework.beans.factory.annotation.AnnotatedBeanDefinition;
import org.springframework.beans.factory.config.BeanDefinition;
import org.springframework.beans.factory.config.ConfigurableListableBeanFactory;
import org.springframework.context.annotation.Condition;
import org.springframework.context.annotation.ConditionContext;
import org.springframework.context.annotation.Conditional;
import org.springframework.context.annotation.Configuration;
import org.springframework.core.type.AnnotatedTypeMetadata;
import org.springframework.core.type.AnnotationMetadata;
import org.springframework.core.type.MethodMetadata;
import org.springframework.util.CollectionUtils;
import org.springframework.util.ReflectionUtils;

import java.lang.annotation.Annotation;
import java.lang.reflect.Method;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;

/**
 * <p>判断容器是否存在存在配置注解的bean的逻辑实现
 *
 * <p>使用场景：用于判断{@link org.springframework.context.annotation.Bean}注解注入的{@link BeanDefinition}
 * 是否存在有相关注解配置存在。需要注意的是，这里植入spring容器的时间点与spring判断{@link Conditional}系列注解在一起。
 * 所以容易出现{@link BeanDefinition}的注册顺序与预期不一致，导致注解没有达到预期效果的情况，需要注意。
 * 使用{@link org.springframework.context.annotation.ComponentScan}扫包注入的{@link Configuration}
 * 配置类就可以解决这个顺序问题，保证需要被判断的BD在自动化注入之前被加入spring容器。
 * 或者业务场景允许的前提下，所有相关BD的注入都添加相应的{@link ConditionalOnMissingBeanWithAnnotation}注解</>
 *
 * <p>使用样例：
 * 项目已经注册的BD：
 * <pre>{@code
 * @Bean
 * @LoadBalanced RestTemplate restTemplate() {
 * return new RestTemplate();
 * }
 * }
 * </pre>
 *
 * <p>自动化注入的BD，就不会被注入容器了：
 * <pre>{@code
 * @Bean
 * @LoadBalanced
 * @ConditionalOnMissingBeanWithAnnotation(annotation={LoadBalanced.class}) RestTemplate lbRestTemplate() {
 * return new RestTemplate();
 * }
 * }
 *  </pre>
 *
 * @author blockWilling
 * @date 2023/5/9 16:40
 * @mail kangjin@digiwin.com
 */
public class MissingBeanWithAnnotationCondition implements Condition {
    public static final String MISSING_BEAN_WITH_ANNOTATION_NAME = ConditionalOnMissingBeanWithAnnotation.class.getName();
    public static final String MISSING_BEAN_WITH_ANNOTATION_ATTR_VALUE = "value";
    public static final String MISSING_BEAN_WITH_ANNOTATION_ATTR_ANNOTATION = "annotation";


    @Override
    public boolean matches(ConditionContext context, AnnotatedTypeMetadata metadata) {

        try {
            //避免重复判断配置类
            List<String> skipClassNames = new LinkedList<>();
            Map<String, Object> methodMetadataAttrs = metadata.getAnnotationAttributes(MISSING_BEAN_WITH_ANNOTATION_NAME);
            if (CollectionUtils.isEmpty(methodMetadataAttrs)) {
                return false;
            }
            Object value = methodMetadataAttrs.get(MISSING_BEAN_WITH_ANNOTATION_ATTR_VALUE);
            Class<?> returnType = (Class) value;
            if (returnType == Void.class) {
                //获取@Bean方法依附的方法元数据
                if (metadata instanceof MethodMetadata) {
                    MethodMetadata methodMetadata = (MethodMetadata) metadata;
                    String returnTypeName = methodMetadata.getReturnTypeName();
                    returnType = Class.forName(returnTypeName);
                } else {
                    return false;
                }
            }
            ConfigurableListableBeanFactory beanFactory = context.getBeanFactory();
            String[] beanDefinitionNames = beanFactory.getBeanDefinitionNames();
            boolean foundMatchedMethod = false;
            //遍历容器所有的BD名
            for (String key : beanDefinitionNames) {
                try {
                    foundMatchedMethod = false;
                    //过滤自身配置类
                    BeanDefinition beanDefinition = beanFactory.getBeanDefinition(key);
                    String beanClassName;
                    if (metadata instanceof MethodMetadata) {
                        MethodMetadata methodMetadata = (MethodMetadata) metadata;
                        if ((beanClassName = beanDefinition.getBeanClassName()) != null && beanClassName.equals(methodMetadata.getDeclaringClassName())) {
                            continue;
                        }
                    } else {
                        continue;
                    }

                    if (beanDefinition instanceof AnnotatedBeanDefinition) {
                        AnnotatedBeanDefinition annotatedBeanDefinition = (AnnotatedBeanDefinition) beanDefinition;
                        AnnotationMetadata meta = annotatedBeanDefinition.getMetadata();
                        String className = meta.getClassName();
                        if (skipClassNames.contains(className)) {
                            continue;
                        }
    //                    methods = ReflectUtil.getMethods(Class.forName(className));
                        Method[] methods = ReflectionUtils.getDeclaredMethods(Class.forName(className));
                        //遍历@Bean所在的配置类的方法,过滤不同class类型的@Bean方法
                        for (Method method : methods) {
                            if (returnType.getName().equals(method.getReturnType().getName())) {
                                foundMatchedMethod = true;
                                break;
                            }
                        }
                        skipClassNames.add(className);
                        if (foundMatchedMethod) {
                            //匹配了@Bean类型之后，开始匹配注解
                            Class<? extends Annotation>[] annos = (Class<? extends Annotation>[]) methodMetadataAttrs.get(MISSING_BEAN_WITH_ANNOTATION_ATTR_ANNOTATION);
                            for (Class<? extends Annotation> anno : annos) {
                                if (meta.hasAnnotatedMethods(anno.getName())) {
                                    return false;
                                }
                            }
                        }
                    }
                } catch (Throwable ex) {
                    continue;
                }
            }
            return true;
        } catch (Throwable ex) {
            return false;
        }
    }
}
