package com.digiwin.dap.middleware.commons.util;

import java.nio.charset.Charset;
import java.nio.charset.StandardCharsets;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

/**
 * 字符串工具类
 *
 * @author fobgochod
 * @since 1.0.0
 */
public class StrUtils {

    public static final Charset UTF_8 = StandardCharsets.UTF_8;

    private StrUtils() {
    }

    /**
     * 检查字符串是不为空
     *
     * @param str 对象
     * @return 是否为空串
     */
    public static boolean isEmpty(String str) {
        return str == null || str.isEmpty();
    }

    /**
     * 检查字符串是不为空
     *
     * @param str 对象
     * @return 是否不为空串
     */
    public static boolean isNotEmpty(String str) {
        return str != null && !str.isEmpty();
    }

    /**
     * 将byte数组转为字符串
     *
     * @param bytes byte数组
     * @return 字符串
     */
    public static String str(byte[] bytes) {
        return str(bytes, UTF_8);
    }

    /**
     * 解码字节码
     *
     * @param data    字符串
     * @param charset 字符集，如果此字段为空，则解码的结果取决于平台
     * @return 解码后的字符串
     */
    public static String str(byte[] data, Charset charset) {
        if (data == null) {
            return null;
        }

        if (null == charset) {
            return new String(data);
        }
        return new String(data, charset);
    }

    /**
     * 编码字符串
     *
     * @param str 字符串
     * @return 编码后的字节码
     */
    public static byte[] bytes(String str) {
        return bytes(str, UTF_8);
    }

    /**
     * 编码字符串
     *
     * @param str     字符串
     * @param charset 字符集，如果此字段为空，则解码的结果取决于平台
     * @return 编码后的字节码
     */
    public static byte[] bytes(String str, Charset charset) {
        if (str == null) {
            return new byte[0];
        }
        return str.getBytes(charset);
    }

    /**
     * 根据正则表达式判断字符是否有效
     *
     * @param str     对象
     * @param pattern 正则
     * @return 是否匹配
     */
    public static boolean isValid(String str, String pattern) {
        if (isEmpty(str) || isEmpty(pattern)) {
            return false;
        }
        Matcher matcher = Pattern.compile(pattern).matcher(str);
        return matcher.matches();
    }

    /**
     * 字符是大小写字母，数字
     *
     * @param str 对象
     * @return 是否匹配
     */
    public static boolean isLetterOrNumber(String str) {
        return isValid(str, "^[A-Za-z0-9]+$");
    }

    /**
     * 字符是小写字母，数字，1-255位
     *
     * @param str 对象
     * @return 是否匹配
     */
    public static boolean isLowerLetterOrNumber(String str) {
        return isValid(str, "^[a-z0-9]{1,255}$");
    }

    /**
     * 生成随机码
     *
     * @return java.lang.String
     * @author michael
     * @since 1.0.8
     **/
    @SuppressWarnings({"findsecbugs:PREDICTABLE_RANDOM", "java:S2245"})
    public static String getRandomCode() {
        StringBuilder randomCode = new StringBuilder();
        int i = 0;
        while (i < 4) {
            // 52个字母与6个大小写字母间的符号；范围为91~96
            int value = (int) (Math.random() * 58 + 65);
            while (value >= 91 && value <= 96) {
                value = (int) (Math.random() * 58 + 65);
            }
            randomCode.append((char) value);
            i++;
        }
        return randomCode.toString();
    }
}
