package com.digiwin.dap.middleware.commons.core.codec;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.math.BigInteger;
import java.nio.charset.StandardCharsets;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;

/**
 * Base58工具类，提供Base58的编码和解码方案<br>
 * 参考： <a href="https://github.com/Anujraval24/Base58Encoding">Base58Encoding</a><br>
 * 规范见：<a href="https://en.bitcoin.it/wiki/Base58Check_encoding">Base58Check_encoding</a>
 *
 * @author fobgochod
 */
public class Base58 {

    private static final Logger logger = LoggerFactory.getLogger(Base58.class);
    private static final String DIGITS = "123456789ABCDEFGHJKLMNPQRSTUVWXYZabcdefghijkmnopqrstuvwxyz";

    /**
     * C#版本策略ID(policy)加密模式
     *
     * @param str 被加密数据
     * @return 加密信息
     */
    public static String policyId(String str) {
        String encodeStr;
        try {
            MessageDigest messageDigest = MessageDigest.getInstance("SHA-256");
            byte[] bytes = messageDigest.digest(str.getBytes(StandardCharsets.UTF_8));
            bytes = messageDigest.digest(bytes);
            encodeStr = encode(bytes).substring(0, 20);
        } catch (Exception e) {
            throw new RuntimeException(e);
        }
        return encodeStr;
    }

    /**
     * Encodes data with a 4-byte checksum
     *
     * @param data Data to be encoded
     * @return String
     */
    public static String encode(byte[] data) {
        return encodePlain(addCheckSum(data));
    }

    /**
     * Encodes data in plain Base58, without any checksum.
     *
     * @param data The data to be encoded
     * @return String
     */
    private static String encodePlain(byte[] data) {
        // Decode byte[] to BigInteger
        BigInteger intData = new BigInteger("0");
        for (byte aData : data) {
            int d = aData & 0xff;
            intData = new BigInteger("256").multiply(intData).add(new BigInteger(d + ""));
        }
        // encode BigInteger to Base58 String
        StringBuilder result = new StringBuilder();
        while (0 != intData.intValue()) {
            BigInteger remainder = intData.mod(new BigInteger("58"));
            intData = intData.divide(new BigInteger("58"));
            result.insert(0, DIGITS.split("")[remainder.intValue()]);
        }
        // Append `1` for each leading 0 byte
        for (int i = 0; i < data.length && data[i] == 0; i++) {
            result.insert(0, '1');
        }
        return result.toString();
    }

    private static byte[] addCheckSum(byte[] data) {
        byte[] checkSum = getCheckSum(data);
        return concatArrays(data, checkSum);
    }

    private static byte[] getCheckSum(byte[] data) {
        byte[] result = new byte[4];
        try {
            MessageDigest md = MessageDigest.getInstance("SHA-256");
            byte[] hash1 = md.digest(data);
            byte[] hash2 = md.digest(hash1);
            System.arraycopy(hash2, 0, result, 0, result.length);
        } catch (NoSuchAlgorithmException e) {
            logger.error(e.getMessage(), e);
        }
        return result;
    }

    private static byte[] concatArrays(byte[] arr1, byte[] arr2) {
        byte[] result = new byte[arr1.length + arr2.length];
        System.arraycopy(arr1, 0, result, 0, arr1.length);
        System.arraycopy(arr2, 0, result, arr1.length, arr2.length);
        return result;
    }
}
