package com.digiwin.dap.middleware.repository.impl;

import com.digiwin.dap.middleware.entity.BaseEntity;
import com.digiwin.dap.middleware.repository.BaseEntityRepository;
import com.digiwin.dap.middleware.util.SnowFlake;
import org.springframework.data.jpa.repository.support.JpaEntityInformation;
import org.springframework.data.jpa.repository.support.JpaEntityInformationSupport;
import org.springframework.data.jpa.repository.support.SimpleJpaRepository;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.Assert;

import javax.persistence.EntityManager;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;

/**
 * @author chenxsa
 */
@Transactional(readOnly = true)
public class BaseEntityRepositoryImpl<T extends BaseEntity, ID> extends SimpleJpaRepository<T, ID> implements BaseEntityRepository<T, ID> {

    private final JpaEntityInformation<T, ?> entityInformation;
    private final EntityManager em;

    public BaseEntityRepositoryImpl(JpaEntityInformation<T, ?> entityInformation, EntityManager entityManager) {
        super(entityInformation, entityManager);
        this.entityInformation = entityInformation;
        this.em = entityManager;
    }

    public BaseEntityRepositoryImpl(Class<T> domainClass, EntityManager em) {
        this(JpaEntityInformationSupport.getEntityInformation(domainClass, em), em);
    }

    /**
     * 批量插入
     *
     * @param entities
     * @param <S>
     * @return
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public <S extends T> void insert(Iterable<S> entities) {
        Assert.notNull(entities, "The given Iterable of entities not be null!");

        Iterator<S> var3 = entities.iterator();
        int index = 0;
        while (var3.hasNext()) {
            S entity = var3.next();
            if (entity.getSid() == 0) {
                entity.setSid(SnowFlake.getInstance().newId());
            }
            this.em.persist(entity);
            index++;
            if (index % 500 == 0) {
                em.flush();
                em.clear();
            }
        }
        if (index % 500 != 0) {
            em.flush();
            em.clear();
        }
    }

    /**
     * 批量修改
     *
     * @param entities
     * @param <S>
     * @return
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public <S extends T> void update(Iterable<S> entities) {
        Assert.notNull(entities, "The given Iterable of entities not be null!");

        Iterator<S> var3 = entities.iterator();
        int index = 0;
        while (var3.hasNext()) {
            S entity = var3.next();
            this.em.merge(entity);
            index++;
            if (index % 500 == 0) {
                em.flush();
                em.clear();
            }
        }
        if (index % 500 != 0) {
            em.flush();
            em.clear();
        }
    }

    /**
     * 批量保存数据
     *
     * @param entities
     * @return
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public <S extends T> List<S> saveAll(Iterable<S> entities) {
        Assert.notNull(entities, "The given Iterable of entities not be null!");
        List<S> result = new ArrayList<>();
        Iterator<S> var3 = entities.iterator();
        int index = 0;
        while (var3.hasNext()) {
            S entity = var3.next();
            if (this.entityInformation.isNew(entity)) {
                entity.setSid(SnowFlake.getInstance().newId());
                this.em.persist(entity);
            } else {
                this.em.merge(entity);
            }
            index++;
            if (index % 500 == 0) {
                em.flush();
                em.clear();
            }
            result.add(entity);
        }
        if (index % 500 != 0) {
            em.flush();
            em.clear();
        }
        return result;
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public <S extends T> S save(S entity) {
        if (this.entityInformation.isNew(entity)) {
            entity.setSid(SnowFlake.getInstance().newId());
            this.em.persist(entity);
            return entity;
        } else {
            return this.em.merge(entity);
        }
    }
}
