package com.digiwin.dap.middleware.service.impl;

import com.digiwin.dap.middleware.domain.CommonErrorCode;
import com.digiwin.dap.middleware.domain.DapEnv;
import com.digiwin.dap.middleware.domain.DeployAreaEnum;
import com.digiwin.dap.middleware.entity.BaseEntityWithPartition;
import com.digiwin.dap.middleware.exception.BusinessException;
import com.digiwin.dap.middleware.repository.BaseEntityWithPartitionRepository;
import com.digiwin.dap.middleware.service.EntityWithPartitionManagerService;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;

import javax.persistence.EntityManager;
import javax.persistence.PersistenceContext;
import java.util.List;

public abstract class BaseEntityWithPartitionManagerService<T extends BaseEntityWithPartition> extends BaseEntityManagerService<T> implements EntityWithPartitionManagerService<T> {

    private static final Logger logger = LoggerFactory.getLogger(BaseEntityWithPartitionManagerService.class);

    @PersistenceContext
    private EntityManager entityManager;

    @Autowired
    private DapEnv dapEnv;

    @Override
    protected abstract BaseEntityWithPartitionRepository getRepository();


    public void checkPartition(long tenantSid) {
        if (tenantSid > 0) {
            return;
        }
        if (DeployAreaEnum.isDev(dapEnv.getDeployArea())) {
            logger.error(CommonErrorCode.PARTITION_CHECK.getErrorMessage(new Object[]{getMappingTableName()}));
            // throw new SystemException(CommonErrorCode.PARTITION_CHECK, new Object[]{getMappingTableName()});
        } else if (DeployAreaEnum.isTest(dapEnv.getDeployArea())) {
            logger.warn(CommonErrorCode.PARTITION_CHECK.getErrorMessage(new Object[]{getMappingTableName()}));
        }
    }

    @Override
    public long create(T entity) {
        this.checkPartition(entity.getTenantSid());
        return super.create(entity);
    }


    @Override
    public void deleteById(long sid) {
        this.checkPartition(0);
        super.deleteById(sid);
    }


    @Override
    public void update(T entity) {
        this.checkPartition(entity.getTenantSid());
        super.update(entity);
    }


    @Override
    public void update(T entity, boolean force) {
        this.checkPartition(entity.getTenantSid());
        super.update(entity, force);
    }


    @Override
    public T findBySid(long sid) {
        this.checkPartition(0);
        return super.findBySid(sid);
    }


    @Override
    public boolean exists(long sid) {
        this.checkPartition(0);
        return super.exists(sid);
    }


    @Override
    public void disable(long sid) {
        this.checkPartition(0);
        super.disable(sid);
    }


    @Override
    public void enable(long sid) {
        this.checkPartition(0);
        super.enable(sid);
    }


    @Override
    public void insertAll(Iterable<T> entities) {
        for (T entity : entities) {
            this.checkPartition(entity.getTenantSid());
        }
        super.insertAll(entities);
    }


    @Override
    public void updateAll(Iterable<T> entities) {
        for (T entity : entities) {
            this.checkPartition(entity.getTenantSid());
        }
        super.updateAll(entities);
    }


    @Override
    public void saveAll(Iterable<T> entities) {
        for (T entity : entities) {
            this.checkPartition(entity.getTenantSid());
        }
        super.saveAll(entities);
    }


    public List<T> findAll() {
        if (DeployAreaEnum.isTest(dapEnv.getDeployArea())) {
            logger.warn("该表[{}]是分区表，正在进行全区查询，请注意。", getMappingTableName());
        }
        return super.findAll();
    }


    @Override
    public List<T> findByTenantSid(long tenantSid) {
        return this.getRepository().findByTenantSid(tenantSid);
    }


    @Override
    public void deleteBySidAndTenantSid(long sid, long tenantSid) {
        this.getRepository().deleteBySidAndTenantSid(sid, tenantSid);
    }


    @Override
    public long getSidBySidAndTenantSid(long sid, long tenantSid) {
        BaseEntityWithPartition entity = getRepository().findBySidAndTenantSid(sid, tenantSid);
        if (entity == null) {
            return 0;
        }
        return entity.getSid();
    }


    @Override
    public T findBySidAndTenantSid(long sid, long tenantSid) {
        return (T) getRepository().findBySidAndTenantSid(sid, tenantSid);
    }


    @Override
    public boolean existsBySidAndTenantSid(long sid, long tenantSid) {
        return this.getRepository().existsBySidAndTenantSid(sid, tenantSid);
    }


    @Override
    public void enable(long sid, long tenantSid) {
        T result = (T) this.getRepository().findBySidAndTenantSid(sid, tenantSid);
        if (result != null) {
            result.setDisabled(false);
            this.getRepository().save(result);
        } else {
            throw new BusinessException("数据已经被删除.sid:" + sid + ",tenantSid:" + tenantSid);
        }
    }


    @Override
    public void disable(long sid, long tenantSid) {
        T result = (T) this.getRepository().findBySidAndTenantSid(sid, tenantSid);
        if (result != null) {
            result.setDisabled(true);
            this.getRepository().save(result);
        } else {
            throw new BusinessException("数据已经被删除.sid:" + sid + ",tenantSid:" + tenantSid);
        }
    }
}
