package com.digiwin.dap.middle.database.encrypt.desensitization.service.impl;

import com.digiwin.dap.middle.database.encrypt.desensitization.context.DesensitizationConvertContext;
import com.digiwin.dap.middle.database.encrypt.desensitization.processor.DesensitizationProcessorFactory;
import com.digiwin.dap.middle.database.encrypt.desensitization.register.DesensitizationProcessorRegistry;
import com.digiwin.dap.middle.database.encrypt.desensitization.service.DesensitizationConverter;
import com.digiwin.dap.middle.database.encrypt.enums.DatabaseEncryptExceptionEnum;
import com.digiwin.dap.middle.database.encrypt.enums.DesensitizationConvertEnum;
import com.digiwin.dap.middle.database.encrypt.exception.DatabaseEncryptException;
import com.digiwin.dap.middle.database.encrypt.model.ObjectRelationalMapping;
import com.digiwin.dap.middle.database.encrypt.model.SensitiveWordProperty;
import com.digiwin.dap.middle.database.encrypt.sensitive.word.context.SensitiveWordContext;
import com.digiwin.dap.middle.database.encrypt.sensitive.word.register.SensitiveWordDatabaseRegistry;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.util.CollectionUtils;

import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.stream.Collectors;

/**
 * @author michael
 */
public class DesensitizationConverterImpl<C> implements DesensitizationConverter<C> {

    private final static Logger LOGGER = LoggerFactory.getLogger(DesensitizationConverterImpl.class);

    private final SensitiveWordDatabaseRegistry sensitiveWordDatabaseRegistry;

    private final DesensitizationProcessorRegistry desensitizationProcessorRegistry;

    public DesensitizationConverterImpl(SensitiveWordDatabaseRegistry sensitiveWordDatabaseRegistry, DesensitizationProcessorRegistry desensitizationProcessorRegistry) {
        this.sensitiveWordDatabaseRegistry = sensitiveWordDatabaseRegistry;
        this.desensitizationProcessorRegistry = desensitizationProcessorRegistry;
    }

    @Override
    public Object desensitize(DesensitizationConvertContext<C> context) {
        if (Objects.isNull(context.getContext())) {
            LOGGER.debug("===>{}()参数为空不需要进行脱敏处理.", context.getTargetMethod());
            return context.getContext();
        }
        Map<String, SensitiveWordProperty> sensitiveWordPropertyMap = getSensitiveWordPropertyMap(context);
        if (sensitiveWordPropertyMap.isEmpty()) {
            LOGGER.debug("===>{}()参数不涉及敏感字段不需要进行脱敏处理.", context.getTargetMethod());
            return context.getContext();
        }
        DesensitizationProcessorFactory desensitizationProcessorFactory =
                new DesensitizationProcessorFactory(desensitizationProcessorRegistry, sensitiveWordPropertyMap,
                        context.getContext(), DesensitizationConvertEnum.DESENSITIZATION);
        try {
            LOGGER.debug("===>{}()对参数中{}属性进行脱敏",
                    context.getTargetMethod(), String.join(", ", sensitiveWordPropertyMap.keySet()));
            return desensitizationProcessorFactory.process();
        } catch (Exception e) {
            throw new DatabaseEncryptException(DatabaseEncryptExceptionEnum.DESENSITIZATION_FAIL, context.getTargetMethod());
        }
    }

    @Override
    public Object revert(DesensitizationConvertContext<C> context) {
        if (Objects.isNull(context.getContext())) {
            LOGGER.debug("===>{}()返回为空不需要进行还原处理.", context.getTargetMethod());
            return context.getContext();
        }
        Map<String, SensitiveWordProperty> sensitiveWordPropertyMap = getSensitiveWordPropertyMap(context);
        if (sensitiveWordPropertyMap.isEmpty()) {
            LOGGER.debug("===>{}()返回不涉及敏感字段不需要进行还原处理.", context.getTargetMethod());
            return context.getContext();
        }
        DesensitizationProcessorFactory desensitizationProcessorFactory =
                new DesensitizationProcessorFactory(desensitizationProcessorRegistry, sensitiveWordPropertyMap,
                        context.getContext(), DesensitizationConvertEnum.REVERT);
        try {
            LOGGER.debug("===>{}()对返回结果中{}属性进行还原",
                    context.getTargetMethod(), String.join(", ", sensitiveWordPropertyMap.keySet()));
            return desensitizationProcessorFactory.process();
        } catch (Exception e) {
            throw new DatabaseEncryptException(DatabaseEncryptExceptionEnum.REVERT_FAIL, context.getTargetMethod());
        }
    }

    private Map<String, SensitiveWordProperty> getSensitiveWordPropertyMap(DesensitizationConvertContext<C> context) {
        Map<String, SensitiveWordProperty> sensitiveWordPropertyMap = new HashMap<>(16);
        List<ObjectRelationalMapping> mappings = context.getObjectRelationalMappings();
        if (!CollectionUtils.isEmpty(mappings)) {
            for (Map.Entry<String, List<ObjectRelationalMapping>> entry :
                    mappings.stream().collect(Collectors.groupingBy(ObjectRelationalMapping::getTableName)).entrySet()) {
                String tableName = entry.getKey();
                SensitiveWordContext<Object> sensitiveWordContext = sensitiveWordDatabaseRegistry.findSensitiveWord(tableName);
                if (Objects.isNull(sensitiveWordContext) || CollectionUtils.isEmpty(sensitiveWordContext.getSensitiveWordPropertyList())) {
                    continue;
                }
                for (ObjectRelationalMapping mapping : entry.getValue()) {
                    for (SensitiveWordProperty sensitiveWordProperty : sensitiveWordContext.getSensitiveWordPropertyList()) {
                        if (Objects.equals(sensitiveWordProperty.getDataBaseColumnName(), mapping.getDatabaseColumnName())) {
                            sensitiveWordPropertyMap.put(mapping.getObjectPropertyName(), sensitiveWordProperty);
                        }
                    }
                }
            }
        }
        return sensitiveWordPropertyMap;
    }
}
