package com.digiwin.dap.middle.database.encrypt.sensitive.word.register;

import com.digiwin.dap.middle.database.encrypt.annotation.Desensitization;
import com.digiwin.dap.middle.database.encrypt.annotation.SensitiveField;
import com.digiwin.dap.middle.database.encrypt.model.SensitiveWordProperty;
import com.digiwin.dap.middle.database.encrypt.sensitive.word.context.SensitiveWordContext;
import com.digiwin.dap.middle.database.encrypt.utils.CamelToSnakeUtils;
import org.reflections.Reflections;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.core.env.Environment;
import org.springframework.util.StringUtils;

import javax.annotation.PostConstruct;
import javax.persistence.Column;
import javax.persistence.Entity;
import javax.persistence.Table;
import java.lang.reflect.Field;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.CopyOnWriteArrayList;


/**
 * @author michael
 */
public class DefaultSensitiveWordDatabaseRegistry implements SensitiveWordDatabaseRegistry {

    private final static Logger LOGGER = LoggerFactory.getLogger(DefaultSensitiveWordDatabaseRegistry.class);

    private final Environment environment;

    public DefaultSensitiveWordDatabaseRegistry(Environment environment) {
        this.environment = environment;
    }

    private final Map<String, SensitiveWordContext<Object>> sensitiveWordContextMap = new ConcurrentHashMap<>();

    @Override
    public SensitiveWordContext<Object> findSensitiveWord(String tableName) {
        if (sensitiveWordContextMap.containsKey(tableName)) {
            return sensitiveWordContextMap.get(tableName);
        }
        return null;
    }

    @PostConstruct
    public void run() {
        LOGGER.debug("===>初始化数据库敏感词库");
        String packageName = environment.getProperty("mybatis.typeAliasesPackage");
        Reflections reflections = new Reflections(packageName);
        // 1.获取所有实体类
        Set<Class<?>> entityClasses = reflections.getTypesAnnotatedWith(Entity.class);
        if (Objects.isNull(entityClasses) || entityClasses.isEmpty()) {
            LOGGER.error("===>扫描【{}】未获取到实体类.", packageName);
        }
        for (Class<?> entityClass : entityClasses) {
            // 2.判断是否包含敏感信息
            if (!entityClass.isAnnotationPresent(Desensitization.class)) {
                continue;
            }
            // 2.判断是否包含@Table
            if (!entityClass.isAnnotationPresent(Table.class)) {
                continue;
            }
            // 3.获取表名
            Table table = entityClass.getAnnotation(Table.class);
            String tableName = table.name();

            // 4.判断是否脱敏
            Desensitization desensitization = entityClass.getAnnotation(Desensitization.class);
            if (!desensitization.enabled()) {
                LOGGER.warn("===>【{}】标注敏感字段,但是未开启脱敏!", entityClass.getName());
                continue;
            }

            // 4.遍历敏感信息获取敏感字段已经脱敏规则
            Field[] fields = entityClass.getDeclaredFields();
            for (Field field : fields) {
                field.setAccessible(true);
                SensitiveField sensitiveField = field.getAnnotation(SensitiveField.class);
                if (Objects.isNull(sensitiveField)) {
                    continue;
                }
                String entityPropertyName = field.getName();
                Column column = field.getAnnotation(Column.class);
                String dataBaseColumnName = column != null && StringUtils.hasLength(column.name()) ? column.name() : CamelToSnakeUtils.convertCamelToSnake(entityPropertyName);
                LOGGER.debug("===>数据库表【{}】字段【{}】为敏感字段,属性名为【{}】", tableName, dataBaseColumnName, field.getName());
                SensitiveWordProperty sensitiveWordProperty = new SensitiveWordProperty(tableName.toLowerCase(), dataBaseColumnName, entityPropertyName, sensitiveField.mode(), sensitiveField.checker());
                if (!sensitiveWordContextMap.containsKey(tableName)) {
                    sensitiveWordContextMap.put(tableName, new SensitiveWordContext<>(tableName.toLowerCase(), new CopyOnWriteArrayList<>()));
                }
                sensitiveWordContextMap.get(tableName).getSensitiveWordPropertyList().add(sensitiveWordProperty);
            }
        }
        LOGGER.debug("===>数据库敏感词库初始化完成");
    }
}
