package com.digiwin.dap.middle.database.encrypt.registry.desensitization;

import com.digiwin.dap.middle.database.encrypt.enums.DesensitizationMode;
import com.digiwin.dap.middle.database.encrypt.annotation.DesensitizationStrategy;
import com.digiwin.dap.middle.database.encrypt.processor.desensitization.DesensitizationProcessor;
import org.springframework.beans.BeansException;
import org.springframework.beans.factory.config.BeanFactoryPostProcessor;
import org.springframework.beans.factory.config.ConfigurableListableBeanFactory;
import org.springframework.core.annotation.AnnotatedElementUtils;
import org.springframework.stereotype.Component;

import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;

/**
 * @author michael
 * @version 1.0.0
 * @see
 */
@Component
public class DefaultDesensitizationProcessorRegistry implements DesensitizationProcessorRegistry, BeanFactoryPostProcessor {

    private Map<String, DesensitizationProcessor> desensitizationProcessorMap = new ConcurrentHashMap<>();

    @Override
    public DesensitizationProcessor findDesensitizationProcessor(DesensitizationMode desensitizationMode) {
        if (!desensitizationProcessorMap.containsKey(desensitizationMode.name())) {
            throw new RuntimeException("No desensitization processor found for " + desensitizationMode.name());
        }
        return desensitizationProcessorMap.get(desensitizationMode.name());
    }

    @Override
    public void postProcessBeanFactory(ConfigurableListableBeanFactory beanFactory) throws BeansException {
        for (String beanName : beanFactory.getBeanNamesForAnnotation(DesensitizationStrategy.class)) {
            Object bean = beanFactory.getBean(beanName);
            if (bean instanceof DesensitizationProcessor) {
                DesensitizationStrategy desensitizationStrategy = AnnotatedElementUtils.findMergedAnnotation(bean.getClass(), DesensitizationStrategy.class);
                DesensitizationMode desensitizationMode = desensitizationStrategy.desensitizationMode();
                desensitizationProcessorMap.put(desensitizationMode.name(), (DesensitizationProcessor) bean);
            }
        }
    }
}
