package com.digiwin.dap.middle.database.encrypt.desensitization;

import com.digiwin.dap.middle.database.encrypt.enums.DesensitizationMode;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import java.lang.reflect.Field;
import java.util.List;
import java.util.Map;

/**
 * @author michael
 */
@Component
public class DesensitizationProcessorFactory {

    private final static Logger LOGGER = LoggerFactory.getLogger(DesensitizationProcessorFactory.class);

    private DefaultDesensitizationProcessorRegistry desensitizationProcessorRegistry;

    @Autowired
    public DesensitizationProcessorFactory(@Autowired DefaultDesensitizationProcessorRegistry desensitizationProcessorRegistry) {
        this.desensitizationProcessorRegistry = desensitizationProcessorRegistry;
    }

    /**
     * <p>对结果集进行还原
     * <p>1.jpa返回结果集可以进行还原的数据类型:String、List<String>、实体类、List<实体类></p>
     * <p>2.mybatis返回结果集可以进行还原的数据类型:String、List<String>、对象、List<对象>、map</p>
     * <p/>
     *
     * @param fieldName
     * @param result
     * @param fieldAndDesensitizationModeMap
     * @return java.lang.Object
     * @author michael
     * @date 2024/6/19 11:28
     **/
    public Object revert(String fieldName, Object result, Map<String, DesensitizationMode> fieldAndDesensitizationModeMap) {
        try {
            if (fieldAndDesensitizationModeMap == null || fieldAndDesensitizationModeMap.isEmpty() || result == null) {
                LOGGER.warn("===>敏感词库或者结果集为空,不需要进行还原");
                return result;
            }
            if (result instanceof List) {
                for (Object o : (List<?>) result) {
                    revertObject(fieldName, o, fieldAndDesensitizationModeMap);
                }
            } else {
                revertObject(fieldName, result, fieldAndDesensitizationModeMap);
            }
        } catch (Exception e) {
            LOGGER.error("revert error", e);
        }
        return result;
    }

    /**
     * 对jpa参数进行脱敏,可以进行脱敏的参数类型:String、List<String>、实体类、List<实体类>
     *
     * @param fieldName
     * @param parameter
     * @param fieldAndDesensitizationModeMap
     * @return java.lang.Object
     * @author michael
     * @date 2024/6/19 11:00
     **/
    public Object desensitize(String fieldName, Object parameter, Map<String, DesensitizationMode> fieldAndDesensitizationModeMap) {
        try {
            if (fieldAndDesensitizationModeMap == null || fieldAndDesensitizationModeMap.isEmpty() || parameter == null) {
                LOGGER.warn("===>敏感词库或者参数为空,不需要进行加密");
                return parameter;
            }
            if (parameter instanceof List) {
                for (Object o : (List<?>) parameter) {
                    desensitizeObject(fieldName, o, fieldAndDesensitizationModeMap);
                }
            } else {
                desensitizeObject(fieldName, parameter, fieldAndDesensitizationModeMap);
            }
        } catch (Exception e) {
            LOGGER.error("desensitization error", e);
        }
        return parameter;
    }

    private Object desensitizeObject(String fieldName, Object o, Map<String, DesensitizationMode> fieldAndDesensitizationModeMap) throws IllegalAccessException {
        if (o instanceof String) {
            if (fieldAndDesensitizationModeMap.containsKey(fieldName)) {
                DesensitizationProcessor desensitizationProcessor = getDesensitizationProcessor(fieldAndDesensitizationModeMap.get(fieldName));
                desensitizationProcessor.desensitize(fieldName, (String) o);
            }
        } else {
            Field[] fields = o.getClass().getDeclaredFields();
            for (Field field : fields) {
                field.setAccessible(true);
                if (fieldAndDesensitizationModeMap.containsKey(field.getName())) {
                    DesensitizationProcessor desensitizationProcessor = getDesensitizationProcessor(fieldAndDesensitizationModeMap.get(field.getName()));
                    field.set(o, desensitizationProcessor.desensitize(field.getName(), (String) field.get(o)));
                }
            }
        }
        return o;
    }

    private Object revertObject(String fieldName, Object o, Map<String, DesensitizationMode> fieldAndDesensitizationModeMap) throws IllegalAccessException {
        if (o instanceof String) {
            if (fieldAndDesensitizationModeMap.containsKey(fieldName)) {
                DesensitizationProcessor desensitizationProcessor = getDesensitizationProcessor(fieldAndDesensitizationModeMap.get(fieldName));
                desensitizationProcessor.revert(fieldName, (String) o);
            }
        } else if (o instanceof Map) {
            Map<String, Object> paramMap = (Map<String, Object>) o;
            for (Map.Entry<String, Object> entry : paramMap.entrySet()) {
                if (fieldAndDesensitizationModeMap.containsKey(entry.getKey())) {
                    DesensitizationProcessor desensitizationProcessor = getDesensitizationProcessor(fieldAndDesensitizationModeMap.get(entry.getKey()));
                    entry.setValue(desensitizationProcessor.revert(entry.getKey(), (String) entry.getValue()));
                }
            }
        } else {
            Field[] fields = o.getClass().getDeclaredFields();
            for (Field field : fields) {
                field.setAccessible(true);
                if (fieldAndDesensitizationModeMap.containsKey(field.getName())) {
                    DesensitizationProcessor desensitizationProcessor = getDesensitizationProcessor(fieldAndDesensitizationModeMap.get(field.getName()));
                    field.set(o, desensitizationProcessor.revert(field.getName(), (String) field.get(o)));
                }
            }
        }
        return o;
    }


    private DesensitizationProcessor getDesensitizationProcessor(DesensitizationMode mode) {
        return desensitizationProcessorRegistry.findDesensitizationProcessor(mode);
    }
}
