package com.digiwin.dap.middle.database.encrypt.utils;

import com.digiwin.dap.middle.database.encrypt.model.ResultSetMappingRelation;
import net.sf.jsqlparser.expression.CaseExpression;
import net.sf.jsqlparser.expression.Function;
import net.sf.jsqlparser.parser.CCJSqlParserUtil;
import net.sf.jsqlparser.schema.Table;
import net.sf.jsqlparser.statement.Statement;
import net.sf.jsqlparser.statement.delete.Delete;
import net.sf.jsqlparser.statement.insert.Insert;
import net.sf.jsqlparser.statement.select.*;
import net.sf.jsqlparser.statement.update.Update;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.util.StringUtils;

import java.util.ArrayList;
import java.util.List;
import java.util.Objects;

/**
 * @author michael
 */
public class SqlParserUtils {

    private final static Logger LOGGER = LoggerFactory.getLogger(SqlParserUtils.class);

    private final static String MYSQL_DIALECT = "`";

    /**
     * 解析mybatis中select语句且返回为resultType的sql
     *
     * @param sql
     * @return
     * @return java.util.List<com.digiwin.dap.middle.database.encrypt.utils.SqlParserUtils.ColumnInfo>
     * @author michael
     * @date 2024/6/13 09:24
     **/
    public static List<ResultSetMappingRelation> parseQuerySql(String sql) {
        List<ResultSetMappingRelation> resultSetMappingRelations = new ArrayList<>();
        try {
            Statement statement = CCJSqlParserUtil.parse(sql);
            // 1.获取sql中出现的表名以及其别名的映射关系
            List<TableInfo> tableInfoList = parseTableFromStatement(statement);
            // 2.select语句获取结果集表名、表别名、字段名、字段别名的映射关系注意
            resultSetMappingRelations = statement instanceof Select ? parseResultSetFromStatement((Select) statement, tableInfoList) : null;
        } catch (Exception e) {
            LOGGER.error("SQL解析异常", e);
        }
        return resultSetMappingRelations;
    }

    private static List<ResultSetMappingRelation> parseResultSetFromStatement(Select select, List<TableInfo> tableInfoList) {
        List<ResultSetMappingRelation> resultSetMappingRelations = new ArrayList<>();
        SelectBody selectBody = select.getSelectBody();
        selectBody.accept(new SelectVisitorAdapter() {
            @Override
            public void visit(PlainSelect plainSelect) {
                List<SelectItem> selectItems = plainSelect.getSelectItems();
                for (SelectItem selectItem : selectItems) {
                    if (selectItem instanceof SelectExpressionItem) {
                        SelectExpressionItem expressionItem = (SelectExpressionItem) selectItem;
                        if (expressionItem.getExpression() instanceof Function) {
                            // 函数列不支持自动解密
                            continue;
                        }
                        if (expressionItem.getExpression() instanceof CaseExpression) {
                            // case when 不支持自动加密密,会出现多个数据库字段映射到对象中同一属性无法判断是否需要解密
                            continue;
                        }
                        String expressionString = expressionItem.getExpression().toString();
                        String aliasTableName = expressionString.contains(".") ? expressionString.substring(0, expressionString.indexOf(".")) : "";
                        String columnName = expressionString.contains(".") ? expressionString.substring(expressionString.indexOf(".") + 1) : expressionString;
                        String aliasColumnName = expressionItem.getAlias() != null ? expressionItem.getAlias().getName() : columnName;
                        String tableName = StringUtils.hasLength(aliasTableName) ? tableInfoList.stream().filter(x -> Objects.equals(x.getAliasTableName(), aliasTableName)).findFirst().get().getTableName().toLowerCase() :
                                tableInfoList.get(0).getTableName().toLowerCase();
                        // 去掉列名上的``
                        columnName = columnName.replaceAll(MYSQL_DIALECT, "");
                        resultSetMappingRelations.add(new ResultSetMappingRelation(tableName, columnName, aliasColumnName));
                        LOGGER.debug("===>数据库【{}】表,字段【{}】,映射到对象中属性名为【{}】", tableName, columnName, aliasColumnName);
                    }
                }
            }
        });
        return resultSetMappingRelations;
    }


    private static List<TableInfo> parseTableFromStatement(Statement statement) {
        List<TableInfo> tableInfoList = new ArrayList<>();
        if (statement instanceof Select) {
            Select selectStatement = (Select) statement;
            SelectBody selectBody = selectStatement.getSelectBody();
            selectBody.accept(new SelectVisitorAdapter() {
                @Override
                public void visit(PlainSelect plainSelect) {
                    FromItem fromItem = plainSelect.getFromItem();
                    if (fromItem instanceof Table) {
                        Table table = (Table) fromItem;
                        String tableName = table.getName();

                        String[] parts = tableName.split("\\s+");
                        tableName = parts[0];
                        String aliasTableName = table.getAlias() != null ? table.getAlias().getName() : null;
                        // 去掉表名上的``
                        tableName = tableName.replaceAll(MYSQL_DIALECT, "");
                        tableInfoList.add(new TableInfo(tableName, aliasTableName));
                    }

                    if (plainSelect.getJoins() != null) {
                        for (Join join : plainSelect.getJoins()) {
                            FromItem joinItem = join.getRightItem();
                            if (joinItem instanceof Table) {
                                Table table = (Table) joinItem;
                                String tableName = table.getName();
                                String[] parts = tableName.split("\\s+");
                                tableName = parts[0];
                                String aliasTableName = table.getAlias() != null ? table.getAlias().getName() : null;
                                // 去掉表名上的``
                                tableName = tableName.replaceAll(MYSQL_DIALECT, "");
                                tableInfoList.add(new TableInfo(tableName, aliasTableName));
                            }
                        }
                    }
                }
            });
        } else if (statement instanceof Update) {
            Update updateStatement = (Update) statement;
            Table table = updateStatement.getTable();
            tableInfoList.add(new TableInfo(table.getName(), table.getAlias().getName()));
            if (updateStatement.getJoins() != null) {
                for (Join join : updateStatement.getJoins()) {
                    FromItem fromItem = join.getRightItem();
                    if (fromItem instanceof Table) {
                        Table table1 = (Table) fromItem;
                        tableInfoList.add(new TableInfo(table1.getName(), table1.getAlias().getName()));
                    }
                }
            }
        } else if (statement instanceof Delete) {
            Delete deleteStatement = (Delete) statement;
            List<Table> tableList = deleteStatement.getTables();
            for (Table table : tableList) {
                tableInfoList.add(new TableInfo(table.getName(), table.getAlias().getName()));
            }
            if (deleteStatement.getJoins() != null) {
                for (Join join : deleteStatement.getJoins()) {
                    FromItem fromItem = join.getRightItem();
                    if (fromItem instanceof Table) {
                        Table table1 = (Table) fromItem;
                        tableInfoList.add(new TableInfo(table1.getName(), table1.getAlias().getName()));
                    }
                }
            }
        } else if (statement instanceof Insert) {
            Insert insertStatement = (Insert) statement;
            Table table = insertStatement.getTable();
            tableInfoList.add(new TableInfo(table.getName(), table.getAlias().getName()));
        }
        return tableInfoList;
    }

    /**
     * SQL中表名与表别名的映射关系
     *
     * @author michael
     * @date 2024/6/6 13:28
     * @see
     **/
    static class TableInfo {
        private String tableName;

        private String aliasTableName;

        public TableInfo(String tableName, String aliasTableName) {
            this.tableName = tableName;
            this.aliasTableName = aliasTableName;
        }

        public String getTableName() {
            return tableName;
        }

        public String getAliasTableName() {
            return aliasTableName;
        }
    }
}
