package com.digiwin.dap.middle.database.encrypt.desensitization;

import com.digiwin.dap.middle.database.encrypt.enums.DesensitizationMode;
import com.digiwin.dap.middle.database.encrypt.exception.DatabaseEncryptException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import java.lang.reflect.Field;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Optional;

/**
 * @author michael
 */
@Component
public class DesensitizationProcessorFactory {

    private final static Logger LOGGER = LoggerFactory.getLogger(DesensitizationProcessorFactory.class);

    private final DefaultDesensitizationProcessorRegistry desensitizationProcessorRegistry;

    @Autowired
    public DesensitizationProcessorFactory(@Autowired DefaultDesensitizationProcessorRegistry desensitizationProcessorRegistry) {
        this.desensitizationProcessorRegistry = desensitizationProcessorRegistry;
    }

    /**
     * <p>对结果集进行还原
     * <p>1.jpa返回结果集可以进行还原的数据类型:String、List<String>、实体类、List<实体类></p>
     * <p>2.mybatis返回结果集可以进行还原的数据类型:String、List<String>、对象、List<对象>、map</p>
     * <p/>
     *
     * @return java.lang.Object
     **/
    public Object revert(String fieldName, Object result, Map<String, DesensitizationMode> fieldAndDesensitizationModeMap) {
        try {
            if (fieldAndDesensitizationModeMap == null || fieldAndDesensitizationModeMap.isEmpty() || result == null) {
                return result;
            }
            if (result instanceof Optional) {
                Optional<?> optional = (Optional<?>) result;
                Object optionalValue = optional.orElse(null);
                if (optionalValue != null) {
                    Object newResult = revertObject(fieldName, optionalValue, fieldAndDesensitizationModeMap);
                    result = Optional.ofNullable(newResult);
                }
            } else {
                if (result instanceof List) {
                    List<Object> newResult = new ArrayList<>();
                    for (Object o : (List<?>) result) {
                        newResult.add(revertObject(fieldName, o, fieldAndDesensitizationModeMap));
                    }
                    result = newResult;
                } else {
                    result = revertObject(fieldName, result, fieldAndDesensitizationModeMap);
                }
            }
        } catch (Exception e) {
            throw new DatabaseEncryptException(e);
        }
        return result;
    }

    /**
     * 对jpa参数进行脱敏,可以进行脱敏的参数类型:String、List<String>、实体类、List<实体类>
     *
     * @return java.lang.Object
     **/
    public Object desensitize(String fieldName, Object parameter, Map<String, DesensitizationMode> fieldAndDesensitizationModeMap) {
        try {
            if (fieldAndDesensitizationModeMap == null || fieldAndDesensitizationModeMap.isEmpty() || parameter == null) {
                return parameter;
            }
            if (parameter instanceof Optional) {
                Optional<?> optional = (Optional<?>) parameter;
                Object optionalValue = optional.orElse(null);
                if (optionalValue != null) {
                    Object newParameter = desensitizeObject(fieldName, optionalValue, fieldAndDesensitizationModeMap);
                    parameter = Optional.ofNullable(newParameter);
                }
            } else {
                if (parameter instanceof List) {
                    List<Object> newParameter = new ArrayList<>();
                    for (Object o : (List<?>) parameter) {
                        newParameter.add(desensitizeObject(fieldName, o, fieldAndDesensitizationModeMap));
                    }
                    parameter = newParameter;
                } else {
                    parameter = desensitizeObject(fieldName, parameter, fieldAndDesensitizationModeMap);
                }
            }
        } catch (Exception e) {
            throw new DatabaseEncryptException(e);
        }
        return parameter;
    }

    private Object desensitizeObject(String fieldName, Object o, Map<String, DesensitizationMode> fieldAndDesensitizationModeMap) {

        if (o instanceof String) {
            if (fieldAndDesensitizationModeMap.containsKey(fieldName)) {
                DesensitizationProcessor desensitizationProcessor = getDesensitizationProcessor(fieldAndDesensitizationModeMap.get(fieldName));
                o = desensitizationProcessor.desensitize(fieldName, (String) o);
            }
        } else {
            Field[] fields = o.getClass().getDeclaredFields();
            for (Field field : fields) {
                field.setAccessible(true);
                if (fieldAndDesensitizationModeMap.containsKey(field.getName())) {
                    DesensitizationProcessor desensitizationProcessor = getDesensitizationProcessor(fieldAndDesensitizationModeMap.get(field.getName()));
                    try {
                        String oldValue = (String) field.get(o);
                        String newValue = desensitizationProcessor.desensitize(field.getName(), oldValue);
                        field.set(o, newValue);
                    } catch (Exception e) {
                        if (e instanceof DatabaseEncryptException) {
                            LOGGER.error("===>对象【{}】字段【{}】加密失败,失败信息【{}】", o.getClass().getName(), field.getName(), e.getMessage());
                        }
                        if (e instanceof IllegalAccessException) {
                            LOGGER.error("===>对象【{}】字段【{}】加密时无法获取或者设置值", o.getClass().getName(), field.getName());
                        }
                    }
                }
            }
        }
        return o;
    }

    private Object revertObject(String fieldName, Object o, Map<String, DesensitizationMode> fieldAndDesensitizationModeMap) {
        try {
            if (o instanceof String) {
                if (fieldAndDesensitizationModeMap.containsKey(fieldName)) {
                    DesensitizationProcessor desensitizationProcessor = getDesensitizationProcessor(fieldAndDesensitizationModeMap.get(fieldName));
                    o = desensitizationProcessor.revert(fieldName, (String) o);
                }
            } else if (o instanceof Map) {
                Map<String, Object> paramMap = (Map<String, Object>) o;
                for (Map.Entry<String, Object> entry : paramMap.entrySet()) {
                    if (fieldAndDesensitizationModeMap.containsKey(entry.getKey())) {
                        DesensitizationProcessor desensitizationProcessor = getDesensitizationProcessor(fieldAndDesensitizationModeMap.get(entry.getKey()));
                        entry.setValue(desensitizationProcessor.revert(entry.getKey(), (String) entry.getValue()));
                    }
                }
            } else {
                Field[] fields = o.getClass().getDeclaredFields();
                for (Field field : fields) {
                    field.setAccessible(true);
                    if (fieldAndDesensitizationModeMap.containsKey(field.getName())) {
                        DesensitizationProcessor desensitizationProcessor = getDesensitizationProcessor(fieldAndDesensitizationModeMap.get(field.getName()));
                        try {
                            String oldValue = (String) field.get(o);
                            String newValue = desensitizationProcessor.revert(field.getName(), oldValue);
                            field.set(o, newValue);
                        } catch (Exception e) {
                            if (e instanceof DatabaseEncryptException) {
                                LOGGER.error("===>对象【{}】字段【{}】解密失败,失败信息【{}】", o.getClass().getName(), field.getName(), e.getMessage());
                            }
                            if (e instanceof IllegalAccessException) {
                                LOGGER.error("===>对象【{}】字段【{}】解密时无法获取或者设置值", o.getClass().getName(), field.getName());
                            }
                        }
                    }
                }
            }
        } catch (Exception e) {
            LOGGER.error("===>数据库加解密,解密异常", e);
        }
        return o;
    }


    private DesensitizationProcessor getDesensitizationProcessor(DesensitizationMode mode) {
        return desensitizationProcessorRegistry.findDesensitizationProcessor(mode);
    }
}
