package com.digiwin.dap.middle.database.encrypt.sensitive.word;

import com.digiwin.dap.middle.database.encrypt.annotation.Desensitization;
import com.digiwin.dap.middle.database.encrypt.annotation.SensitiveField;
import com.digiwin.dap.middle.database.encrypt.model.SensitiveWordDatabase;
import org.reflections.Reflections;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.core.env.Environment;

import javax.annotation.PostConstruct;
import javax.persistence.Entity;
import javax.persistence.Table;
import java.lang.reflect.Field;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.CopyOnWriteArrayList;


/**
 * @author michael
 */
public class DefaultSensitiveWordDatabaseRegistry implements SensitiveWordDatabaseRegistry {

    private final static Logger LOGGER = LoggerFactory.getLogger(DefaultSensitiveWordDatabaseRegistry.class);

    private Environment environment;

    public DefaultSensitiveWordDatabaseRegistry(Environment environment) {
        this.environment = environment;
    }

    private Map<String, List<SensitiveWordDatabase>> sensitiveWordDatabaseMap = new ConcurrentHashMap<>();

    @Override
    public List<SensitiveWordDatabase> findSensitiveWordDatabase(String tableName) {
        return sensitiveWordDatabaseMap.get(tableName);
    }

    @PostConstruct
    public void run() {
        LOGGER.info("------初始化数据库敏感词库------");
        String packageName = environment.getProperty("mybatis.typeAliasesPackage");
        Reflections reflections = new Reflections(packageName);
        // 1.获取所有实体类
        Set<Class<?>> entityClasses = reflections.getTypesAnnotatedWith(Entity.class);
        if (Objects.isNull(entityClasses) || entityClasses.isEmpty()) {
            LOGGER.error("===>扫描【{}】未获取到实体类.", packageName);
        }
        for (Class<?> entityClass : entityClasses) {
            // 2.判断是否包含敏感信息
            if (!entityClass.isAnnotationPresent(Desensitization.class)) {
                continue;
            }
            // 2.判断是否包含@Table
            if (!entityClass.isAnnotationPresent(Table.class)) {
                continue;
            }
            // 3.获取表名
            Table table = entityClass.getAnnotation(Table.class);
            String tableName = table.name();

            // 4.判断是否脱敏
            Desensitization desensitization = entityClass.getAnnotation(Desensitization.class);
            if (!desensitization.enabled()) {
                LOGGER.warn("【{}】标注敏感字段,但是未开启脱敏!", entityClass.getName());
                continue;
            }

            // 4.遍历敏感信息获取敏感字段已经脱敏规则
            Field[] fields = entityClass.getDeclaredFields();
            for (Field field : fields) {
                field.setAccessible(true);
                SensitiveField sensitiveField = field.getAnnotation(SensitiveField.class);
                if (Objects.isNull(sensitiveField)) {
                    continue;
                }
                LOGGER.info("===>数据库表【{}】字段【{}】为敏感字段,实体属性名为【{}】.", tableName, sensitiveField.columnName(), field.getName());
                SensitiveWordDatabase sensitiveWordDatabase = new SensitiveWordDatabase(tableName.toLowerCase(), sensitiveField.columnName(), field.getName(), sensitiveField.mode());
                if (!sensitiveWordDatabaseMap.containsKey(tableName)) {
                    sensitiveWordDatabaseMap.put(tableName, new CopyOnWriteArrayList<>());
                }
                sensitiveWordDatabaseMap.get(tableName).add(sensitiveWordDatabase);
            }
        }
        LOGGER.info("------数据库敏感词库初始化完成------");
    }
}
